package com.af.v4.system.common.cdc.connector.sink;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.SpringUtils;
import com.af.v4.system.common.mq.RocketMQProducer;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.streaming.api.functions.sink.RichSinkFunction;
import org.apache.rocketmq.client.producer.SendStatus;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Sink基类
 */
public abstract class Sink extends RichSinkFunction<String> {
    private static final Logger LOGGER = LoggerFactory.getLogger(Sink.class);
    private static final Logger CDC_ERROR_LOGGER = LoggerFactory.getLogger("cdcErrorLog");

    protected String globalConfigStr;
    protected RocketMQProducer rocketMQProducer;

    public Sink(String globalConfigStr) {
        this.globalConfigStr = globalConfigStr;
    }

    /**
     * 在open()方法中动态注入Spring容器的类
     * 在启动SpringBoot项目时加载了Spring容器，其他地方可以使用@Autowired获取Spring容器中的类；
     * 但是Flink启动的项目中，默认启动了多线程执行相关代码，导致在其他线程无法获取Spring容器，
     * 只有在Spring所在的线程才能使用@Autowired，故在Flink自定义的Sink的open()方法中初始化Spring容器
     */
    @Override
    public void open(Configuration parameters) throws Exception {
        super.open(parameters);
        rocketMQProducer = SpringUtils.getBean(RocketMQProducer.class);
    }

    @Override
    public void invoke(String value, Context context) {
        // 将数据转换为JSONObject
        JSONObject valueObj = new JSONObject(value);

        JSONObject globalConfig = new JSONObject(globalConfigStr);
        JSONObject sink = globalConfig.getJSONObject("sink");
        String table = sink.getString("table");

        JSONObject paramsData = new JSONObject();
        if (valueObj.has("type")) {
            paramsData.put("type", valueObj.get("type").toString());
        }
        JSONObject data = valueObj.getJSONObject("data");
        // 对数据进行处理
        try {
            paramsData.put("dataSourceName", sink.getString("name"));
            paramsData.put("entityData", data);
            paramsData.put("table", table);
            // 处理数据
            exec(globalConfig, paramsData);
            // 执行逻辑
            JSONObject result = rocketMQProducer.syncSend(RocketMQProducer.DEFAULT_LOGIC_TOPIC, globalConfig.getString("dataSyncLogic"), paramsData.toString());
            if (!result.getString("sendStatus").equals(SendStatus.SEND_OK.name())) {
                throw new ServiceException("数据同步失败，信息：" + result);
            }
        } catch (Exception e) {
            LOGGER.error("数据处理失败.", e);
            CDC_ERROR_LOGGER.info(paramsData + ",");
        }
    }

    public abstract void exec(JSONObject globalConfig, JSONObject paramsData) throws Exception;
}
