package com.af.v4.system.common.cdc.connector.source;

import com.af.v4.system.common.cdc.enums.ConfigureEnum;
import com.af.v4.system.common.cdc.utils.JsonDebeziumDeserializationSchema;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.datasource.MyDruidDataSource;
import com.ververica.cdc.debezium.DebeziumSourceFunction;
import org.apache.flink.api.common.restartstrategy.RestartStrategies;
import org.apache.flink.contrib.streaming.state.EmbeddedRocksDBStateBackend;
import org.apache.flink.streaming.api.CheckpointingMode;
import org.apache.flink.streaming.api.environment.CheckpointConfig;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.flink.streaming.api.functions.sink.SinkFunction;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;

/**
 * Source基类
 */
public abstract class Source {
    private static final Logger LOGGER = LoggerFactory.getLogger(Source.class);
    protected StreamExecutionEnvironment env;

    protected JsonDebeziumDeserializationSchema schema = new JsonDebeziumDeserializationSchema();

    protected abstract DebeziumSourceFunction buildDataSource(JSONObject globalConfig, JSONObject sourceConfig, MyDruidDataSource dataSource);

    /**
     * 开始任务
     *
     * @throws Exception
     */
    public void start(JSONObject config) throws Exception {
        LOGGER.info("准备捕获变更数据......");
        env = StreamExecutionEnvironment.getExecutionEnvironment();
        // 设置并行度
        env.setParallelism(ConfigureEnum.NumEnum.PARALLELISM.getNum());
        // 设置检查点
        configureCheckpointing();
        // 设置重启策略
        configureRestartStrategy();
        // 加载数据源
        Map<String, MyDruidDataSource> dataSourceMap = DynamicDataSource.getDataSourceList();
        // 数据库连接信息
        JSONObject sourceConfig = config.getJSONObject("source");
        JSONObject sinkConfig = config.getJSONObject("sink");
        String sourceName = sourceConfig.getString("name");
        String sinkName = sinkConfig.getString("name");
        MyDruidDataSource sourceDataSource = dataSourceMap.get(sourceName);
        DebeziumSourceFunction debeziumSourceFunction = buildDataSource(config, sourceConfig, sourceDataSource);
        // 构建数据源和目标
        env.addSource(debeziumSourceFunction, sourceName)
                .addSink((SinkFunction) Class.forName(sinkConfig.getString("className")).getDeclaredConstructor(String.class).newInstance(config.toString()))
                .name(sinkName);
        // 执行
        env.execute(config.getString("jobName"));
    }

    /**
     * 设置检查点
     */
    private void configureCheckpointing() {
        // 开启检查点、设置多久执行一次、设置启动模式(正好一次/至少一次)
        env.enableCheckpointing(ConfigureEnum.TimeEnum.CHECKPOINT_INTERVAL.getTime(), CheckpointingMode.EXACTLY_ONCE);
        // 设置检查点超时时间
        env.getCheckpointConfig().setCheckpointTimeout(ConfigureEnum.TimeEnum.CHECKPOINT_TIMEOUT.getTime());
        // 启用外部化检查点，并指定存储路径
        env.setStateBackend(new EmbeddedRocksDBStateBackend());
        env.getCheckpointConfig().setCheckpointStorage("file:///D:/flink-checkpoint");
        env.getCheckpointConfig().setExternalizedCheckpointCleanup(CheckpointConfig.ExternalizedCheckpointCleanup.RETAIN_ON_CANCELLATION);
        env.setRestartStrategy(RestartStrategies.fixedDelayRestart(3, 2000L));
    }

    /**
     * 设置重启策略
     */
    private void configureRestartStrategy() {
        // 重启次数
        int restartCount = ConfigureEnum.NumEnum.RESTART_COUNT.getNum();
        // 重启间隔时间
        long attemptToRestartDelay = ConfigureEnum.TimeEnum.ATTEMPT_TO_RESTART_DELAY.getTime();
        env.setRestartStrategy(RestartStrategies.fixedDelayRestart(restartCount, attemptToRestartDelay));
    }
}
