package com.af.v4.system.api;

import com.af.v4.system.api.factory.RemoteEntityFallbackFactory;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.constant.ServiceNameConstants;
import com.af.v4.system.common.core.domain.R;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

/**
 * 实体服务
 *
 * @author Mr.river
 */
@FeignClient(contextId = "remoteEntityService",
        value = ServiceNameConstants.GATEWAY_SERVICE,
        fallbackFactory = RemoteEntityFallbackFactory.class)
public interface RemoteEntityService {
    /**
     * 实体保存/更新
     *
     * @param entityName 实体名称
     * @param entity     实体内容
     * @param source     请求来源
     * @return 结果
     */
    @PostMapping("/entity/save/{entityName}")
    R<Map<String, Object>> save(@PathVariable("entityName") String entityName,
                                @RequestBody String entity,
                                @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

    /**
     * 批量实体保存/更新
     *
     * @param entityName    实体名称
     * @param entityContent 实体内容
     * @param source        请求来源
     * @return 结果
     */
    @PostMapping("/entity/batchSave/{entityName}")
    R<List<Object>> saveBatch(@PathVariable("entityName") String entityName,
                              @RequestBody String entityContent,
                              @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

    /**
     * 根据指定键对实体保存/更新
     *
     * @param entityName 实体名称
     * @param params     参数
     * @param source     请求来源
     * @return 结果
     */
    @PostMapping("/entity/saveByKey/{entityName}")
    R<Map<String, Object>> saveByKey(@PathVariable("entityName") String entityName,
                                     @RequestBody String params,
                                     @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

    /**
     * 删除实体
     *
     * @param entityName 实体名称
     * @param id         ID
     * @param source     请求来源
     * @return 结果
     */
    @DeleteMapping("/entity/{entityName}/{id}")
    R<Map<String, Object>> deleteById(@PathVariable("entityName") String entityName,
                                      @PathVariable("id") String id,
                                      @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

    /**
     * 批量删除实体
     *
     * @param entityName 实体名称
     * @param entity     参数
     * @param source     请求来源
     * @return 结果
     */
    @DeleteMapping("/entity/{entityName}")
    R<Map<String, Object>> deleteAllByIds(@PathVariable("entityName") String entityName,
                                          @RequestBody String entity,
                                          @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

    /**
     * 根据ID查询数据
     *
     * @param entityName 实体名称
     * @param id         ID
     * @param source     请求来源
     * @return 结果
     */
    @PostMapping("/entity/query/{entityName}/{id}")
    R<Map<String, Object>> getById(@PathVariable("entityName") String entityName,
                                   @PathVariable("id") String id,
                                   @RequestBody String entity,
                                   @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

    /**
     * 根据ID集合查询所有数据
     *
     * @param entityName 实体名称
     * @param entity     参数
     * @param source     请求来源
     * @return 结果
     */
    @PostMapping("/entity/query/{entityName}")
    R<List<Object>> findAllByIds(@PathVariable("entityName") String entityName,
                                 @RequestBody String entity,
                                 @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

    /**
     * 查询所有数据
     *
     * @param entityName 实体名称
     * @param entity     参数
     * @param source     请求来源
     * @return 结果
     */
    @PostMapping("/entity/queryAll/{entityName}")
    R<List<Object>> findAll(@PathVariable("entityName") String entityName,
                            @RequestBody String entity,
                            @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

    /**
     * 查询实体的总数量
     *
     * @param entityName 实体名称
     * @param source     请求来源
     * @return 结果
     */
    @PostMapping("/entity/queryCount/{entityName}")
    R<Map<String, Object>> getCount(@PathVariable("entityName") String entityName,
                                    @RequestHeader(SecurityConstants.FROM_SOURCE) String source);
}
