package com.af.system.easyexcel.utils;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.io.resource.ClassPathResource;
import cn.hutool.core.util.IdUtil;
import com.af.system.easyexcel.constant.ExcelConstant;
import com.af.system.easyexcel.convert.BaseDateConverter;
import com.af.system.easyexcel.convert.CustomStringStringConverter;
import com.af.system.easyexcel.convert.ExcelBigNumberConvert;
import com.af.system.easyexcel.core.CellColorSheetWriteHandler;
import com.af.system.easyexcel.template.ExcelTemplate_Grey;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.support.ExcelTypeEnum;
import com.alibaba.excel.write.builder.ExcelWriterSheetBuilder;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.alibaba.excel.write.metadata.style.WriteCellStyle;
import com.alibaba.excel.write.metadata.style.WriteFont;
import com.alibaba.excel.write.style.HorizontalCellStyleStrategy;
import com.alibaba.excel.write.style.column.LongestMatchColumnWidthStyleStrategy;
import com.alibaba.excel.write.style.row.SimpleRowHeightStyleStrategy;
import org.apache.poi.ss.usermodel.*;
import org.json.JSONObject;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.util.*;



/**
 * @Description Excel处理工具
 * @Author Eraser
 */
public class ExcelUtil {

    /**
     * 同步导入(适用于小数据量)
     *
     * @param is 输入流
     * @return 转换后集合
     */
    public static <T> List<T> importExcel(InputStream is, Set<String> headSet) {
        return EasyExcel.read(is).head(getHead(headSet)).autoCloseStream(false).sheet().doReadSync();
    }

    /**
     * 重置响应体
     */
    private static void resetResponse(String sheetName, HttpServletResponse response) throws UnsupportedEncodingException {
        String filename = encodingFilename(sheetName);
        FileUtils.setAttachmentResponseHeader(response, filename);
        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet;charset=UTF-8");
    }

    /**
     * 编码文件名
     */
    public static String encodingFilename(String filename) {
        return IdUtil.fastSimpleUUID() + "_" + filename + ".xlsx";
    }

    /**
     * 写入构造器
     * @param os           输出流
     * @param sheetName    表名
     * @param constants    模板对象
     * @param hasStyle     是否有样式
     * @param style        表格样式
     * @param hasAppoint   是否有单元格样式
     * @param appointCell  单元格样式
     * @return ExcelWriterSheetBuilder 构造器对象
     */
    public static ExcelWriterSheetBuilder write(OutputStream os, String sheetName, ExcelConstant constants, boolean hasStyle, List<Object> style, boolean hasAppoint, List<Object> appointCell) {
        //执行导出
        ExcelWriterSheetBuilder builder = EasyExcel.write(os)
                //不自动关闭流
                .autoCloseStream(false)
                //大数值自动转换
                .registerConverter(new ExcelBigNumberConvert())
                //String转换
                .registerConverter(new CustomStringStringConverter())
                //Timestamp转换
                .registerConverter(new BaseDateConverter.LocalDateTimeConverter())
                //自动适配
                .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
                .excelType(ExcelTypeEnum.XLSX)
                //设置头和内容行高
                .registerWriteHandler(new SimpleRowHeightStyleStrategy((short) 20, (short) 20))
                .sheet(sheetName);
        if (hasStyle) {
            //设置表头样式
            HorizontalCellStyleStrategy horizontalCellStyleStrategy = exportStyle(style, constants);
            builder.registerWriteHandler(horizontalCellStyleStrategy);
        }
        if (hasAppoint) {
            //设置表头样式
            HashMap<Integer, List<Integer>> map = (HashMap<Integer, List<Integer>>) appointCell.get(0);
            CellColorSheetWriteHandler writeHandler = new CellColorSheetWriteHandler(map, IndexedColors.RED.index, IndexedColors.YELLOW.index);
            builder.registerWriteHandler(writeHandler);
        }
        return builder;
    }

    /**
     * 生成Excel文件放到服务器
     * 动态表头，动态数据，生成Excel文件
     * @param dataList 表格内容数据
     * @param headMap  表头map 中文名->字段名
     * @param sheetName 表名
     * @return  String filepath 文件存放路径
     * @throws FileNotFoundException
     */
    public static String createExcel(List<Object> dataList, Map<String, Object> headMap, String sheetName) throws FileNotFoundException {
        try {
            String filepath = getFilePath(sheetName+".xlsx");
            FileOutputStream fos = new FileOutputStream(filepath);//字节流
            Set<String> headSet = headMap.keySet();

            ExcelWriterSheetBuilder builder = write(fos, sheetName, new ExcelTemplate_Grey(), true, null, false, null);
            //获得文件
            builder.head(getHead(headSet)).doWrite(dataList);
            return filepath;
        }catch (Exception e){
            throw new RuntimeException("创建Excel异常！");
        }
    }


    /**
     * 导出Excel
     * @param dataList 表格内容数据
     * @param headMap  表头map 中文名->字段名
     * @param sheetName    表名
     * @param constants    模板对象
     * @param hasStyle     是否有样式
     * @param style        表格样式
     * @param hasAppoint   是否有单元格样式
     * @param appointCell  单元格样式
     * @param response  响应体
     */
    public static void exportExcel(List<Object> dataList, Map<String, Object> headMap, String sheetName, List<Object> style, boolean hasStyle, ExcelConstant constants, boolean hasAppoint, List<Object> appointCell, HttpServletResponse response) {
        try {
            resetResponse(sheetName, response);
            ServletOutputStream os = response.getOutputStream();
            //组装头
            Set<String> headSet = headMap.keySet();
            //写入数据
            ExcelWriterSheetBuilder builder = write(os, sheetName, constants, hasStyle, style, hasAppoint, appointCell);
            builder.head(getHead(headSet)).doWrite(dataList);
        } catch (IOException e) {
            throw new RuntimeException("导出Excel异常！");
        }
    }
    /**
     * 导出Excel
     * @param dataList 表格内容数据
     * @param sheetName    表名
     * @param constants    模板对象
     * @param hasStyle     是否有样式
     * @param style        表格样式
     * @param hasAppoint   是否有单元格样式
     * @param appointCell  单元格样式
     * @param response  响应体
     */
    public static void exportExcel(List<Object> dataList, String sheetName, List<Object> style, boolean hasStyle, ExcelConstant constants, boolean hasAppoint, List<Object> appointCell, HttpServletResponse response) {
        try {
            resetResponse(sheetName, response);
            ServletOutputStream os = response.getOutputStream();
            //组装表格头
            Set<String> headSet = ((Map<String, Object>) dataList.get(0)).keySet();
            //组装内容
            List<List<Object>> data = getData(dataList);
            //写入数据
            ExcelWriterSheetBuilder builder = write(os, sheetName, constants, hasStyle, style, hasAppoint, appointCell);
            builder.head(getHead(headSet));
            if (dataList.get(0) instanceof List) {
                builder.doWrite(dataList);
            } else {
                builder.doWrite(getData(dataList));
            }
        } catch (IOException e) {
            throw new RuntimeException("导出Excel异常！");
        }
    }

    /**
     * Excel导出方法
     *
     * @param dataList  数据列表
     * @param sheetName 表名
     * @param response  响应值
     */
    public static void exportExcel(List<Object> dataList, String sheetName, HttpServletResponse response) {
        //默认导出模板为灰色
        exportExcel(dataList, sheetName, null, true, new ExcelTemplate_Grey(), false, null, response);
    }

    /**
     * Excel导出方法
     *
     * @param dataList  数据列表
     * @param head      表头
     * @param sheetName 表名
     * @param response  响应值
     */
    public static void exportExcel(List<Object> dataList, Map<String, Object> head, String sheetName, HttpServletResponse response) {
        //默认导出模板为灰色
        exportExcel(dataList, head, sheetName, null, true, new ExcelTemplate_Grey(), false, null, response);
    }

    /**
     * 单表多数据模板导出 模板格式为 {.属性} JDK17模板暂只支持模板为xls格式
     *
     * @param fileName     文件名
     * @param templatePath 模板路径 resource 目录下的路径包括模板文件名
     *                     例如: excel/temp.xlsx
     *                     重点: 模板文件必须放置到启动类对应的 resource 目录下
     * @param data         模板需要的数据
     */
    public static void exportTemplate(List<Object> data, String fileName, String templatePath, HttpServletResponse response) {
        try {
            resetResponse(fileName, response);
            ClassPathResource templateResource = new ClassPathResource(templatePath);
            ExcelWriter excelWriter = EasyExcel.write(response.getOutputStream())
                    .withTemplate(templateResource.getStream())
                    .excelType(ExcelTypeEnum.XLSX)
                    .autoCloseStream(false)
                    // 大数值自动转换 防止失真
                    .registerConverter(new ExcelBigNumberConvert())
                    .build();
            WriteSheet writeSheet = EasyExcel.writerSheet().build();
            if (CollUtil.isEmpty(data)) {
                throw new IllegalArgumentException("数据为空");
            }
            // 单表多数据导出 模板格式为 {.属性}
            //组装数据体
            excelWriter.fill(data, writeSheet);

            excelWriter.finish();
        } catch (IOException e) {
            throw new RuntimeException("导出Excel异常");
        }
    }

    /**
     * excel样式设定
     *
     * @param styleList
     * @param constants
     * @return 样式注册对象
     */
    public static HorizontalCellStyleStrategy exportStyle(List<Object> styleList, ExcelConstant constants) {
        try {
            if (styleList != null) {
                Map<String, Object> styleMap = (Map<String, Object>) styleList.get(0);
                short head_font_size = Short.parseShort(styleMap.get("EXCEL_HEAD_FONT_SIZE").toString());
                String head_align = styleMap.get("EXCEL_HEAD_ALIGN").toString().toUpperCase();
                String head_font_color = styleMap.get("EXCEL_HEAD_FONT_COLOR").toString().toUpperCase();
                String head_full_color = styleMap.get("EXCEL_HEAD_FULL_COLOR").toString().toUpperCase();
                //样式设定
                //--表头
                constants.EXCEL_HEAD_FONT_SIZE = head_font_size;
                constants.EXCEL_HEAD_ALIGN = HorizontalAlignment.valueOf(head_align);
                constants.EXCEL_HEAD_FONT_COLOR = IndexedColors.valueOf(head_font_color).getIndex();
                constants.EXCEL_HEAD_FULL_COLOR = IndexedColors.valueOf(head_full_color).getIndex();
            }
            //设置表头样式
            WriteCellStyle headWriteCellStyle = new WriteCellStyle();
            WriteFont headFont = new WriteFont();
            //--设置表头字体
            headFont.setColor(constants.EXCEL_HEAD_FONT_COLOR);
            headFont.setFontName(constants.EXCEL_HEAD_FONT_NAME);
            headFont.setFontHeightInPoints(constants.EXCEL_HEAD_FONT_SIZE);
            headWriteCellStyle.setWriteFont(headFont);
            //--设置表头居中对齐
            headWriteCellStyle.setHorizontalAlignment(constants.EXCEL_HEAD_ALIGN);
            //--设置表头填充颜色
            headWriteCellStyle.setFillForegroundColor(constants.EXCEL_HEAD_FULL_COLOR);
            headWriteCellStyle.setShrinkToFit(true);
            List<WriteCellStyle> cellStyleList = new ArrayList<>();
            //设置单元格
            return new HorizontalCellStyleStrategy(headWriteCellStyle, cellStyleList);
        } catch (Exception e) {
            throw new RuntimeException("Excel样式设定异常");
        }

    }

    /**
     * 获取表格表头
     *
     * @param headSet
     * @return
     */
    private static List<List<String>> getHead(Set<String> headSet) {
        List<List<String>> list = new ArrayList<>();
        headSet.forEach(str -> {
            List<String> head = new ArrayList<>();
            head.add(str);
            list.add(head);
        });
        return list;
    }

    /**
     * 获取表格内容
     * @param dataList
     * @return
     */
    private static List<List<Object>> getData(List<Object> dataList) {
        //组装数据体
        List<List<Object>> lists = new ArrayList<>(dataList.size());
        dataList.forEach(item -> {
            Collection<Object> values = (((Map<String, Object>) item).values());
            List<Object> data = new ArrayList<>();
            values.forEach(v -> {
                if (JSONObject.NULL == v) {
                    v = "--";
                }
                data.add(v);
            });
            lists.add(data);
        });
        return lists;
    }

    /**
     * 获取文件路径
     * @param name
     * @return
     */
    private static String getFilePath(String name) {
        String excelFileName = (name == null ? UUID.randomUUID() + ".xlsx" : name);
        String path = ExcelUtil.class.getClassLoader().getResource("config.json")
                .getPath();
        String rootPath = path.split("WEB-INF")[0];
        String filePath = rootPath + "excel/" + excelFileName;
        return filePath;
    }
}