package com.aote.util;

import com.af.plugins.RestTools;
import com.aote.v4.compatibility.core.redis.RedisUtil;
import com.aote.v4.compatibility.core.utils.InnerSystemConfig;
import com.aote.weixin.Config;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.concurrent.atomic.AtomicReference;

/**
 * @description:
 * @author: 唐梓烨
 * @time: 2024/1/16 10:56
 */
@Component
public class WechatApiUtil {
    public static final Integer ERROR_AUTH_CODE = 40001;
    /**
     * 微信apiToken缓存key名  根据的是加上  appid
     */
    private static final String WECHAT_API_TOKEN_CACHE_KEY = "wechatApiToken@";
    static Logger LOGGER = LoggerFactory.getLogger(WechatApiUtil.class);
    private RedisUtil redisUtil = RedisUtil.getInstance();

    private static String getWechatApiTokenRedisKey(String appid) {
        return WECHAT_API_TOKEN_CACHE_KEY + appid;
    }
    /**
     * 获取TOKEN
     *
     * @param appid        appid
     * @param secret       secret
     * @param forceRefresh 是否强制刷新
     * @return TOKEN值
     */
    public String getAuthToken(String appid, String secret, Boolean forceRefresh) {
        String key = getWechatApiTokenRedisKey(appid);
        String token = (String) redisUtil.get(key);
        if (token != null) {
            LOGGER.info(appid + ":获取现有token");
            return token;
        } else {
            synchronized (WechatApiUtil.class) {
                AtomicReference<String> newAccessToken = new AtomicReference<>();
                redisUtil.lock(key, () -> {
                    // 再次尝试从缓存获取，避免集群多实例重复请求获取auth
                    String newToken = (String) redisUtil.get(key);
                    if (newToken != null) {
                        LOGGER.info(appid + ":获取现有token");
                        newAccessToken.set(newToken);
                    }
                    if (forceRefresh) {
                        LOGGER.info("微信：" + appid + ":进行强制刷新鉴权操作");
                    } else {
                        LOGGER.info("微信：" + appid + ":进行鉴权操作");
                    }
                    //组织请求报文
                    JSONObject content = new JSONObject();
                    content.put("grant_type", "client_credential");
                    content.put("appid", appid);
                    content.put("secret", secret);
                    content.put("force_refresh", forceRefresh);
                    JSONObject result = new JSONObject(RestTools.post(WechatUrl.AUTH_URL, content));
                    if (result.has("errcode")) {
                        throw new RuntimeException(result.toString());
                    } else {
                        String accessToken = result.getString("access_token");
                        int expiresIn = result.getInt("expires_in");
                        redisUtil.set(key, accessToken, expiresIn - 180);
                        newAccessToken.set(accessToken);
                    }
                    return content;
                });
                return newAccessToken.get();
            }
        }
    }
    public String getAuthToken() {
        String orgName = InnerSystemConfig.System.getTenantName();
        return getAuthToken(orgName);
    }
    public String getAuthToken(String orgName) {
        if (orgName.isEmpty()){
            orgName = InnerSystemConfig.System.getTenantName();
        }
        JSONObject config = Config.getConfig(orgName);
        return getAuthToken(config.getString("appId"), config.getString("appSecret"));
    }
    public String getAuthToken(String appid, String secret) {
        return getAuthToken(appid, secret, false);
    }
    public String getAuthTokenByRequest() {
        String orgName = InnerSystemConfig.System.getTenantName();
        return getAuthTokenByRequest(orgName);
    }
    public String getAuthTokenByRequest(String orgName) {
        if (orgName.isEmpty()){
            orgName = InnerSystemConfig.System.getTenantName();
        }
        JSONObject config = Config.getConfig(orgName);
        return getAuthTokenByRequest(config.getString("appId"), config.getString("appSecret"), false);
    }
    public String getAuthTokenByRequest(String appid, String secret, Boolean forceRefresh) {
        String key = getWechatApiTokenRedisKey(appid);
        synchronized (WechatApiUtil.class) {
            AtomicReference<String> newAccessToken = new AtomicReference<>();
            redisUtil.lock(key, () -> {
                // 再次尝试从缓存获取，避免集群多实例重复请求获取auth
                String newToken = (String) redisUtil.get(key);
                if (newToken != null) {
                    LOGGER.info(appid + ":获取现有token");
                    newAccessToken.set(newToken);
                }
                if (forceRefresh) {
                    LOGGER.info("微信：" + appid + ":进行强制刷新鉴权操作");
                } else {
                    LOGGER.info("微信：" + appid + ":进行鉴权操作");
                }
                //组织请求报文
                JSONObject content = new JSONObject();
                content.put("grant_type", "client_credential");
                content.put("appid", appid);
                content.put("secret", secret);
                content.put("force_refresh", forceRefresh);
                JSONObject result = new JSONObject(RestTools.post(WechatUrl.AUTH_URL, content));
                if (result.has("errcode")) {
                    throw new RuntimeException(result.toString());
                } else {
                    String accessToken = result.getString("access_token");
                    int expiresIn = result.getInt("expires_in");
                    redisUtil.set(key, accessToken, expiresIn - 180);
                    newAccessToken.set(accessToken);
                }
                return content;
            });
            return newAccessToken.get();
        }
    }

        /**
         * 清空api的调用quota
         *
         * @apiNote 本接口用于清空公众号/小程序/第三方平台等接口的每日调用接口次数
         */
    public JSONObject clearQuota(JSONObject authInfo) {
        JSONObject body = new JSONObject();
        body.put("appid", authInfo.getString("appid"));
        String accessToken = getAuthToken(authInfo.getString("appid"), authInfo.getString("secret"));
        String requestUrl = WechatUrl.CLEAR_QUOTA_URL + "?access_token=" + accessToken;
        return new JSONObject(RestTools.post(requestUrl, body));
    }

    public JSONObject clearQuota() {
        String orgName = InnerSystemConfig.System.getTenantName();
        JSONObject config = Config.getConfig(orgName);
        return clearQuota(config);
    }

    public JSONObject clearQuota(String orgName) {
        JSONObject config = Config.getConfig(orgName);
        return clearQuota(config);
    }

    public JSONArray clearQuotaAll() {
        JSONArray jsonArray = new JSONArray();
        String filiales = Config.wechatConfig.getString("openAccessTokenFiliale");
        if (filiales.isEmpty()){
            // 当前租户
            JSONObject wxConfig = Config.getConfig(InnerSystemConfig.System.getTenantName());
            jsonArray.put(clearQuota(wxConfig));
        } else {
            String[] filialeArrray = filiales.split("-");
            for (int i = 0; i < filialeArrray.length; i++) {
                JSONObject wxConfig = Config.getConfig(filialeArrray[i]);
                jsonArray.put(clearQuota(wxConfig));
            }
        }
        return jsonArray;
    }
}
