package com.aote.webmeter.tools.iot;

import com.af.plugins.HttpConnectionPoolUtil;
import com.aote.webmeter.enums.TelecomApiEnum;
import com.aote.webmeter.tools.WebMeterInfo;
import com.iotplatform.client.NorthApiException;
import com.iotplatform.client.dto.AuthOutDTO;
import com.iotplatform.constant.AuthConstant;
import com.iotplatform.constant.ExceptionEnum;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.log4j.Logger;
import org.json.JSONObject;

import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;
import java.io.IOException;
import java.io.InputStream;
import java.security.KeyStore;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 电信IOT平台 应用接口鉴权
 * @author Mr.river
 */
public class AuthenticationTools {

	private static final Logger LOGGER = Logger.getLogger(AuthenticationTools.class);

	/**
	 * 鉴权有效时间
	 */
	private static final Map<String,Long> STAMP = new ConcurrentHashMap<>();
	/**
	 * 鉴权TOKEN
	 */
	private static final Map<String,String> AOD = new ConcurrentHashMap<>();
	/**
	 * 超时时间值
	 */
	private static final Integer TOKEN_TIMEOUT_VALUE = 1000 * 60 * 30;

	/**
	 * 发送请求的客户端单例
 	 */
	private static CloseableHttpClient httpClient;

	static {
		setHttpClient();
	}

	/**
	 * 获取HTTP客户端单例
	 */
	public static CloseableHttpClient getHttpClient(){
		if(httpClient == null) {
			setHttpClient();
		}
		return httpClient;
	}

	/**
	 * 设置HTTP客户端
	 */
	private static synchronized void setHttpClient(){
		try {
			httpClient = HttpConnectionPoolUtil.getHttpClient(initSslConfig());
		} catch (NorthApiException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 获取TOKEN
	 * @return TOKEN值
	 */
	public static String getAuthToken() {
		long current = System.currentTimeMillis();
		String apiKey = WebMeterInfo.getString("telecomAPIKey");
		String accessToken = AOD.get(apiKey);
		Long stampValue = STAMP.get(apiKey);
		if(accessToken != null && stampValue != null && (current - stampValue) < TOKEN_TIMEOUT_VALUE) {
			LOGGER.info(apiKey+":获取现有token");
	    	return accessToken;
		} else {
			try {
				return getWebAuthToken();
			} catch (IOException e) {
				e.printStackTrace();
				return null;
			}
		}
	}

	/**
	 * 发起获取TOKEN的请求
	 * @return TOKEN信息
	 * @throws IOException IO异常
	 */
	private static String getWebAuthToken() throws IOException {
		String appId = WebMeterInfo.getString("telecomAPIKey");
		String secret = WebMeterInfo.getString("telecomSecret");

		LOGGER.info(appId + ":进行鉴权操作");
		Map<String, String> paramCommand = new HashMap<>(2);
		paramCommand.put("appId", appId);
		paramCommand.put("secret", secret);

		String responseBody = HttpConnectionPoolUtil.requestFormUrlEncoded(
				TelecomApiEnum.AUTH_URL.getValue(),
				paramCommand,
				new HttpPost(),
				httpClient
		);
		AuthOutDTO authOutDTO = new AuthOutDTO();
		authOutDTO.setAccessToken(new JSONObject(responseBody).getString("accessToken"));
		String newAccessToken = authOutDTO.getAccessToken();
		AOD.put(appId, newAccessToken);
		STAMP.put(appId, System.currentTimeMillis());
		return newAccessToken;
	}

	/**
	 * 初始化SSL证书
	 * @return SSL连接工厂
	 * @throws NorthApiException IOT平台请求异常
	 */
	protected static SSLConnectionSocketFactory initSslConfig() throws NorthApiException {
		try {
			InputStream isTrustCa = SignalDeliveryTools.class.getResourceAsStream("/ca.jks");
			InputStream isSelfCert = SignalDeliveryTools.class.getResourceAsStream("/outgoing.CertwithKey.pkcs12");
			KeyStore selfCert = KeyStore.getInstance("pkcs12");
			selfCert.load(isSelfCert, AuthConstant.SELFCERTPWD.toCharArray());
			KeyManagerFactory kmf = KeyManagerFactory.getInstance("sunx509");
			kmf.init(selfCert, AuthConstant.SELFCERTPWD.toCharArray());
			KeyStore caCert = KeyStore.getInstance("jks");
			caCert.load(isTrustCa, AuthConstant.TRUSTCAPWD.toCharArray());
			TrustManagerFactory tmf = TrustManagerFactory.getInstance("sunx509");
			tmf.init(caCert);
			SSLContext sc = SSLContext.getInstance("TLS");
			sc.init(kmf.getKeyManagers(), tmf.getTrustManagers(), null);
			SSLConnectionSocketFactory sslConnectionSocketFactory = new SSLConnectionSocketFactory(sc, NoopHostnameVerifier.INSTANCE);
			isTrustCa.close();
			isSelfCert.close();
			return sslConnectionSocketFactory;
		} catch (Exception var10) {
			throw new NorthApiException(ExceptionEnum.CLIENT_SSL_CONFIG_ERROR, var10.getMessage());
		}
	}
}
