package com.aote.webmeter.tools.iot;

import com.af.plugins.HttpConnectionPoolUtil;
import com.aote.webmeter.enums.IOTBusinessTypeEnum;
import com.aote.webmeter.enums.TelecomApiEnum;
import com.aote.webmeter.tools.WebMeterInfo;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.log4j.Logger;
import org.json.JSONObject;

import java.io.IOException;

/**
 * 电信IOT平台通用工具
 * @author Mr.river
 */
public class IotCommonTools {

    private static final Logger LOGGER = Logger.getLogger(IotCommonTools.class);

    /**
     * 是否AEP模式
     */
    private static final ThreadLocal<Boolean> IS_AEP;

    static {
        IS_AEP = new ThreadLocal<>();
    }

    /**
     * 错误标识
     */
    public static final String ERROR_PARAMS_KEY = "error_code";

    public static JSONObject post(IOTBusinessTypeEnum typeEnum, HttpEntityEnclosingRequestBase base, JSONObject contentObj) throws IOException {
        return post(typeEnum,base,contentObj,null);
    }
    public static JSONObject post(IOTBusinessTypeEnum typeEnum, HttpEntityEnclosingRequestBase base, JSONObject contentObj, String urlAppend) throws IOException {
        Boolean isAep = IS_AEP.get();
        if(isAep == null){
            isAep = false;
        }

        String header = null;
        String content = null;
        if(typeEnum != IOTBusinessTypeEnum.AUTH && typeEnum != IOTBusinessTypeEnum.REFRESH_AUTH){
            String accessToken = AuthenticationTools.getAuthToken();
            String appId = isAep ? WebMeterInfo.getString("aepAPIKey") : WebMeterInfo.getString("telecomAPIKey");
            JSONObject headerObj = new JSONObject();
            headerObj.put("app_key", appId);
            headerObj.put("Authorization", "Bearer" + " " + accessToken);
            headerObj.put("Content-Type","application/json");
            header = headerObj.toString();
        }

        if(contentObj != null){
            content = contentObj.toString();
        }

        String url;
        CloseableHttpClient client;
        //如果是AEP模式，使用AEP地址和客户端
        switch (typeEnum){
            case AUTH:
                if(!isAep){
                    url = TelecomApiEnum.AUTH_URL.getValue();
                } else {
                    url = TelecomApiEnum.AEP_AUTH_URL.getValue();
                }
                break;
            case REFRESH_AUTH:
                if(!isAep){
                    url = TelecomApiEnum.REFRESH_AUTH_URL.getValue();
                } else {
                    url = TelecomApiEnum.AEP_REFRESH_AUTH_URL.getValue();
                }
                break;
            case CREATE_DEVICE:
                if(!isAep){
                    url = TelecomApiEnum.DEVICE_CREATE_URL.getValue();
                } else {
                    url = TelecomApiEnum.AEP_DEVICE_CREATE_URL.getValue();
                }
                break;
            case MODIFY_DEVICE:
            case REMOVE_DEVICE:
                if(!isAep){
                    url = TelecomApiEnum.DEVICE_UPDATE_URL.getValue() + "/" + urlAppend;
                } else {
                    url = TelecomApiEnum.AEP_DEVICE_UPDATE_URL.getValue() + "/" + urlAppend;
                }
                break;
            case SEND_COMMAND:
                if(!isAep){
                    url = TelecomApiEnum.SEND_INSTRUCT_URL.getValue();
                } else {
                    url = TelecomApiEnum.AEP_SEND_INSTRUCT_URL.getValue();
                }
                break;
            case SUBSCRIPTIONS:
                if(!isAep){
                    url = TelecomApiEnum.SUBSCRIPTIONS_URL.getValue();
                } else {
                    url = TelecomApiEnum.AEP_SUBSCRIPTIONS_URL.getValue();
                }
                break;
            default:
                throw new RuntimeException("不支持的业务类型："+typeEnum.name());
        }
        if(isAep){
            client = HttpConnectionPoolUtil.getHttpClient();
        } else {
            client = AuthenticationTools.getHttpClient();
        }

        String responseBody;
        if(typeEnum == IOTBusinessTypeEnum.AUTH){
            assert contentObj != null;
            responseBody = HttpConnectionPoolUtil.requestFormUrlEncoded(url, contentObj, base, client);
        } else {
            LOGGER.debug("请求的地址："+ url);
            LOGGER.debug("请求头："+ header);
            LOGGER.debug("请求体："+ content);
            responseBody = HttpConnectionPoolUtil.request(url, content, header, base, client);
        }
        JSONObject result;
        if(responseBody != null){
            result = new JSONObject(responseBody);
        } else {
            result = new JSONObject();
        }
        LOGGER.debug("调用平台API结果：" + result);
        return result;
    }

    /**
     * 清除设置的AEP模式
     */
    public static void removeIsAep(){
        IS_AEP.remove();
    }

    /**
     * 设置AEP模式
     */
    public static void setIsAep(Boolean isAep){
        IS_AEP.set(isAep);
    }

    /**
     * 获取AEP模式
     */
    public static Boolean getIsAep(){
        Boolean b = IS_AEP.get();
        if (b== null){
            return false;
        }
        return b;
    }
}
