package com.aote.webmeter.tools;

import com.af.plugins.JsonTools;
import com.aote.webmeter.enums.ModuleEnum;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;

/**
 * 物联网表指令-逻辑映射工具（V1.2）
 * 指令-逻辑映射配置：mainInstructs.json
 *
 * @author Mrriver
 * @date 2020/6/18
 */
public class GetInstructsTools {
    private static final Logger LOGGER = Logger.getLogger(GetInstructsTools.class);
    /**
     * 指令-逻辑映射配置路径
     */
    private static final String CONFIG_URL = "webmeter/mainInstructs.json";
    private static final String TEMPLATE_CONFIG_URL = "webmeter/mainInstructTemplates.json";
    /**
     * 指令-逻辑映射列表
     */
    private static final JSONObject INSTRUCT_LIST;
    /**
     * 指令模板-逻辑映射列表
     */
    private static final JSONObject INSTRUCT_TEMPLATE_LIST;

    static {
        if (GetInstructsTools.class.getClassLoader().getResourceAsStream(CONFIG_URL) != null) {
            INSTRUCT_LIST = JsonTools.readJsonFile(CONFIG_URL);
            LOGGER.debug("加载指令-逻辑映射配置文件完成");
        } else {
            throw new RuntimeException("缺少指令-逻辑映射配置文件：mainInstructs.json");
        }
        if (GetInstructsTools.class.getClassLoader().getResourceAsStream(TEMPLATE_CONFIG_URL) != null) {
            INSTRUCT_TEMPLATE_LIST = JsonTools.readJsonFile(TEMPLATE_CONFIG_URL);
            LOGGER.debug("加载指令模板-逻辑映射配置文件完成");
        } else {
            throw new RuntimeException("缺少指令模板-逻辑映射配置文件：mainInstructTemplates.json");
        }
    }

    /**
     * 获取指令-业务逻辑名
     *
     * @param instructName 指令中文名
     * @return 业务逻辑名
     */
    public static String getInstruct(String instructName) {
        if (INSTRUCT_LIST.has(instructName)) {
            String moduleName = WebMeterInfo.getModuleName();
            //如果是公共模块，代表测试用
            if (ModuleEnum.toType(moduleName) == ModuleEnum.PUBLIC_MODULE) {
                return "debug";
            }
            String logicName = INSTRUCT_LIST.getString(instructName);
            return moduleName + logicName;
        } else {
            throw new RuntimeException("未映射名为【" + instructName + "】的指令");
        }
    }

    /**
     * 获取指令模板-业务逻辑名
     *
     * @param instructName 指令中文名
     * @return 业务逻辑名
     */
    public static String getInstructTemplate(String instructName) {
        if (INSTRUCT_TEMPLATE_LIST.has(instructName)) {
            String moduleName = WebMeterInfo.getModuleName();
            //如果是公共模块，代表测试用
            if (ModuleEnum.toType(moduleName) == ModuleEnum.PUBLIC_MODULE) {
                return "syncBasicRunTemplate";
            }
            return INSTRUCT_TEMPLATE_LIST.getString(instructName);
        } else {
            //返回通用执行指令模板
            return "syncBasicRunTemplate";
        }
    }

    /**
     * 获取待执行指令
     *
     * @param instructs 待执行的指令列表
     * @return 待执行指令
     */
    public static JSONObject getWaitSendInstruct(JSONArray instructs) {
        // 如果待执行命令数量大于1，并指定了高优先级队列，则优先寻找指定高优先级命令类型的命令
        String highPriorityCommandTypeStr = WebMeterInfo.getString("highPriorityCommandTypeList");
        if (highPriorityCommandTypeStr != null) {
            JSONArray highPriorityCommandList = new JSONArray(highPriorityCommandTypeStr);
            if (instructs.length() > 1 && highPriorityCommandList.length() > 0) {
                for (Object item : instructs) {
                    JSONObject waitInstruct = (JSONObject) item;
                    String waitInstructType = waitInstruct.getString("f_instruct_type");
                    for (Object instructTypeItem : highPriorityCommandList) {
                        String highPriorityInstructType = (String) instructTypeItem;
                        if (waitInstructType.equals(highPriorityInstructType)) {
                            return waitInstruct;
                        }
                    }
                }
            }
        }
        return instructs.length() > 0 ? instructs.getJSONObject(0) : null;
    }
}
