package com.aote.webmeter.common.dao;

import com.af.plugins.DateTools;
import com.aote.sql.SqlServer;
import com.aote.util.JsonHelper;
import com.aote.webmeter.common.basic.manage.param.instructmanage.SetInstructStateParam;
import com.aote.webmeter.common.basic.manage.result.GetWaitInstructResult;
import com.aote.webmeter.common.dto.GetInstructDto;
import com.aote.webmeter.common.utils.Condition;
import com.aote.webmeter.enums.SaleVersionEnum;
import com.aote.webmeter.enums.business.InstructInputtorEnum;
import com.aote.webmeter.enums.business.InstructStateEnum;
import com.aote.webmeter.enums.business.InstructTypeEnum;
import com.aote.webmeter.tools.CompatTools;
import com.aote.webmeter.tools.WebMeterInfo;
import lombok.SneakyThrows;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashMap;
import java.util.Optional;

/**
 * 指令数据库操作
 *
 * @author Mr.river
 */
@Transactional(rollbackFor=Exception.class)
@Component
public class InstructDao extends AbstractDao {
    private final SqlServer sqlServer;

    public InstructDao(SqlServer sqlServer) {
        this.sqlServer = sqlServer;
    }

    /**
     * 获取待发送的指令信息
     * @param condition 条件表达式
     * @return 获取待执行指令返回结果
     * @throws Exception 查询异常
     */
    @SneakyThrows
    public GetWaitInstructResult getWaitInstructs(Condition condition) {
        //获取营收系统版本名
        SaleVersionEnum saleName = WebMeterInfo.getSaleNameEnum();
        //SQL名称
        String sqlName;
        //表档案ID列名
        String userIdColumn = null;
        switch (saleName){
            case SILVER_LIGHT:
            case SILVER_LIGHT2:
                sqlName = "yinGuangGetInstruct";
                break;
            default:
                userIdColumn = CompatTools.getUserFilesId();
                sqlName = "getInstruct";
                break;
        }
        //查询待发送的指令
        JSONObject sqlParams = new JSONObject();
        sqlParams.put("userIdColumn", userIdColumn);
        sqlParams.put("condition", condition);
        JSONArray instructs = sqlServer.query(sqlName,sqlParams);
        int nums = instructs.length();
        return new GetWaitInstructResult(nums,
                JsonHelper.toParseList(instructs, GetInstructDto.class));
    }

    /**
     * 更新冗余的阀控指令
     * @param userId 表档案ID
     */
    public void updateRedundancyValveControlInstruct(String userId){
        String userIdColumn = CompatTools.getUserFilesId();
        //获取指令表名称
        String tableName = CompatTools.getInstructTableName();
        HashMap<String,Object> params = new HashMap<>(1);
        params.put("f_instruct_state", InstructStateEnum.CANCELED.getValue());
        sqlServer.run(formatUpdateSql(tableName,params,
                Condition.build()
                        .eq("f_instruct_state",InstructStateEnum.READY.getValue()).and()
                        .eq(userIdColumn,userId).and()
                        .eq("f_instruct_type",InstructTypeEnum.ValveControl.getValue()),
                true));
    }

    /**
     * 更新冗余的其他指令
     * @param userId 表档案ID
     */
    public void updateRedundancyOtherInstruct(String userId, InstructTypeEnum instructTypeEnum, String instructTitle){
        String userIdColumn = CompatTools.getUserFilesId();
        //获取指令表名称
        String tableName = CompatTools.getInstructTableName();
        HashMap<String,Object> params = new HashMap<>(1);
        params.put("f_instruct_state", InstructStateEnum.CANCELED.getValue());
        sqlServer.run(formatUpdateSql(tableName,params,
                Condition.build()
                        .eq("f_instruct_state",InstructStateEnum.READY.getValue()).and()
                        .eq(userIdColumn,userId).and()
                        .eq("f_instruct_type",instructTypeEnum.getValue()).and()
                        .asFirst(Condition.build().isNull("f_instruct_title").or()
                                .eq("f_instruct_title",instructTitle)),
                true));
    }

    /**
     * 更新尚未响应结果的临时指令状态为无响应
     * @param userId 表档案ID
     */
    public void updateNoResponseTemporaryInstruct(String userId){
        String userIdColumn = CompatTools.getUserFilesId();
        //获取指令表名称
        String tableName = CompatTools.getInstructTableName();
        HashMap<String,Object> params = new HashMap<>(2);
        params.put("f_instruct_state", InstructStateEnum.NO_RESPONSE.getValue());
        params.put("f_receive_state", InstructStateEnum.NO_RESPONSE.getReceiveMsg());
        sqlServer.run(formatUpdateSql(tableName,params,
                Condition.build()
                        .eq(userIdColumn,userId).and()
                        .eq("f_instruct_state",InstructStateEnum.ALREADY_SENT.getValue()).and()
                        .eq("f_inputtor", InstructInputtorEnum.SYSTEM_DEFAULT.getValue()),
                true));
    }
    /**
     * 更新尚未响应结果和响应超时的持久指令状态为待发送（重发）
     * @param userId 表档案ID
     */
    public void updateNoResponseAndTimeOutLastingInstruct(String userId){
        String userIdColumn = CompatTools.getUserFilesId();
        //获取指令表名称
        String tableName = CompatTools.getInstructTableName();
        HashMap<String,Object> params = new HashMap<>(1);
        params.put("f_instruct_state", InstructStateEnum.READY.getValue());
        sqlServer.run(formatUpdateSql(tableName,params,
                Condition.build()
                        .eq(userIdColumn,userId).and()
                        .in("f_instruct_state",
                                InstructStateEnum.ALREADY_SENT.getValue(),
                                InstructStateEnum.TIME_OUT.getValue()).and()
                        .neq("f_inputtor", InstructInputtorEnum.SYSTEM_DEFAULT.getValue()),
                true));
    }
    /**
     * 更新尚未发送的临时指令状态为忽略冗余
     * @param userId 表档案ID
     */
    public void updateReadyTemporaryInstruct(String userId){
        String userIdColumn = CompatTools.getUserFilesId();
        //获取指令表名称
        String tableName = CompatTools.getInstructTableName();
        HashMap<String,Object> params = new HashMap<>(1);
        params.put("f_instruct_state", InstructStateEnum.CANCELED.getValue());
        params.put("f_receive_state", InstructStateEnum.CANCELED.getReceiveMsg());
        sqlServer.run(formatUpdateSql(tableName,params,
                Condition.build()
                        .eq(userIdColumn,userId).and()
                        .eq("f_instruct_state",InstructStateEnum.READY.getValue()).and()
                        .eq("f_inputtor", InstructInputtorEnum.SYSTEM_DEFAULT.getValue()),
                true));
    }

    /**
     * 更新指令状态
     * @param param 设置指令状态参数器
     * @return 影响行数
     */
    public int updateInstructState(SetInstructStateParam param){
        //SQL片段 更新指令发送，响应时间
        String sendDate = null;
        String callBackDate = null;
        String now = DateTools.getNow2();
        switch (param.getState()){
            case SUCCESSFUL:
            case FAILED:
                sendDate = Condition.SQL_GRAMMAR_TAG + "isnull(f_send_date,'"+now+"'),";
                callBackDate = now;
                break;
            case ALREADY_SENT:
                sendDate = now;
                break;
            case TIME_OUT:
                callBackDate = now;
                break;
            default:
                break;
        }
        sendDate = Optional.ofNullable(sendDate).orElse(Condition.SQL_GRAMMAR_TAG + "f_send_date");
        callBackDate = Optional.ofNullable(callBackDate).orElse(Condition.SQL_GRAMMAR_TAG + "f_callback_date");
        //获取指令表名称
        String tableName = CompatTools.getInstructTableName();
        HashMap<String,Object> params = new HashMap<>(6);
        params.put("f_instruct_state",param.getState().getValue());
        params.put("f_receive_state",param.getReceiveMsg());
        params.put("f_send_date",sendDate);
        params.put("f_callback_date",callBackDate);
        params.put("f_instruct_meta_data",param.getMetaData());
        params.put("f_commandId", param.getSyncCommandId());
        return sqlServer.runSQL(formatUpdateSql(tableName,params,param.getCondition(),true));
    }
}
