package com.aote.webmeter.tools;

import org.json.JSONArray;
import org.json.JSONObject;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;
import java.util.Stack;
import java.util.concurrent.atomic.AtomicInteger;

import static com.aote.webmeter.tools.EncodingUtil.hexToBinaryWithReversedBits;
import static com.aote.webmeter.tools.EncodingUtil.reverseBinaryString;

public class IotCodeService {

    /**
     * 如果input是数字,乘以10的exponent倍返回,否则返回-1
     */
    public static long calculatePowerOfTen(Object input, int exponent) {
        if (input instanceof Number) {
            BigDecimal value = new BigDecimal(input.toString());
            BigDecimal exponentValue = new BigDecimal("10").pow(exponent);
            BigDecimal result = value.multiply(exponentValue);
            // 如果有小数去除小数
            if (result.scale() > 0) {
                result = result.setScale(0, RoundingMode.DOWN);
            }
            // 将结果转换为 long 类型，处理可能的精度问题
            try {
                return result.longValueExact();
            } catch (Exception e) {
                // 处理转换异常，如果超出了 long 范围
                return -1;
            }
        } else if (input instanceof String) {
            try {
                BigDecimal value = new BigDecimal((String) input);
                return calculatePowerOfTen(value, exponent);
            } catch (NumberFormatException e) {
                // 处理字符串转换为数字的异常
                return -1;
            }
        } else {
            return -1;
        }
    }

    /**
     * 计算value除以10的exponent倍
     */
    public static BigDecimal calculateValue(long value, int exponent) {
        // 将value转换为BigDecimal
        BigDecimal bdValue = new BigDecimal(value);

        // 计算10的exponent次方
        BigDecimal bdExponent = BigDecimal.TEN.pow(exponent);

        // 将value除以10的exponent次方
        return bdValue.divide(bdExponent, exponent, RoundingMode.DOWN);
    }

    /**
     * 物联网通信数据编码
     *
     * @param instructType 指令类型
     * @param params       指令参数
     * @return 返回值{"code":0,"data":"","err":""},code不为0时出错,错误信息在err
     */
    public JSONObject encodeInstruction(String instructType, JSONObject params) throws Exception {
        //获取配置文件
        JSONObject modelConf = getModelConf();
        return encodeInstruction(instructType, params, modelConf);
    }

    public JSONObject encodeInstruction(String instructType, JSONObject params, JSONObject config) throws Exception {
        //加载指令
        JSONObject instructConf = getInstructConf(config, instructType);
        if (instructConf == null) {
            return ENCODE_RESULT.INVALID_INSTRUCT.getDescription();
        }
        //开始解析配置内容
        return parseNGenerate(instructConf, params);
    }

    /**
     * 物联网通信数据解码
     *
     * @param uploadData 表上报的数据
     * @return 返回值{"code":0,"data":{},"err":""},code不为0时出错,错误信息在err
     */
    public JSONObject decodeInstruction(String uploadData) throws Exception {
        //获取配置文件
        JSONObject modelConf = getModelConf();
        return decodeInstruction(uploadData, modelConf);
    }

    public JSONObject decodeInstruction(String uploadData, JSONObject config) throws Exception {
        //加载指令
        JSONObject instructConf = getInstructConf_Decode(config, uploadData);
        if (instructConf == null) {
            return ENCODE_RESULT.INVALID_INSTRUCT.getDescription();
        }
        //开始解析配置内容
        return parseNGenerate_Decode(instructConf, uploadData);
    }

    /**
     * 解析配置并生成返回内容
     *
     * @param instructConf 指令配置
     * @param uploadData   上报字节码
     * @return 生成的字节码, 转换成了hex字符串
     */
    private JSONObject parseNGenerate_Decode(JSONObject instructConf, String uploadData) throws Exception {
        // 生成字节码的格式信息
        String responseBody = instructConf.getString("responseBody");
        // 返回值的类型,如果获取不到,则返回字符串
        JSONObject params = instructConf.getJSONObject("params");
        // 返回值
        JSONObject result = new JSONObject();
        // 开始遍历格式
        String[] dataFormats = responseBody.split(";");
        // 当前解析的下标位置
        AtomicInteger index = new AtomicInteger(0);
        // 循环标记,可认为是循环的名称
        StringBuilder loopTag = new StringBuilder();
        // 循环体,只有在循环的时候使用,记录内容
        StringBuilder loopBody = new StringBuilder();
        // 嵌套层数,用于多层嵌套循环体
        AtomicInteger nestingLevel = new AtomicInteger(0);
        // 循环体循环次数保存,目前的程序只能保存当前层的,即如果有双层嵌套循环,内部循环的次数必须生命在同级
        JSONObject loopCounts = new JSONObject();

        for (String item : dataFormats) {
            // 循环处理
            if (loopDevice(item, loopTag, loopBody, nestingLevel, loopCounts, index, uploadData, instructConf, result)) {
                continue;
            }
            // 变量加载,可以给配置中加载已经获取内容的对象
            item = parseNestedParentheses1(item, result);

            // 额外引用,引用会自动处理,直接进入下一个参数
            if (parseNestedParentheses2(item, instructConf, result, index, uploadData)) {
                continue;
            }

            String[] dataFormat = item.split(",");
            if (dataFormat.length < 3) {
                index.addAndGet(item.length());
                continue;
            }

            // 本条参数名称
            String dataFormatName = getDataFormatName(dataFormat);
            // 输入字节码的类型
            DATA_TYPE_ENCODE type_upload = getEncodeDataType(dataFormat);
            // 长度,一般是指字节,所以对于字符串长度要*2
            int dataFormatLen = getEncodeDataLength(dataFormat) * 2;
            // 截取的原始数据
            String data = uploadData.substring(index.get(), index.get() + dataFormatLen);

            // 二进制格式的特殊处理
            if (type_upload == DATA_TYPE_ENCODE.BIT) {
                //去掉dataFormat里的无效项
                String[] newDataFormat = new String[3];
                System.arraycopy(dataFormat, 0, newDataFormat, 0, 3);
                dataFormat = newDataFormat;


                int index_bit = 0;
                String[] dataFormats_bit = item.split("\\|");
                // 16进制字符串转二进制字符串
                String data_bin = EncodingUtil.hexToBinary(data);
                for (int i = 1; i < dataFormats_bit.length; i++) {
                    String dataFormat_bit = dataFormats_bit[i];
                    String[] dataFormat_bit_param = dataFormat_bit.split(",");
                    if (dataFormat_bit_param.length < 2) {
                        index_bit += dataFormat_bit.length();
                        continue;
                    }
                    // 重新加载子项目名称
                    dataFormatName = getDataFormatName(dataFormat_bit_param);
                    // 加载返回值的类型
                    DATA_TYPE_DECODE type_result = getDecodeDataType(dataFormatName, params);
                    // 重新加载子项目长度
                    int dataFormatLen_bit = Integer.parseInt(dataFormat_bit_param[1]);
                    String data_bit = data_bin.substring(index_bit, index_bit + dataFormatLen_bit);
                    // 处理完后修改index,准备下一轮循环
                    index_bit += dataFormatLen_bit;
                    // 新加载子项目放大倍数,内容为10的次幂
                    int exponent = dataFormat_bit_param.length > 2 ? Integer.parseInt(dataFormat_bit_param[2]) : 0;
                    // 截取需要的2进制数据
                    getValues(dataFormatName, type_upload, type_result, exponent, params, data_bit, result);
                }
            }
            // 16进制,正常处理
            else {
                // 加载返回值的类型
                DATA_TYPE_DECODE type_result = getDecodeDataType(dataFormatName, params);
                // 数字放大倍数,内容为10的次幂
                int exponent = getNumExponent(dataFormat);
                // 解析
                getValues(dataFormatName, type_upload, type_result, exponent, params, data, result);
            }
            // 加载循环次数
            if (dataFormat.length >= 5) {
                loopCounts.put(dataFormat[4], result.getInt(dataFormatName));
            }
            // 处理完后修改index,准备下一轮循环
            index.addAndGet(dataFormatLen);

        }
        result.put("DecodeLastIndex_ForRecursion", index.get());
        return ENCODE_RESULT.SUCCESS.getDescription(result);
    }

    /**
     * 分层提取括号内内容并且做处理
     *
     * @param input
     * @return
     * @throws Exception
     */
    public String parseNestedParentheses1(String input, JSONObject data)
            throws Exception {
        Stack<Integer> stack = new Stack<>();
        StringBuilder result = new StringBuilder(input);

        for (int i = 0; i < result.length(); i++) {
            char c = result.charAt(i);

            if (c == '(') {
                stack.push(i);
            } else if (c == ')') {
                if (stack.isEmpty()) {
                    throw new Exception("Mismatched parentheses");
                }

                int start = stack.pop();
                String innerExpression = result.substring(start + 1, i);
                String evaluatedExpression = evaluateExpression(innerExpression, data);
                result.replace(start, i + 1, evaluatedExpression);

                i = start + evaluatedExpression.length() - 1; // Adjust the current position
            }
        }

        if (!stack.isEmpty()) {
            throw new Exception("Mismatched parentheses");
        }

        return result.toString();
    }

    /**
     * 分层提取大括号内容并且做处理
     *
     * @param input
     * @return
     * @throws Exception
     */
    public boolean parseNestedParentheses2(String input, JSONObject data, JSONObject result_final,
                                           AtomicInteger index, String uploadData)
            throws Exception {
        Stack<Integer> stack = new Stack<>();
        StringBuilder input_ = new StringBuilder(input);
        boolean res = false;
        for (int i = 0; i < input_.length(); i++) {
            char c = input_.charAt(i);

            if (c == '{') {
                stack.push(i);
            } else if (c == '}') {
                if (stack.isEmpty()) {
                    throw new Exception("Mismatched parentheses");
                }
                res = true;
                int start = stack.pop();
                String innerExpression = input_.substring(start + 1, i);
                evaluateExpression2(innerExpression, data, result_final, index, uploadData);
            }
        }
        return res;
    }

    private String evaluateExpression(String expression, JSONObject data) {
        // 这里可以根据需要替换成更复杂的逻辑。
        // 在此处只是从data中拿出已存在的变量内容,并代替
        return data.getString(expression);
    }

    private void evaluateExpression2(String expression, JSONObject data, JSONObject result, AtomicInteger index,
                                     String uploadData) throws Exception {
        // 此处从data中拿出additionalConf配置并递归调用
        // 组织递归数据
        String[] expression_ = expression.split(",");
        JSONObject additionalConf = data.getJSONObject("additionalConf");
        for (int i = 0; i < expression_.length - 1; i++) {
            additionalConf = additionalConf.getJSONObject(expression_[i]);
        }
        String conf = additionalConf.getString(expression_[expression_.length - 1]);
        JSONObject instructConf = new JSONObject(data.toString());
        instructConf.put("responseBody", conf);
        // 截取的原始数据,以当前位置开始,截取后面全部数据
        String uploadData_Recursion = uploadData.substring(index.get());
        JSONObject result_Recursion = parseNGenerate_Decode(instructConf, uploadData_Recursion);
        JSONObject result_Recursion_data = result_Recursion.getJSONObject("data");
        int offset = result_Recursion_data.getInt("DecodeLastIndex_ForRecursion");
        result_Recursion_data.remove("DecodeLastIndex_ForRecursion");
        // 正式装填结果
        EncodingUtil.mergeJSONObjects(result, result_Recursion);
        // 下标后移
        index.addAndGet(offset);
    }

    /**
     * 解析配置并生成返回内容
     *
     * @param item          字节码块配置内容
     * @param loopTag       循环块标志/名称
     * @param loopBody      新构建的循环内配置
     * @param nestingLevel  循环嵌套层数
     * @param loopCounts    已解析出的循环次数
     * @param index         当前上报数据字节码下标位置
     * @param uploadData    当前上报数据字节码内容
     * @param instructConf_ 参数
     * @param result        返回的内容
     * @return 是否开始或者处于循环中
     * @throws Exception
     */
    private boolean loopDevice(String item, StringBuilder loopTag, StringBuilder loopBody, AtomicInteger nestingLevel,
                               JSONObject loopCounts, AtomicInteger index, String uploadData, JSONObject instructConf_,
                               JSONObject result) throws Exception {
        boolean res = false;

        List<Integer> startSymbol = EncodingUtil.findCharPositions(item, '<');
        int startIndex = !startSymbol.isEmpty() ? startSymbol.get(0) + 1 : 0;
        List<Integer> endSymbol = EncodingUtil.findCharPositions(item, '>');
        int endIndex = !endSymbol.isEmpty() ? endSymbol.get(endSymbol.size() - 1) : item.length();
        // 有起始符号
        if (!startSymbol.isEmpty()) {
            // 当前未开始一个循环
            if (nestingLevel.get() == 0) {
                loopTag.setLength(0);
                loopTag.append(item, 0, startIndex - 1);
            }
            // 当前已经开始一个循环
            else {
                // 已开始循环,但依旧存在startSymbol,说明是嵌套,不作截取,让递归处理
                startIndex = 0;
            }
            // 记录循环层数
            nestingLevel.addAndGet(startSymbol.size());
        }
        // 有结束符号
        if (!endSymbol.isEmpty()) {
            // 退出一定的循环层数
            nestingLevel.addAndGet(-endSymbol.size());
            // 拼接数据处理
            if (nestingLevel.get() == 0) {
                item = item.substring(startIndex, endIndex);
                loopBody.append(item);
                res = true;

                // 此处做结束逻辑处理,开始递归
                // 加载循环次数
                int loopcount = loopCounts.getInt(loopTag.toString());
                // 组织递归数据
                JSONObject instructConf_loop = new JSONObject();
                instructConf_loop.put("responseBody", loopBody.toString());
                instructConf_loop.put("params", instructConf_.get("params"));
                instructConf_loop.put("additionalConf", instructConf_.optJSONObject("additionalConf"));
                JSONArray result_Loop = new JSONArray();
                // 正式开始循环体
                for (int i = 0; i < loopcount; i++) {
                    // 截取的原始数据,以当前位置开始,截取后面全部数据
                    String uploadData_Recursion = uploadData.substring(index.get());
                    JSONObject result_Recursion = parseNGenerate_Decode(instructConf_loop, uploadData_Recursion);
                    JSONObject result_Recursion_data = result_Recursion.getJSONObject("data");
                    int offset = result_Recursion_data.getInt("DecodeLastIndex_ForRecursion");
                    result_Recursion_data.remove("DecodeLastIndex_ForRecursion");
                    result_Loop.put(result_Recursion_data);
                    // 下标后移
                    index.addAndGet(offset);
                }
                // 正式装填结果
                result.put(loopTag.toString(), result_Loop);

            } else {
                // 包含嵌套.为子循环,子循环不作处理,让递归处理
                endIndex = item.length();
            }
        }
        if (nestingLevel.get() > 0) {
            // 未结束,拼接
            loopBody.append(item, startIndex, endIndex).append(";");
            res = true;
        }
        return res;
    }

    /**
     * 按照配置解析内容
     *
     * @param dataFormatName 参数名
     * @param type_upload    上传的字节码类型
     * @param type_result    返回的参数类型
     * @param exponent       上报数据放大或缩小的10的指数
     * @param uploadData     上报的数据
     * @param params         配置中指定的返回类型
     * @param result         返回数据,JSON格式
     */
    private void getValues(String dataFormatName, DATA_TYPE_ENCODE type_upload, DATA_TYPE_DECODE type_result, int exponent, JSONObject params, String uploadData, JSONObject result) throws Exception {
        if (type_result == DATA_TYPE_DECODE.LONG || type_result == DATA_TYPE_DECODE.DOUBLE) {
            //返回值需要数字,字节码不存在小数,所以统一先转成LONG
            BigDecimal value_ = getNumValue(type_upload, uploadData, exponent);
            if (type_result == DATA_TYPE_DECODE.LONG) {
                result.put(dataFormatName, value_.longValue());
            } else {
                result.put(dataFormatName, value_.doubleValue());
            }
        }
        //字符串格式
        else if (type_result == DATA_TYPE_DECODE.STRING) {
            if (type_upload == DATA_TYPE_ENCODE.BCD) {
                result.put(dataFormatName, uploadData);
            } else if (type_upload == DATA_TYPE_ENCODE.HEX || type_upload == DATA_TYPE_ENCODE.HEX_LE || type_upload == DATA_TYPE_ENCODE.BIT) {
                BigDecimal value_ = getNumValue(type_upload, uploadData, exponent);
                result.put(dataFormatName, value_.toString());
            }
        } else if (type_result == DATA_TYPE_DECODE.STRING_BIT) {
            if (type_upload == DATA_TYPE_ENCODE.BIT) {
                result.put(dataFormatName, reverseBinaryString(uploadData));
            } else {
                result.put(dataFormatName, hexToBinaryWithReversedBits(uploadData));
            }
        }
    }

    private BigDecimal getNumValue(DATA_TYPE_ENCODE type_upload, String uploadData, int exponent) {
        long value = 0L;
        if (type_upload == DATA_TYPE_ENCODE.BCD) {
            value = Long.parseLong(uploadData);
        } else if (type_upload == DATA_TYPE_ENCODE.HEX) {
            value = Long.parseLong(uploadData, 16);
        } else if (type_upload == DATA_TYPE_ENCODE.HEX_LE) {
            value = Long.parseLong(EncodingUtil.reverseHexStringByEndian(uploadData, EncodingUtil.Endian.LITTLE_ENDIAN), 16);
        } else if (type_upload == DATA_TYPE_ENCODE.BIT) {
            value = Long.parseLong(uploadData, 2);
        }
        return calculateValue(value, exponent);
    }

    /**
     * 解析配置并生成字节码
     *
     * @param instructConf 指令配置
     * @param param        传入参数
     * @return 生成的字节码, 转换成了hex字符串
     */
    private JSONObject parseNGenerate(JSONObject instructConf, JSONObject param) throws Exception {
        //生成字节码的格式信息
        String requestBody = instructConf.getString("requestBody");
        //默认参数的信息
        JSONObject var = instructConf.getJSONObject("var");
        //返回值的类型,如果获取不到,则返回字符串
        JSONObject params = instructConf.getJSONObject("params");
        //返回值
        StringBuilder result = new StringBuilder();
        //开始遍历格式
        String[] dataFormats = requestBody.split(";");
        for (String item : dataFormats) {
            if (!item.contains(",")) {
                //不含分隔符,也可以认为它是固定值
                result.append(item);
                continue;
            }
            String[] dataFormat = item.split(",");
            //本条参数名称
            String dataFormatName = getDataFormatName(dataFormat);
            //输出字节码的类型
            DATA_TYPE_ENCODE type_output = getEncodeDataType(dataFormat);
            //加载传入值的类型
            DATA_TYPE_DECODE type_input = getDecodeDataType(dataFormatName, params);
            //长度,一般是指字节,所以对于字符串长度要*2
            int dataFormatLen = getEncodeDataLength(dataFormat) * 2;

            //预计返回的字节数据
            String data = "";
            if (type_output == DATA_TYPE_ENCODE.HEX_CRC || type_output == DATA_TYPE_ENCODE.HEX_CRC_BE) {
                // CRC校验,获取CRC校验区间
                int[] CRCInterval = getCRCInterval(dataFormat);
                int beginIndex = CRCInterval[0] * 2;
                int endIndex = result.length();
                if (CRCInterval[1] > 0) {
                    endIndex = CRCInterval[1] * 2;
                }
                data = EncodingUtil.calculateCRC16(result.substring(beginIndex, endIndex), type_output == DATA_TYPE_ENCODE.HEX_CRC);
            } else {
                //二进制,需要循环遍历
                if (type_output == DATA_TYPE_ENCODE.BIT) {
                    StringBuilder data_bit_buffer = new StringBuilder();
                    String[] dataFormats_bit = item.split("\\|");
                    for (int i = 1; i < dataFormats_bit.length; i++) {
                        String data_bit = "";
                        String dataFormat_bit = dataFormats_bit[i];
                        if (!dataFormat_bit.contains(",")) {
                            //不含分隔符,也可以认为它是固定值
                            data_bit_buffer.append(dataFormat_bit);
                            continue;
                        }
                        String[] dataFormat_bit_param = dataFormat_bit.split(",");
                        //重新加载子项目名称
                        dataFormatName = getDataFormatName(dataFormat_bit_param);
                        //重新加载传入值的类型
                        type_input = getDecodeDataType(dataFormatName, params);
                        //重新加载子项目长度
                        int dataFormatLen_bit = Integer.parseInt(dataFormat_bit_param[1]);
                        //新加载子项目放大倍数,内容为10的次幂
                        int exponent_bit = dataFormat_bit_param.length > 2 ? Integer.parseInt(dataFormat_bit_param[2]) : 0;
                        //先从默认值中加载
                        data_bit = loadValues(dataFormatName, type_output, type_input, dataFormatLen_bit, exponent_bit, var, data_bit);
                        //如果有传入参数,从param中加载,覆盖旧的
                        data_bit = loadValues(dataFormatName, type_output, type_input, dataFormatLen_bit, exponent_bit, param, data_bit);
                        data_bit_buffer.append(EncodingUtil.padLeft(data_bit, dataFormatLen_bit, '0', true));
                    }
                    //此处右侧补齐
                    data = EncodingUtil.padRight(data_bit_buffer.toString(), dataFormatLen * 4, '0', true);
                    //先把二进制转16进制
                    data = EncodingUtil.binaryToHex(data).toUpperCase();
                } else {
                    //数字放大倍数,内容为10的次幂
                    int exponent = getNumExponent(dataFormat);
                    //先从默认值中加载
                    data = loadValues(dataFormatName, type_output, type_input, dataFormatLen, exponent, var, data);
                    //如果有传入参数,从param中加载,覆盖旧的
                    data = loadValues(dataFormatName, type_output, type_input, dataFormatLen, exponent, param, data);
                }

            }
            result.append(data);

        }
        return ENCODE_RESULT.SUCCESS.getDescription(result.toString());
    }

    /**
     * 从默认配置中加载内容
     *
     * @param dataFormatName 参数名
     * @param type_output    在字节码中输出的类型
     * @param type_input     传入参数的类型
     * @param dataFormatLen  在字节码中占用的长度
     * @param exponent       放大或缩小的10的指数
     * @param source         数据源
     * @param data           返回数据,如果不修改则返回原数据
     * @return 返回字节码字符串
     */
    private String loadValues(String dataFormatName, DATA_TYPE_ENCODE type_output, DATA_TYPE_DECODE type_input, int dataFormatLen, int exponent, JSONObject source, String data) throws Exception {
        if (source.has(dataFormatName)) {
            Object value_ = source.get(dataFormatName);
            long value;
            //输入内容只有这一种特殊情况,当它是STRING_BIT的时候,先将其转换成数字,有溢出风险
            if (type_input == DATA_TYPE_DECODE.STRING_BIT) {
                //规定STRING_BIT低位在前,此处先反转
                String value_string_bit = reverseBinaryString((String) value_);
                value = Long.parseLong(value_string_bit, 2);
            } else {
                value = calculatePowerOfTen(value_, exponent);
            }
            //是数字,且转换成功
            if (value != -1) {
                //凡是能转换成数字的,皆可以BCD或者HEX返回
                if (type_output == DATA_TYPE_ENCODE.BCD) {
                    data = Long.toString(value);
                } else if (type_output == DATA_TYPE_ENCODE.HEX || type_output == DATA_TYPE_ENCODE.HEX_LE) {
                    data = Long.toHexString(value).toUpperCase();
                    if (type_output == DATA_TYPE_ENCODE.HEX_LE) {
                        //先补齐,再反转
                        data = EncodingUtil.padLeft(data, dataFormatLen, '0', true);
                        data = EncodingUtil.reverseHexStringByEndian(data, EncodingUtil.Endian.LITTLE_ENDIAN);
                    }
                } else if (type_output == DATA_TYPE_ENCODE.BIT) {
                    //字节码形式,只有数字可以,直接转回二进制
                    data = Long.toBinaryString(value);
                }
            } else if (value_ instanceof String) {
                //字符对象不包含数字,只能以BCD编码返回
                if (type_output == DATA_TYPE_ENCODE.BCD) {
                    data = source.getString(dataFormatName);
                }
            }
        }
        //返回前左侧补0凑足长度
        return EncodingUtil.padLeft(data, dataFormatLen, '0', true);
    }

    /**
     * 获取数字参数的放大倍数
     *
     * @param dataFormat 当前项的所有内容
     * @return 10的次幂
     */
    private int getNumExponent(String[] dataFormat) {
        if (dataFormat.length > 3) {
            return Integer.parseInt(dataFormat[3]);
        } else {
            return 0;
        }
    }

    /**
     * 获取在字节码中占用的长度
     *
     * @param dataFormat 当前项的所有内容
     * @return 长度
     * @throws Exception 未加载到时抛出异常
     */
    private int getEncodeDataLength(String[] dataFormat) throws Exception {
        if (dataFormat.length > 2) {
            return Integer.parseInt(dataFormat[2]);
        } else {
            throw new Exception("未知的参数长度->" + dataFormat[0]);
        }
    }

    /**
     * 获取CRC校验的区间,返回0,0的话,则校验位前的所有数据全都参与,否则第0位代表起始位置,第1位结束位置(不包括)
     *
     * @param dataFormat 当前项的所有内容
     * @return 起始位置和长度
     */
    private int[] getCRCInterval(String[] dataFormat) {
        int[] res = new int[]{0, 0};
        if (dataFormat.length > 3) {
            res[0] = Integer.parseInt(dataFormat[3]);
        }
        if (dataFormat.length > 4) {
            res[1] = Integer.parseInt(dataFormat[4]);
        }
        return res;
    }

    /**
     * 获取最终生成字节码的目标类型
     *
     * @param dataFormatName 当前项的名称
     * @param params         配置文件定义的类型
     * @return 类型, 未加载到指定的类型, 则返回String
     */
    private DATA_TYPE_DECODE getDecodeDataType(String dataFormatName, JSONObject params) throws Exception {
        if (params.has(dataFormatName)) {
            return DATA_TYPE_DECODE.identifyDataType(params.getString(dataFormatName));
        } else {
            return DATA_TYPE_DECODE.STRING;
        }
    }

    /**
     * 获取最终生成字节码的目标类型
     *
     * @param dataFormat 当前项的所有内容
     * @return 类型
     * @throws Exception 未加载到时抛出异常
     */
    private DATA_TYPE_ENCODE getEncodeDataType(String[] dataFormat) throws Exception {
        if (dataFormat.length > 1) {
            return DATA_TYPE_ENCODE.identifyDataType(dataFormat[1]);
        } else {
            throw new Exception("未知的参数类型->" + dataFormat[0]);
        }
    }

    /**
     * 获取当前项的参数名称
     *
     * @param dataFormat 当前项的所有内容
     * @return 名称
     * @throws Exception 未加载到时抛出异常
     */
    private String getDataFormatName(String[] dataFormat) throws Exception {
        if (dataFormat[0].isEmpty()) {
            throw new Exception("配置参数错误->" + dataFormat[0]);
        }
        return dataFormat[0];
    }

    /**
     * 按类型处理
     *
     * @param data 数据
     * @param type 类型信息
     * @return 处理过的数据
     */
    private String processDataByType(String data, DATA_TYPE_ENCODE type) {
        if (type == DATA_TYPE_ENCODE.BCD) {
            //bcd
        } else if (type == DATA_TYPE_ENCODE.HEX) {

        }
        //此处是否需要抛出异常?
        return "";
    }

    /**
     * 从模块配置中查找指令配置
     *
     * @param modelConf    模块配置
     * @param instructType 指令类型
     * @return 指令配置
     */
    private JSONObject getInstructConf(JSONObject modelConf, String instructType) {
        JSONArray data = modelConf.getJSONArray("data");
        for (Object itemObj : data) {
            JSONObject item = (JSONObject) itemObj;
            if (item.getString("name").equals(instructType)) {
                return item;
            }
        }
        return null;
    }

    /**
     * 从模块配置中查找指令配置
     *
     * @param modelConf  模块配置
     * @param uploadData 表上传数据
     * @return 指令配置
     */
    private JSONObject getInstructConf_Decode(JSONObject modelConf, String uploadData) throws Exception {
        //获取识别码的名称
        String key = modelConf.getString("key");
        //加载所有指令的配置
        JSONArray alldata = modelConf.getJSONArray("data");
        for (Object itemObj : alldata) {
            //加载一个指令
            JSONObject instructConf = (JSONObject) itemObj;

            String responseBody = instructConf.getString("responseBody");
            // 返回值的类型,如果获取不到,则返回字符串
            JSONObject params = instructConf.getJSONObject("params");
            // 返回值
            JSONObject result = new JSONObject();
            // 开始遍历格式
            String[] dataFormats = responseBody.split(";");
            // 当前解析的下标位置
            AtomicInteger index = new AtomicInteger(0);
            // 循环标记,可认为是循环的名称
            StringBuilder loopTag = new StringBuilder();
            // 循环体,只有在循环的时候使用,记录内容
            StringBuilder loopBody = new StringBuilder();
            // 嵌套层数,用于多层嵌套循环体
            AtomicInteger nestingLevel = new AtomicInteger(0);
            // 循环体循环次数保存,目前的程序只能保存当前层的,即如果有双层嵌套循环,内部循环的次数必须生命在同级
            JSONObject loopCounts = new JSONObject();

            //解析上报格式信息
            for (String item : dataFormats) {

                // 循环处理
                // 循环处理
                if (loopDevice(item, loopTag, loopBody, nestingLevel, loopCounts, index, uploadData, instructConf, result)) {
                    continue;
                }
                // 变量加载,可以给配置中加载已经获取内容的对象
                item = parseNestedParentheses1(item, result);

                // 额外引用,引用会自动处理,直接进入下一个参数
                if (parseNestedParentheses2(item, instructConf, result, index, uploadData)) {
                    continue;
                }
                String[] dataFormat = item.split(",");
                if (dataFormat.length < 3) {
                    index.addAndGet(item.length());
                    continue;
                }

                // 本条参数名称
                String dataFormatName = getDataFormatName(dataFormat);
                // 输入字节码的类型
                DATA_TYPE_ENCODE type_upload = getEncodeDataType(dataFormat);

                DATA_TYPE_DECODE type_input = getDecodeDataType(dataFormatName, params);
                //默认参数的信息
                JSONObject var = instructConf.getJSONObject("var");
                // 长度,一般是指字节,所以对于字符串长度要*2
                int dataFormatLen = getEncodeDataLength(dataFormat) * 2;
                // 截取的原始数据
                String data = uploadData.substring(index.get(), index.get() + dataFormatLen);


                // 二进制格式的特殊处理
                if (type_upload == DATA_TYPE_ENCODE.BIT) {
                    //去掉dataFormat里的无效项
                    String[] newDataFormat = new String[3];
                    System.arraycopy(dataFormat, 0, newDataFormat, 0, 3);
                    dataFormat = newDataFormat;
                } else {
                    //数字放大倍数,内容为10的次幂
                    int exponent = getNumExponent(dataFormat);
                    //查找到识别码所在的数据位置
                    if (dataFormatName.equals(key)) {
                        //找到识别码位置,并对识别码进行编码

                        String keyEncode = loadValues(dataFormatName, type_upload, type_input, dataFormatLen, exponent, var, "");
                        if (keyEncode.equals(data)) {
                            return instructConf;
                        } else {
                            //因为已经校验了本指令内的识别码,所以直接跳出进入洗衣歌指令
                            break;
                        }
                    }
                }
                if (dataFormat.length >= 5) {
                    loopCounts.put(dataFormat[4], result.getInt(dataFormatName));
                }
                // 处理完后修改index,准备下一轮循环
                index.addAndGet(dataFormatLen);
            }
        }
        return null;
    }

    /**
     * 获取配置文件
     *
     * @return 配置文件
     */
    private JSONObject getModelConf() {
        String configStr = WebMeterInfo.getString("coding");
        if (configStr != null) {
            return new JSONObject(configStr);
        } else {
            throw new RuntimeException("未找到模块对应的Coding配置");
        }
    }


    //数据类型
    enum DATA_TYPE_ENCODE {
        BCD,//一个字节表示两个0-9的数字,可以简单理解为字符串
        HEX,//16进制数字,跨多个字节时,默认使用大端序存储,即高位在前,等于HEX
        HEX_LE,//16进制数字,跨多个字节时,使用小端序存储,低位在前(当前本项目新奥表使用此类型)
        HEX_CRC,//校验位,当前只支持CRC-16校验
        HEX_CRC_BE,//校验位,当前只支持CRC-16校验,高位在前
        BIT//以二进制表示含义,当配置文件出现BIT时,BIT中的长度不再代表字节数,而是代表比特数,BIT配置请遵循高位在前
        ;

        private static DATA_TYPE_ENCODE identifyDataType(String DataType) throws Exception {
            try {
                Field field = DATA_TYPE_ENCODE.class.getField(DataType.toUpperCase());
                return (DATA_TYPE_ENCODE) field.get(null);
            } catch (NoSuchFieldException | IllegalAccessException e) {
                throw new Exception("未知的参数类型->" + DataType);
            }
        }
    }


    //数据类型
    enum DATA_TYPE_DECODE {
        LONG,
        DOUBLE,
        STRING,
        STRING_BIT;

        private static DATA_TYPE_DECODE identifyDataType(String DataType) throws Exception {
            if (DataType.equalsIgnoreCase("INT") || DataType.equalsIgnoreCase("LONG") || DataType.equalsIgnoreCase("SHORT")) {
                DataType = "LONG";
            }
            try {
                Field field = DATA_TYPE_DECODE.class.getField(DataType.toUpperCase());
                return (DATA_TYPE_DECODE) field.get(null);
            } catch (NoSuchFieldException | IllegalAccessException e) {
                throw new Exception("未知的参数类型->" + DataType);
            }
        }
    }

    /***
     * 返回值
     */
    enum ENCODE_RESULT {
        SUCCESS,
        INVALID_MODULE(1, "无效模块名"),
        INVALID_INSTRUCT(2, "无效指令类型"),
        EXCEPTION(999, "程序异常");

        private int code = 0;
        private String description = null;

        // 枚举构造函数
        ENCODE_RESULT(int code, String description) {
            this.code = code;
            this.description = description;
        }

        ENCODE_RESULT() {
        }

        // 获取枚举的描述
        public JSONObject getDescription(String data) {
            JSONObject res = new JSONObject("{\"code\":0,\"data\":\"\",\"err\":\"\"}");
            if (code != 0) {
                //生成异常返回
                res.put("code", code);
                res.put("err", description + ":" + data);
            } else {
                //生成正确返回
                res.put("data", data);
            }
            return res;
        }

        public JSONObject getDescription(JSONObject data) {
            JSONObject res = new JSONObject("{\"code\":0,\"data\":\"\",\"err\":\"\"}");
            //生成正确返回
            res.put("data", data);
            return res;
        }

        public JSONObject getDescription() {
            JSONObject res = new JSONObject("{\"code\":0,\"data\":\"\",\"err\":\"\"}");
            if (code != 0) {
                //生成异常返回
                res.put("code", code);
                res.put("err", description);
            }
            return res;
        }
    }
}
