package com.aote.webmeter.tools.iot.aep;

import com.af.expression.exception.ServiceException;
import com.af.plugins.JsonTools;
import com.aote.webmeter.tools.WebMeterInfo;
import com.ctg.ag.sdk.biz.AepDeviceCommandClient;
import com.ctg.ag.sdk.biz.AepDeviceCommandLwmProfileClient;
import com.ctg.ag.sdk.biz.AepDeviceManagementClient;
import com.ctg.ag.sdk.biz.aep_device_command.CreateCommandRequest;
import com.ctg.ag.sdk.biz.aep_device_command_lwm_profile.CreateCommandLwm2mProfileRequest;
import com.ctg.ag.sdk.biz.aep_device_management.CreateDeviceRequest;
import com.ctg.ag.sdk.biz.aep_device_management.DeleteDeviceRequest;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiFunction;

/**
 * 电信物联网API实现(原生AEP模式)
 */
@Component
public class AEPApiService {

    private final static Map<String, AEPClientMapItem> AEP_CLIENT_MAP_ITEM_MAP = new ConcurrentHashMap<>();

    private static final Logger LOGGER = LoggerFactory.getLogger(AEPApiService.class);

    private AEPClientMapItem getClientMap(String appKey) {
        if (!AEP_CLIENT_MAP_ITEM_MAP.containsKey(appKey)) {
            String appSecret = WebMeterInfo.getString(IotModuleConfigEnum.AEP_APP_SECRET.getValue());
            AEP_CLIENT_MAP_ITEM_MAP.put(appKey, new AEPClientMapItem(appKey, appSecret));
        }
        return AEP_CLIENT_MAP_ITEM_MAP.get(appKey);
    }

    /**
     * AEP TCP和LWM2M协议下发透传指令
     *
     * @param deviceId            设备ID
     * @param value               下发参数
     * @param aepProtocolTypeEnum 协议类型
     * @param serviceId           服务ID 非必填
     * @param method              方法名 非必填
     * @return 执行结果
     */
    public JSONObject sendCommand(String deviceId, Object value, AEPProtocolTypeEnum aepProtocolTypeEnum, String serviceId, String method) {
        JSONObject content = new JSONObject();
        if (serviceId != null && method != null) {
            return sendCommandByProfile(serviceId, method, deviceId, value);
        } else if (serviceId != null) {
            switch (aepProtocolTypeEnum) {
                case MQTT:
                case LWM2M:
                case TR069:
                    if (value instanceof JSONObject) {
                        content.put("params", value);
                    } else {
                        throw new ServiceException("非透传模式仅支持JSON格式数据");
                    }
                    content.put("serviceIdentifier", serviceId);
                    break;
                default:
                    throw new ServiceException("该协议不支持非透传模式：" + aepProtocolTypeEnum);
            }
        } else {
            switch (aepProtocolTypeEnum) {
                case MQTT:
                case TR069:
                    if (value instanceof JSONObject) {
                        content.put("payload", value);
                    } else {
                        throw new ServiceException("MQTT,TR069协议的透传模式仅支持JSON格式数据");
                    }
                    break;
                case LWM2M:
                case TCP:
                    if (value instanceof String) {
                        content.put("payload", value);
                    } else {
                        throw new ServiceException("LWM2M,TCP协议的透传模式仅支持String格式数据");
                    }
                    content.put("dataType", 2);
                    break;
                default:
                    throw new ServiceException("该协议不支持透传模式：" + aepProtocolTypeEnum);
            }
        }

        JSONObject requestContent = new JSONObject();
        requestContent.put("content", content);
        return sendCommandAction(deviceId, requestContent, (requestParams, aepClientMapItem) -> {
            CreateCommandRequest request = new CreateCommandRequest();
            request.addParamMasterKey(WebMeterInfo.getString(IotModuleConfigEnum.AEP_MASTER_KEY.getValue()));
            request.setBody(requestParams.toString().getBytes(StandardCharsets.UTF_8));
            byte[] resultByte;
            try {
                resultByte = aepClientMapItem.getAepDeviceCommandClient().CreateCommand(request).getBody();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
            return resultByte;
        });
    }

    public JSONObject sendCommand(String deviceId, Object value, String serviceId, String method) {
        return sendCommand(deviceId, value, AEPProtocolTypeEnum.LWM2M, serviceId, method);
    }

    public JSONObject sendCommand(String deviceId, Object value, String serviceId) {
        return sendCommand(deviceId, value, AEPProtocolTypeEnum.LWM2M, serviceId, null);
    }

    public JSONObject sendCommand(String deviceId, Object value) {
        return sendCommand(deviceId, value, AEPProtocolTypeEnum.LWM2M, null, null);
    }


    /**
     * lwm2m协议有profile指令下发接口
     *
     * @param serviceId 服务名
     * @param method    方法名
     * @param deviceId  设备ID
     * @param value     下发参数
     * @return 执行结果
     */
    public JSONObject sendCommandByProfile(String serviceId, String method, String deviceId, Object value) {
        JSONObject requestContent = new JSONObject();
        requestContent.put("command",
                new JSONObject().put("serviceId", serviceId).put("method", method).put("paras", value));
        return sendCommandAction(deviceId, requestContent, (requestParams, aepClientMapItem) -> {
            CreateCommandLwm2mProfileRequest request = new CreateCommandLwm2mProfileRequest();
            request.addParamMasterKey(WebMeterInfo.getString(IotModuleConfigEnum.AEP_MASTER_KEY.getValue()));
            request.setBody(requestParams.toString().getBytes(StandardCharsets.UTF_8));
            byte[] resultByte;
            try {
                resultByte = aepClientMapItem.getAepDeviceCommandLwmProfileClient().CreateCommandLwm2mProfile(request).getBody();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
            return resultByte;
        });
    }

    private JSONObject sendCommandAction(String deviceId, JSONObject content, BiFunction<JSONObject, AEPClientMapItem, byte[]> function) {
        JSONObject requestParams = new JSONObject();
        requestParams.put("deviceId", deviceId);
        requestParams.put("operator", "业务系统");
        requestParams.put("productId", WebMeterInfo.getString(IotModuleConfigEnum.AEP_PRODUCT_ID.getValue()));
        requestParams.put("ttl", 0);
        JsonTools.addJSON(requestParams, content);
        AEPClientMapItem aepClientMapItem = getClientMap(WebMeterInfo.getString(IotModuleConfigEnum.AEP_APP_KEY.getValue()));
        byte[] resultByte = function.apply(requestParams, aepClientMapItem);
        JSONObject response = JsonTools.convertToJson(new String(resultByte, StandardCharsets.UTF_8));
        LOGGER.info("AEP平台下发指令请求，body: {}，response: {}", requestParams, response);
        if (response.getInt("code") == 0) {
            return new JSONObject().put("code", 0).put("commandId", response.getJSONObject("result").getString("commandId"));
        }
        return response;
    }

    /**
     * AEP注册设备
     *
     * @param imei        IMEI
     * @param meterNumber 表号
     * @return 执行结果
     */
    public JSONObject addDevice(String imei, String meterNumber) throws Exception {
        JSONObject params = new JSONObject();
        params.put("deviceName", meterNumber);
        String deviceSn = imei;
        String deviceSnPrefix = WebMeterInfo.getString("deviceSnPrefix");
        if (deviceSnPrefix != null && !deviceSnPrefix.isEmpty()) {
            deviceSn = deviceSnPrefix + deviceSn;
        }
        params.put("deviceSn", deviceSn);
        params.put("imei", imei);
        params.put("productId", WebMeterInfo.getString(IotModuleConfigEnum.AEP_PRODUCT_ID.getValue()));
        params.put("operator", "业务系统");
        JSONObject other = new JSONObject();
        other.put("autoObserver", 0);
        params.put("other", other);
        CreateDeviceRequest request = new CreateDeviceRequest();
        request.setParamMasterKey(WebMeterInfo.getString(IotModuleConfigEnum.AEP_MASTER_KEY.getValue()));
        request.setBody(params.toString().getBytes(StandardCharsets.UTF_8));
        AEPClientMapItem aepClientMapItem = getClientMap(WebMeterInfo.getString(IotModuleConfigEnum.AEP_APP_KEY.getValue()));
        byte[] resultByte = aepClientMapItem.getAepDeviceManagementClient().CreateDevice(request).getBody();
        JSONObject response = JsonTools.convertToJson(new String(resultByte, StandardCharsets.UTF_8));
        LOGGER.info("AEP平台创建设备请求，body: {}，response: {}", params, response);
        int code = response.getInt("code");
        if (code == 0) {
            return new JSONObject().put("code", 0).put("msg", response.getJSONObject("result").getString("deviceId"));
        } else {
            return response;
        }
    }

    /**
     * AEP删除设备
     *
     * @param deviceId 设备号
     * @return 执行结果
     */
    public JSONObject deleteDevice(String deviceId) throws Exception {
        DeleteDeviceRequest request = new DeleteDeviceRequest();
        request.setParamMasterKey(WebMeterInfo.getString(IotModuleConfigEnum.AEP_MASTER_KEY.getValue()));
        request.setParamProductId(WebMeterInfo.getString(IotModuleConfigEnum.AEP_PRODUCT_ID.getValue()));
        request.setParamDeviceIds(deviceId);
        AEPClientMapItem aepClientMapItem = getClientMap(WebMeterInfo.getString(IotModuleConfigEnum.AEP_APP_KEY.getValue()));
        byte[] resultByte = aepClientMapItem.getAepDeviceManagementClient().DeleteDevice(request).getBody();
        JSONObject response = JsonTools.convertToJson(new String(resultByte, StandardCharsets.UTF_8));
        LOGGER.info("AEP平台删除设备请求，deviceId: {}, response: {}", deviceId, response);
        return response;
    }

    private static class AEPClientMapItem {
        private final AepDeviceCommandClient aepDeviceCommandClient;
        private final AepDeviceCommandLwmProfileClient aepDeviceCommandLwmProfileClient;
        private final AepDeviceManagementClient aepDeviceManagementClient;

        public AEPClientMapItem(String appKey, String appSecret) {
            aepDeviceCommandClient = AepDeviceCommandClient.newClient().appKey(appKey).appSecret(appSecret).build();
            aepDeviceCommandLwmProfileClient = AepDeviceCommandLwmProfileClient.newClient().appKey(appKey).appSecret(appSecret).build();
            aepDeviceManagementClient = AepDeviceManagementClient.newClient().appKey(appKey).appSecret(appSecret).build();
        }

        public AepDeviceCommandClient getAepDeviceCommandClient() {
            return aepDeviceCommandClient;
        }

        public AepDeviceCommandLwmProfileClient getAepDeviceCommandLwmProfileClient() {
            return aepDeviceCommandLwmProfileClient;
        }

        public AepDeviceManagementClient getAepDeviceManagementClient() {
            return aepDeviceManagementClient;
        }
    }

}
