package com.aote.webmeter.tools;

import org.apache.commons.lang3.StringUtils;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

public class EncodingUtil {

    /**
     * 大小端序反转
     */
    public static String reverseHexStringByEndian(String hexString, Endian endian) {
        if (hexString == null || hexString.length() % 2 != 0) {
            throw new IllegalArgumentException("输入的字符串必须是有效的十六进制字符串");
        }

        int length = hexString.length();
        char[] chars = hexString.toCharArray();
        char[] reversedChars = new char[length];

        if (endian == Endian.BIG_ENDIAN) {
            // 按照大端序规则重新排列
            for (int i = 0; i < length; i += 2) {
                reversedChars[i] = chars[i];
                reversedChars[i + 1] = chars[i + 1];
            }
        } else if (endian == Endian.LITTLE_ENDIAN) {
            // 按照小端序规则重新排列
            for (int i = 0; i < length; i += 2) {
                reversedChars[i] = chars[length - i - 2];
                reversedChars[i + 1] = chars[length - i - 1];
            }
        }

        return new String(reversedChars);
    }

    public static String calculateCRC16(String hexInput, Boolean isLE) {
        return byteArrayToHexString(calculateCRC16Modbus(hexStringToByteArray(hexInput),isLE));

    }

    public static byte[] calculateCRC16Modbus(byte[] inputBytes, Boolean isLE) {
        int crc = 0xFFFF; // 初始CRC-16值
        int polynomial = 0xA001; // CRC-16/MODBUS多项式

        for (byte b : inputBytes) {
            crc ^= (b & 0xFF);
            for (int i = 0; i < 8; i++) {
                boolean bit = ((crc & 0x0001) == 0x0001);
                crc >>= 1;
                if (bit) {
                    crc ^= polynomial;
                }
            }
        }

        byte[] crcBytes = new byte[2];
        if(isLE){
            // 构建包含CRC-16/MODBUS结果的字节数组（低字节在前，高字节在后）
            crcBytes[0] = (byte) (crc & 0xFF); // 低字节
            crcBytes[1] = (byte) ((crc >> 8) & 0xFF); // 高字节
        }else{
            crcBytes[1] = (byte) (crc & 0xFF); // 低字节
            crcBytes[0] = (byte) ((crc >> 8) & 0xFF); // 高字节
        }


        return crcBytes;
    }


    public static List<Integer> findCharPositions(String str, char ch) {
        List<Integer> positions = new ArrayList<>();
        for (int i = 0; i < str.length(); i++) {
            if (str.charAt(i) == ch) {
                positions.add(i);
            }
        }
        return positions;
    }

    public enum Endian {
        BIG_ENDIAN,
        LITTLE_ENDIAN
    }

    /**
     * 左侧补0,补齐位数
     */
    public static String padLeft(String input, int length, char paddingChar,boolean checkLength) throws Exception {
        //超长处理
        if(input.length()>length&&checkLength){
            throw new Exception("参数长度超长:"+input);
        }
        return StringUtils.repeat(paddingChar,length - input.length()) + input;
    }

    /**
     * 左侧补0,补齐位数
     */
    public static String padRight(String input, int length, char paddingChar,boolean checkLength) throws Exception {
        //超长处理
        if(input.length()>length&&checkLength){
            throw new Exception("参数长度超长:"+input);
        }
        return input+StringUtils.repeat(paddingChar,length - input.length());
    }


    /**
     * 断字符串是否是步长的整数倍,如果不是,按照左右,在左侧或右侧补足填充字符
     * @param input 字符串
     * @param step 步长
     * @param paddingChar 填充字符
     * @param padLeft true为左侧补,false为右侧
     * @return 新字符串
     */
    public static String padToMultipleWithDirection(String input, int step, char paddingChar, boolean padLeft) {
        int length = input.length();
        int remainder = length % step;

        if (remainder == 0) {
            return input; // 字符串长度已经是步长的整数倍，无需补充
        }

        int paddingLength = step - remainder;
        StringBuilder paddedString = new StringBuilder(input);

        for (int i = 0; i < paddingLength; i++) {
            if (padLeft) {
                paddedString.insert(0, paddingChar); // 在左侧补充填充字符
            } else {
                paddedString.append(paddingChar); // 在右侧补充填充字符
            }
        }

        return paddedString.toString();
    }

    public static byte[] hexStringToByteArray(String hexString) {
        // 去除字符串中的空格和非16进制字符
        hexString = hexString.replaceAll("[^0-9A-Fa-f]", "");

        // 如果字符串长度为奇数，前面补0
        if (hexString.length() % 2 != 0) {
            hexString = "0" + hexString;
        }

        int len = hexString.length() / 2;
        byte[] byteArray = new byte[len];

        for (int i = 0; i < len; i++) {
            int startIndex = i * 2;
            int endIndex = startIndex + 2;
            String hex = hexString.substring(startIndex, endIndex);
            byteArray[i] = (byte) Integer.parseInt(hex, 16);
        }

        return byteArray;
    }

    public static String byteArrayToHexString(byte[] byteArray) {
        StringBuilder hexString = new StringBuilder(2 * byteArray.length);

        for (byte b : byteArray) {
            // 将每个字节转换为两位的16进制字符串，并追加到结果中
            hexString.append(String.format("%02X", b));
        }

        return hexString.toString();
    }

    public static String binaryToHexWithCorrectByteOrder(String binaryString) {
        // 去除字符串中的空格和非二进制字符
        binaryString = binaryString.replaceAll("[^01]", "");
        binaryString = padToMultipleWithDirection(binaryString,8,'0',false);

        StringBuilder hexString = new StringBuilder(binaryString.length() / 8);

        // 每8位二进制数（一个字节）反转后再转换为一个十六进制数字
        for (int i = 0; i < binaryString.length(); i += 8) {
            String binaryByte = binaryString.substring(i, i + 8);
            String reversedBinaryByte = reverseBinaryString(binaryByte);
            int decimalValue = Integer.parseInt(reversedBinaryByte, 2);
            String hexByte = String.format("%02X", decimalValue);
            hexString.append(hexByte);
        }

        return hexString.toString();
    }

    // 反转二进制字符串
    public static String reverseBinaryString(String binaryString) {
        StringBuilder reversedString = new StringBuilder(binaryString.length());
        for (int i = binaryString.length() - 1; i >= 0; i--) {
            reversedString.append(binaryString.charAt(i));
        }
        return reversedString.toString();
    }


    public static String hexToBinaryWithReversedBits(String hexString) throws Exception {
        // 将十六进制字符串转换为字节数组
        byte[] byteArray = hexStringToByteArray(hexString);

        StringBuilder binaryString = new StringBuilder(byteArray.length * 8);

        // 遍历每个字节，反转位顺序并转换为二进制字符串
        for (byte b : byteArray) {
            String binaryByte = Integer.toBinaryString(b & 0xFF);

            // 在二进制字符串前面补0，确保每个字节有8位
            binaryByte=padLeft(binaryByte,8,'0',false);

            // 反转字节的位顺序
            String reversedBinaryByte = reverseBinaryString(binaryByte);
            binaryString.append(reversedBinaryByte);
        }

        return binaryString.toString();
    }


    // 将十六进制字符串转换为二进制字符串
    public static String hexToBinary(String hexString) {
        StringBuilder binaryStringBuilder = new StringBuilder();
        for (int i = 0; i < hexString.length(); i++) {
            char hexChar = hexString.charAt(i);
            StringBuilder binary = new StringBuilder(Integer.toBinaryString(Integer.parseInt(String.valueOf(hexChar), 16)));
            // 每个十六进制字符转换为4位二进制字符串
            // 需要补齐前导零以保证4位
            while (binary.length() < 4) {
                binary.insert(0, "0");
            }
            binaryStringBuilder.append(binary);
        }
        return binaryStringBuilder.toString();
    }

    public static String binaryToHex(String binaryString) {
        // 计算需要填充的0的数量
        int padding_ = binaryString.length() % 8;
        int padding = 8 - (padding_== 0?8: padding_);

        // 如果长度不是8的倍数，左侧补0
        StringBuilder paddedBinary = new StringBuilder();
        for (int i = 0; i < padding; i++) {
            paddedBinary.append('0');
        }
        paddedBinary.append(binaryString);

        StringBuilder hexString = new StringBuilder();
        // 将二进制字符串按每4位一组转换为十六进制字符
        for (int i = 0; i < paddedBinary.length(); i += 4) {
            String binaryChunk = paddedBinary.substring(i, i + 4);
            int decimal = Integer.parseInt(binaryChunk, 2);
            hexString.append(Integer.toHexString(decimal));
        }

        return hexString.toString();
    }
    public static void mergeJSONObjects(JSONObject json1, JSONObject json2) {
        // 添加第二个 JSONObject 的键值对
        Iterator<String> keys2 = json2.keys();
        while (keys2.hasNext()) {
            String key = keys2.next();
            json1.put(key, json2.get(key)); // 如果键重复，json2 的值会覆盖 json1 的值
        }
    }
}
