package com.aote.webmeter.socket.core.client;

import com.af.plugins.ConvertTools;
import com.aote.webmeter.socket.config.SocketConfigItem;
import com.aote.webmeter.socket.core.channel.ChannelData;
import io.netty.channel.Channel;
import io.netty.util.HashedWheelTimer;
import org.apache.commons.codec.DecoderException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

/**
 * 客户端管理器
 */
public class ClientManager {

    private static final Logger LOGGER = LoggerFactory.getLogger(ClientManager.class);

    private static final Map<String, ChannelData> clientChannels = new ConcurrentHashMap<>();
    private static final ThreadLocal<String> channelIdLocal = new ThreadLocal<>();

    // netty时间轮
    private static volatile HashedWheelTimer TIMER;

    private static HashedWheelTimer getTimer() {
        if (TIMER == null) {
            synchronized (ClientManager.class) {
                if (TIMER == null) {
                    TIMER = new HashedWheelTimer();
                }
            }
        }
        return TIMER;
    }


    public static void setChannelId(String channelId) {
        channelIdLocal.set(channelId);
    }

    public static void add(String channelId, ChannelData channelData) {
        clientChannels.put(channelId, channelData);
    }

    public static void remove() {
        String channelId = channelIdLocal.get();
        if (channelId != null) {
            clientChannels.remove(channelId);
            channelIdLocal.remove();
        }
    }

    public static void delayedSend(Object sleepTime, String message) {
        getTimer().newTimeout(timeout -> {
            try {
                send(message);
            } catch (Exception e) {
                LOGGER.error("延时发送消息时出现异常：", e);
            }
        }, Long.parseLong(sleepTime.toString()), TimeUnit.MILLISECONDS);
    }

    public static void send(String message) throws DecoderException {
        if (message == null || message.equals("ok") || message.equals("6f6b")) {
            return;
        }
        String channelId = channelIdLocal.get();
        ChannelData channelData = clientChannels.get(channelId);
        Channel channel = channelData.getChannel();
        LOGGER.debug("返回数据连接,channel：{}", channel);
        byte[] sendMsgBytes;
        if (channel != null) {
            String address = channel.remoteAddress().toString();
            LOGGER.debug("channelId:{}, address:{}, 向客户端发送消息", channelId, address);
            switch (channelData.getConfigItem().getEncodeType()) {
                case BASE64: {
                    sendMsgBytes = ConvertTools.base64Decode(message.getBytes());
                    break;
                }
                case HEX: {
                    sendMsgBytes = ConvertTools.hexStrToByte(message);
                    break;
                }
                case HEX_BASE64: {
                    sendMsgBytes = ConvertTools.base64Decode(ConvertTools.hexStrToByte(message));
                    break;
                }
                default: {
                    sendMsgBytes = message.getBytes();
                    break;
                }
            }
            channel.writeAndFlush(channel.alloc().buffer().writeBytes(sendMsgBytes));
        } else {
            throw new RuntimeException("channel为空！");
        }
    }

    public static SocketConfigItem getConfig() {
        return getChannelData().getConfigItem();
    }

    public static ChannelData getChannelData() {
        String channelId = channelIdLocal.get();
        return clientChannels.get(channelId);
    }



    /**
     * 设置全局参数
     *
     * @param key   参数key
     * @param value 值
     */
    public static void setGlobalParams(String key, Object value) {
        ChannelData channelData = ClientManager.getChannelData();
        JSONObject globalParamsItem = channelData.getGlobalParams();
        globalParamsItem.put(key, value);
    }

    /**
     * 获取全局参数
     *
     * @param key 参数key
     */
    public static Object getGlobalParams(String key) {
        ChannelData channelData = ClientManager.getChannelData();
        JSONObject globalParamsItem = channelData.getGlobalParams();
        return globalParamsItem.opt(key);
    }
}
