package com.aote.webmeter.socket.core.channel;

import com.af.plugins.ConvertTools;
import com.aote.logic.LogicServer;
import com.aote.webmeter.socket.SocketServerManager;
import com.aote.webmeter.socket.config.SocketConfigItem;
import com.aote.webmeter.socket.core.client.ClientManager;
import io.netty.buffer.ByteBuf;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetSocketAddress;
import java.util.Arrays;

public abstract class AbstractChannelHandler<T> extends SimpleChannelInboundHandler<T> {

    private final Logger LOGGER = LoggerFactory.getLogger(AbstractChannelHandler.class);

    @Override
    protected final void channelRead0(ChannelHandlerContext ctx, T msg){
        Channel channel = ctx.channel();
        LOGGER.debug("channelId:{}, 收到消息", channel.id().asLongText());
        String channelId = channel.id().asLongText();
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("channelId:{}, 收到消息", channelId);
        }
        ClientManager.setChannelId(channelId);
        if (ClientManager.getChannelData() == null) {
            initChannelData(ctx);
        }
        // 执行读方法
        read(ctx, msg);
    }

    protected abstract void read(ChannelHandlerContext ctx, T msg);

    private void initChannelData(ChannelHandlerContext ctx) {
        // 存入管道信息
        Channel channel = ctx.channel();
        String port = String.valueOf(((InetSocketAddress) channel.localAddress()).getPort());
        SocketConfigItem configItem = SocketServerManager.getSocketConfigItemByPort(port);
        ClientManager.add(channel.id().asLongText(), new ChannelData(configItem, channel, new JSONObject()));
    }

    protected String runBusiness(ByteBuf content, String remoteAddress) {
        SocketConfigItem configItem = ClientManager.getConfig();
        byte[] data = new byte[content.readableBytes()];
        content.readBytes(data);
        LOGGER.debug("数据原始体：{}", Arrays.toString(data));
        String realData;
        try {
            switch (configItem.getDecodeType()) {
                case HEX: {
                    realData = ConvertTools.byteToHexStr(data);
                    break;
                }
                default: {
                    realData = ConvertTools.bytesToStr(data);
                    break;
                }
            }
        } catch (Exception e) {
            LOGGER.error("数据解析失败：", e);
            return "6f6b";
        }

        LOGGER.debug("服务端收到信息：{}", realData);

        LogicServer logicServer = SocketServerManager.applicationContext.getBean(LogicServer.class);

        try {
            JSONObject params = new JSONObject();
            params.put("value", realData);
            params.put("address", remoteAddress);
            // 调用实际执行具体业务的logic
            return logicServer.run(configItem.getLogicName(), params).toString();
        } catch (Exception e) {
            LOGGER.error("业务处理失败", e);
            throw e;
        }
    }

    @Override
    public void channelActive(ChannelHandlerContext ctx) throws Exception {
        String channelId = ctx.channel().id().asLongText();
        String address = ctx.channel().remoteAddress().toString();
        LOGGER.debug("channelId:{}, address:{}, 成功建立连接", channelId, address);
        super.channelActive(ctx);
    }

    @Override
    public void channelInactive(ChannelHandlerContext ctx) throws Exception {
        try {
            ctx.close();
            ClientManager.remove();
        } catch (Exception e) {
            LOGGER.error("关闭通道时发生异常", e);
        }
        super.channelInactive(ctx);
    }
}
