package com.aote.webmeter.service;

import com.af.plugins.DateTools;
import com.af.plugins.JsonTools;
import com.af.plugins.RestAsyncTools;
import com.af.plugins.RestTools;
import com.aote.entity.EntityServer;
import com.aote.redis.RedisUtil;
import com.aote.sql.SqlServer;
import com.aote.webmeter.enums.DeviceAlarmStrategyCompareTypeEnum;
import com.aote.webmeter.enums.DeviceAlarmStrategyLevelEnum;
import com.aote.webmeter.mapper.AlarmStrategyMapper;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashMap;
import java.util.Map;

/**
 * 物联网告警策略服务
 *
 * @author Mrriver
 * @date 2023/2/14
 */
@Component
@Transactional
public class AlarmStrategyService {

    /**
     * 静态报警映射配置路径
     */
    private static final String CONFIG_URL = "webmeter/mainExceptions.json";
    /**
     * 默认消息类型
     */
    private static final String DEFAULT_MSG_TYPE = "物联网设备报警通知短信";
    /**
     * 异常映射列表
     */
    private static final JSONArray EXCEPTION_LIST;
    /**
     * 默认告警内容模板
     */
    private static final String DEFAULT_EXCEPTION_MSG_TEMPLATE = "设备上报的{监控项名称}达到{告警值}，{告警规则}阈值：{告警阈值}，触发{告警级别}，请及时关注！";
    private static final Logger LOGGER = Logger.getLogger(AlarmStrategyService.class);

    static {
        //加载异常映射
        if (AlarmStrategyService.class.getClassLoader().getResource(CONFIG_URL) != null) {
            EXCEPTION_LIST = JsonTools.readJsonArrayFile(CONFIG_URL);
            LOGGER.debug("加载抄表异常记录映射配置文件完成");
        } else {
            throw new RuntimeException("缺少抄表异常记录映射配置文件：mainExceptions.json");
        }
    }

    @Autowired
    private SqlServer sqlServer;
    @Autowired
    private EntityServer entityServer;

    /**
     * 存储异常列表
     *
     * @param deviceType 设备类型
     * @param tableName  表名 可为空
     * @param meterAlias 表具别名
     * @param datas      抄表数据
     */
    public boolean saveExceptionArray(String deviceType, String tableName, String meterAlias, JSONObject datas) throws Exception {
        JSONArray strategyArray = new JSONArray(EXCEPTION_LIST.toString());
        RedisUtil redisUtil = RedisUtil.getInstance();
        Object strategyStr = redisUtil.get(AlarmStrategyMapper.ALARM_STRATEGY_KEY);
        if (strategyStr != null) {
            JSONObject strategyObj = new JSONObject(strategyStr.toString());
            if (strategyObj.has(meterAlias)) {
                strategyArray.putAll(strategyObj.getJSONArray(meterAlias));
            }
        }
        for (int i = 0; i < strategyArray.length(); i++) {
            JSONObject strategy = strategyArray.getJSONObject(i);
            if (!strategy.has("f_alarm_item")) {
                continue;
            }
            String key = strategy.getString("f_alarm_item");
            Object value = datas.opt(key);
            if (value == null) {
                continue;
            }
            //异常阈值
            String errorValue = strategy.get("f_alarm_value").toString();
            String compareType = strategy.optString("f_alarm_type",null);
            //异常结果值
            boolean isException;
            if (compareType != null && compareType.equals("array")) {
                JSONArray alarmValues = strategy.getJSONArray("f_alarm_values");
                isException = false;
                boolean isArrayException;
                for (int j = 0; j < alarmValues.length(); j++) {
                    errorValue = datas.get(alarmValues.get(j).toString()).toString();
                    switch (strategy.getString("f_compare_type")) {
                        case "=":
                            isArrayException = value.equals(errorValue);
                            break;
                        case "!=":
                            isArrayException = !value.equals(errorValue);
                            break;
                        case ">":
                            isArrayException = Double.doubleToLongBits(Double.parseDouble(value.toString())) >
                                    Double.doubleToLongBits(Double.parseDouble(errorValue));
                            break;
                        case ">=":
                            isArrayException = Double.doubleToLongBits(Double.parseDouble(value.toString())) >=
                                    Double.doubleToLongBits(Double.parseDouble(errorValue));
                            break;
                        case "<":
                            isArrayException = Double.doubleToLongBits(Double.parseDouble(value.toString())) <
                                    Double.doubleToLongBits(Double.parseDouble(errorValue));
                            break;
                        case "<=":
                            isArrayException = Double.doubleToLongBits(Double.parseDouble(value.toString())) <=
                                    Double.doubleToLongBits(Double.parseDouble(errorValue));
                            break;
                        default:
                            isArrayException = false;
                            break;
                    }
                    if (isArrayException){
                        isException = isArrayException;
                        break;
                    }
                }
            } else {
                switch (strategy.getString("f_compare_type")) {
                    case "orWith":
                        long errValue = Double.doubleToLongBits(Double.parseDouble(value.toString()));
                        long maxErrorValue = Double.doubleToLongBits(Double.parseDouble(strategy.get("f_max_alarm_value").toString()));
                        long minErrorValue = Double.doubleToLongBits(Double.parseDouble(strategy.get("f_min_alarm_value").toString()));
                        isException = ( errValue > minErrorValue && errValue < maxErrorValue);
                        break;
                    case "=":
                        isException = value.equals(errorValue);
                        break;
                    case "!=":
                        isException = !value.equals(errorValue);
                        break;
                    case ">":
                        isException = Double.doubleToLongBits(Double.parseDouble(value.toString())) >
                                Double.doubleToLongBits(Double.parseDouble(errorValue));
                        break;
                    case ">=":
                        isException = Double.doubleToLongBits(Double.parseDouble(value.toString())) >=
                                Double.doubleToLongBits(Double.parseDouble(errorValue));
                        break;
                    case "<":
                        isException = Double.doubleToLongBits(Double.parseDouble(value.toString())) <
                                Double.doubleToLongBits(Double.parseDouble(errorValue));
                        break;
                    case "<=":
                        isException = Double.doubleToLongBits(Double.parseDouble(value.toString())) <=
                                Double.doubleToLongBits(Double.parseDouble(errorValue));
                        break;
                    default:
                        isException = false;
                        break;
                }
            }
            String exceptionTableName;
            if (deviceType.equals("物联网表")) {
                exceptionTableName = "t_exception";
            } else {
                exceptionTableName = "t_iot_device_exception";
            }
            int alarmState;
            int times;
            JSONArray historyExceptions = getHistoryExceptions(deviceType, tableName, strategy, datas);
            if (historyExceptions.length() > 0) {
                // 获取历史触发次数
                JSONObject historyException = historyExceptions.getJSONObject(0);
                if (!isException) {
                    // 更新异常记录
                    updateExceptionReturnToNormal(datas.get("id").toString(), historyException.get("id").toString(), exceptionTableName);
                    continue;
                }
                times = historyException.optInt("f_times", 1);
            } else {
                if (!isException) {
                    continue;
                }
                times = 0;
            }
            times += 1;
            int strategyTimes = strategy.optInt("f_alarm_times", 1);
            if (times >= strategyTimes) {
                alarmState = 1;
            } else {
                alarmState = 0;
            }
            // 存储异常记录
            JSONObject exceptionData = this.saveExceptionData(historyExceptions, times, deviceType, tableName, meterAlias, datas, strategy, value.toString(), alarmState);
            // 达到报警条件后，触发推送
            if (alarmState == 1 && exceptionData.opt("type").equals("add")) {
                int pushType = strategy.optInt("f_push_type", -1);
                if (pushType == 0 && strategy.has("f_push_params") && strategy.has("f_alarm_persons")) {
                    try {
                        JSONObject params = new JSONObject();
                        JSONObject extraParams = new JSONObject(strategy.getString("f_push_params"));
                        params.put("f_templateid", extraParams.get("templateId"));
                        params.put("f_businessid", exceptionData.get("id"));
                        params.put("f_send_type", DEFAULT_MSG_TYPE);
                        params.put("f_orgid", exceptionData.opt("f_orgid"));
                        params.put("f_orgname", exceptionData.opt("f_orgname"));
                        params.put("f_userinfo_id", exceptionData.opt("f_userinfo_id"));
                        params.put("f_iot_device_no", exceptionData.opt("f_iot_device_no"));
                        params.put("users", new JSONArray(strategy.getString("f_alarm_persons")));
                        params.put("f_content", exceptionData.get("f_error_msg"));
                        RestTools.post(strategy.getString("f_push_url"), params);
                    } catch (Exception e){
                        LOGGER.error("推送告警失败，原因:", e);
                    }
                }
                return true;
            }
        }
        return hasHistoryInAlarmExceptions(deviceType, datas);
    }

    /**
     * 更新异常记录为正常
     * @param alarmDataId 异常记录所属抄表ID
     * @param id 异常记录ID
     * @param exceptionTableName 异常表名
     */
    public void updateExceptionReturnToNormal(String alarmDataId, String id, String exceptionTableName) throws Exception {
        // 更新异常记录
        JSONObject updateEntity = new JSONObject();
        updateEntity.put("f_alarm_state", "2");
        updateEntity.put("f_eliminate_alarm_date", DateTools.getNow2());
        updateEntity.put("f_eliminate_alarm_data_id", alarmDataId);
        updateEntity.put("id", id);
        entityServer.partialSave(exceptionTableName, updateEntity);
    }
    public void updateExceptionReturnToNormal(String id, String exceptionTableName) throws Exception {
        updateExceptionReturnToNormal(null, id, exceptionTableName);
    }

    private JSONObject getExceptionDataByUploadData(JSONObject param, JSONObject datas) {
        String f_error_level = datas.get("f_error_level").toString();
        if (!(f_error_level.equals("1") || f_error_level.equals("2") || f_error_level.equals("3"))) {
            f_error_level = "1";
        }
        param.put("f_alarm_times", "1");
        param.put("f_error_level", f_error_level);
        param.put("f_error_type", datas.get("f_error_type"));
        String msg = datas.getString("f_error_msg");
        param.put("f_error_msg", msg);
        return param;
    }

    private JSONObject getExceptionDataByStrategy(String commonMapName, String tableName, String errorValue, JSONObject datas, JSONObject strategy) {
        JSONObject param = new JSONObject();
        param.put("f_strategy_id", strategy.opt("id"));
        getAlarmItem(commonMapName, tableName, strategy);
        String alarmItem = getAlarmItem(commonMapName, tableName, strategy);
        param.put("f_error_type", alarmItem);
        String compareTypeStr = DeviceAlarmStrategyCompareTypeEnum.toType(strategy.get("f_compare_type").toString()).getLabel();
        param.put("f_compare_type", compareTypeStr);
        String alarmValue = strategy.opt("f_alarm_value").toString();
        param.put("f_alarm_value", alarmValue);
        param.put("f_alarm_times", strategy.opt("f_alarm_times"));
        String strategyLevel = DeviceAlarmStrategyLevelEnum.toType(strategy.opt("f_alarm_level").toString()).getLabel();
        param.put("f_error_level", strategyLevel);
        Map<String, String> paramsMap = new HashMap<>(5);
        paramsMap.put("监控项名称", alarmItem);
        paramsMap.put("告警值", errorValue);
        paramsMap.put("告警规则", compareTypeStr);
        paramsMap.put("告警阈值", alarmValue);
        paramsMap.put("告警级别", strategyLevel);
        JSONArray variableArray = strategy.optJSONArray("f_alarm_variable");
        String msg = format(strategy.optString("f_alarm_msg", DEFAULT_EXCEPTION_MSG_TEMPLATE), paramsMap, datas, variableArray);
        param.put("f_error_msg", msg);
        return param;
    }

    private String getAlarmItem(String commonMapName, String tableName, JSONObject strategy) {
        String alarmItemValue = strategy.get("f_alarm_item").toString();
        String alarmItem = null;
        if (tableName != null) {
            alarmItem = DeviceAlarmStrategyHelper.getAlarmItem(tableName, alarmItemValue);
        }
        if (alarmItem == null) {
            alarmItem = DeviceAlarmStrategyHelper.getAlarmItem(commonMapName, alarmItemValue);
        }
        return alarmItem;
    }

    private boolean hasHistoryInAlarmExceptions(String deviceType, JSONObject datas) throws Exception {
        String exceptionTableName;
        String columnName;
        if (deviceType.equals("物联网表")) {
            exceptionTableName = "t_exception";
            columnName = "f_meternumber";
        } else {
            exceptionTableName = "t_iot_device_exception";
            columnName = "f_iot_device_id";
        }
        // 查询历史正在发生的异常记录
        return sqlServer.query("getExceptionInAlarmHistory",
                new JSONObject()
                        .put("exceptionTableName", exceptionTableName)
                        .put("columnName", columnName)
                        .put("columnNameValue", datas.get(columnName))
        ).length() > 0;
    }

    private JSONArray getHistoryExceptions(String deviceType, String tableName, JSONObject strategy, JSONObject datas) throws Exception {
        String exceptionTableName;
        String columnName;
        String commonMapName;
        if (deviceType.equals("物联网表")) {
            exceptionTableName = "t_exception";
            columnName = "f_meternumber";
            commonMapName = "meterCommonMap";
        } else {
            exceptionTableName = "t_iot_device_exception";
            columnName = "f_iot_device_id";
            commonMapName = "deviceCommonMap";
        }
        // 查询历史异常记录
        String alarmItem;
        if (strategy != null) {
            alarmItem = getAlarmItem(commonMapName, tableName, strategy);
        } else {
            alarmItem = datas.getString("f_error_type");
        }
        return sqlServer.query("getExceptionHistory",
                new JSONObject()
                        .put("exceptionTableName", exceptionTableName)
                        .put("columnName", columnName)
                        .put("columnNameValue", datas.get(columnName))
                        .put("errorTypeValue", alarmItem)
        );
    }

    public JSONObject saveExceptionData(String deviceType, String tableName, String meterAlias, JSONObject datas, JSONObject strategy,
                                        String alarmItemValue, Integer alarmState) throws Exception {
        return saveExceptionData(null, null, deviceType, tableName, meterAlias,
                datas, strategy, alarmItemValue, alarmState);
    }

    public JSONObject saveExceptionData(JSONArray historyExceptions, Integer times, String deviceType, String tableName, String meterAlias, JSONObject datas, JSONObject strategy,
                                        String alarmItemValue, Integer alarmState) throws Exception {
        if (historyExceptions == null) {
            historyExceptions = getHistoryExceptions(deviceType, tableName, strategy, datas);
        }
        if (times == null) {
            times = 1;
        }
        String commonMapName;
        String exceptionTableName;
        if (deviceType.equals("物联网表")) {
            commonMapName = "meterCommonMap";
            exceptionTableName = "t_exception";
        } else {
            commonMapName = "deviceCommonMap";
            exceptionTableName = "t_iot_device_exception";
        }
        if (historyExceptions.length() > 0) {
            JSONObject exceptionData = historyExceptions.getJSONObject(0);
            // 更新异常记录
            JSONObject updateEntity;
            if (strategy != null) {
                updateEntity = getExceptionDataByStrategy(commonMapName, tableName, alarmItemValue, datas, strategy);
            } else {
                updateEntity = getExceptionDataByUploadData(exceptionData, datas);
            }
            updateEntity.put("f_times", times);
            updateEntity.put("f_alarm_state", alarmState);
            updateEntity.put("f_last_alarm_date", DateTools.getNow2());
            updateEntity.put("id", exceptionData.get("id"));
            entityServer.partialSave(exceptionTableName, updateEntity);
            exceptionData.put("type", "update");
            return exceptionData;
        } else {
            JSONObject param = new JSONObject();
            if (deviceType.equals("物联网表")) {
                param.put("f_user_id", datas.opt("f_user_id"));
                param.put("f_userfiles_id", datas.opt("f_userfiles_id"));
                param.put("f_meternumber", datas.opt("f_meternumber"));
                param.put("f_device_type", "物联网燃气表");
            } else {
                param.put("f_iot_device_id", datas.opt("f_iot_device_id"));
                param.put("f_iot_device_no", datas.optString("f_iot_device_no", "无设备号"));
            }
            param.put("f_userinfo_id", datas.opt("f_userinfo_id"));
            param.put("f_alias", meterAlias);
            param.put("data_id", datas.opt("id"));
            param.put("f_read_table_name", tableName);
            param.put("f_alarm_real_value", alarmItemValue);
            if (strategy != null) {
                JsonTools.addJSON(param, getExceptionDataByStrategy(commonMapName, tableName, alarmItemValue, datas, strategy));
            } else {
                getExceptionDataByUploadData(param, datas);
            }
            param.put("f_alarm_state", alarmState);
            param.put("f_times", times);
            param.put("f_is_read", 0);
            // 分公司信息
            param.put("f_filiale", datas.opt("f_filiale"));
            param.put("f_filialeids", datas.opt("f_filialeids"));
            param.put("f_filialeid", datas.opt("f_filialeid"));
            param.put("f_outlets", datas.opt("f_outlets"));
            param.put("f_orgstr", datas.opt("f_orgstr"));
            param.put("f_orgid", datas.opt("f_orgid"));
            param.put("f_depid", datas.opt("f_depid"));
            param.put("f_orgname", datas.opt("f_orgname"));
            param.put("f_depname", datas.opt("f_depname"));
            param.put("f_error_notes", datas.opt("f_error_notes"));
            param.put("id", new JSONObject(entityServer.partialSave(exceptionTableName, param)).getInt("id"));
            param.put("type", "add");
            return param;
        }
    }


    private String format(String regex, Map<String, String> paramsMap, JSONObject params, JSONArray paramNameArray) {
        String regexStr = regex;
        for (Map.Entry<String, String> entry : paramsMap.entrySet()) {
            regexStr = regexStr.replaceAll("\\{" + entry.getKey() + "}", entry.getValue());
        }
        if (paramNameArray != null) {
            for (Object key : paramNameArray) {
                regexStr = regexStr.replaceAll("\\{" + key + "}", params.optString(key.toString(), "null"));
            }
        }
        return regexStr;
    }
}
