package com.aote.webmeter.socket.core.client;

import com.af.plugins.ConvertTools;
import com.aote.webmeter.socket.config.SocketConfigItem;
import com.aote.webmeter.socket.core.channel.ChannelData;
import io.netty.channel.Channel;
import io.netty.util.HashedWheelTimer;
import org.apache.commons.codec.DecoderException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.TimeUnit;

/**
 * 客户端管理器
 */
public class ClientManager {

    private static final Logger LOGGER = LoggerFactory.getLogger(ClientManager.class);

    private static final ThreadLocal<ChannelData> clientChannels = new ThreadLocal<>();

    // netty时间轮
    private static volatile HashedWheelTimer TIMER;

    private static HashedWheelTimer getTimer(){
        if(TIMER == null){
            synchronized (ClientManager.class){
                if(TIMER == null){
                    TIMER = new HashedWheelTimer();
                }
            }
        }
        return TIMER;
    }


    public static void add(ChannelData channelData) {
        clientChannels.set(channelData);
    }

    public static void remove() {
        clientChannels.remove();
    }

    public static void delayedSend(ChannelData channelData, Object sleepTime, String message){
        getTimer().newTimeout(timeout -> {
            try {
                send(channelData, message);
            } catch (Exception e) {
                LOGGER.error("延时发送消息时出现异常：", e);
            }
        }, Long.parseLong(sleepTime.toString()), TimeUnit.MILLISECONDS);
    }

    public static void delayedSend(Object sleepTime, String message) {
        delayedSend(clientChannels.get(), sleepTime, message);
    }

    public static ChannelData getChannelData(){
        return clientChannels.get();
    }

    public static void send(ChannelData channelData, String message) throws DecoderException {
        if(message == null || message.equals("ok") || message.equals("6f6b")){
            return;
        }
        Channel channel = channelData.getChannel();
        byte[] sendMsgBytes;
        if (channel != null) {
            switch (channelData.getConfigItem().getEncodeType()) {
                default: {
                    sendMsgBytes = message.getBytes();
                    break;
                }
                case BASE64: {
                    sendMsgBytes = ConvertTools.base64Decode(message.getBytes());
                    break;
                }
                case HEX: {
                    sendMsgBytes = ConvertTools.hexStrToByte(message);
                    break;
                }
                case HEX_BASE64: {
                    sendMsgBytes = ConvertTools.base64Decode(ConvertTools.hexStrToByte(message));
                    break;
                }
            };
            channel.writeAndFlush(channel.alloc().buffer().writeBytes(sendMsgBytes));
        }
    }

    public static void send(String message) throws DecoderException {
        send(clientChannels.get(), message);
    }

    public static SocketConfigItem getConfig() {
        return clientChannels.get().getConfigItem();
    }

    /**
     * 设置全局参数
     * @param key 参数key
     * @param value 值
     */
    public static void setGlobalParams(String key, Object value){
        ChannelData channelData = ClientManager.getChannelData();
        JSONObject globalParamsItem = channelData.getGlobalParams();
        if(globalParamsItem == null){
            globalParamsItem = new JSONObject();
            channelData.setGlobalParams(globalParamsItem);
        }
        globalParamsItem.put(key, value);
    }

    /**
     * 获取全局参数
     * @param key 参数key
     */
    public static Object getGlobalParams(String key){
        JSONObject globalParamsItem = ClientManager.getChannelData().getGlobalParams();
        if(globalParamsItem == null){
            return null;
        }
        return globalParamsItem.opt(key);
    }
}
