package com.aote.webmeter.tools.iot;

import com.af.plugins.HttpConnectionPoolUtil;
import com.aote.webmeter.enums.IOTBusinessTypeEnum;
import com.aote.webmeter.tools.WebMeterInfo;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.log4j.Logger;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;

/**
 * 电信IOT平台通用工具
 * @author Mr.river
 */
public class IotCommonTools {

    private static final Logger LOGGER = Logger.getLogger(IotCommonTools.class);

    /**
     * 是否AEP模式
     */
    private static final ThreadLocal<Integer> IS_AEP;

    static {
        IS_AEP = new ThreadLocal<>();
    }

    /**
     * 错误标识
     */
    public static final String ERROR_PARAMS_KEY = "error_code";

    public static JSONObject post(IOTBusinessTypeEnum typeEnum, HttpEntityEnclosingRequestBase base, JSONObject contentObj) throws IOException {
        return post(typeEnum,base,contentObj,null);
    }
    public static JSONObject post(IOTBusinessTypeEnum typeEnum, HttpEntityEnclosingRequestBase base, JSONObject contentObj, String urlAppend) throws IOException {
        int isAep = getIsAep();
        String header = null;
        String content = null;
        String appId = null;
        if(typeEnum != IOTBusinessTypeEnum.AUTH && typeEnum != IOTBusinessTypeEnum.REFRESH_AUTH){
            String accessToken = AuthenticationTools.getAuthToken();
            if(isAep == 1){
                appId = WebMeterInfo.getString("aepAPIKey");
            } else if(isAep == 0){
                appId = WebMeterInfo.getString("telecomAPIKey");
            } else {
                appId = WebMeterInfo.getString("telecom2APIKey");
            }
            JSONObject headerObj = new JSONObject();
            headerObj.put("app_key", appId);
            headerObj.put("Authorization", "Bearer" + " " + accessToken);
            headerObj.put("Content-Type","application/json");
            header = headerObj.toString();
        }

        if(contentObj != null){
            content = contentObj.toString();
        }

        String url;
        CloseableHttpClient client;
        //如果是AEP模式，使用AEP地址和客户端
        switch (typeEnum){
            case AUTH:
                url = TelecomApi.getAuthUrl();
                break;
            case REFRESH_AUTH:
                url = TelecomApi.getRefreshAuthUrl();
                break;
            case CREATE_DEVICE:
                url = TelecomApi.getDeviceCreateUrl();
                break;
            case MODIFY_DEVICE:
            case REMOVE_DEVICE:
                url = TelecomApi.getDeviceUpdateUrl() + "/" + urlAppend;
                break;
            case SEND_COMMAND:
                url = TelecomApi.getSendInstructUrl();
                break;
            case SUBSCRIPTIONS:
                url = TelecomApi.getSubscriptionsUrl();
                break;
            default:
                throw new RuntimeException("不支持的业务类型："+typeEnum.name());
        }
        if(isAep == 1){
            client = HttpConnectionPoolUtil.getHttpClient();
        } else {
            client = AuthenticationTools.getHttpClient();
        }

        String responseBody;
        if(typeEnum == IOTBusinessTypeEnum.AUTH){
            assert contentObj != null;
            LOGGER.debug("请求的地址："+ url);
            LOGGER.debug("请求体："+ content);
            responseBody = HttpConnectionPoolUtil.requestFormUrlEncoded(url, contentObj, base, client);
        } else {
            LOGGER.debug("请求的地址："+ url);
            LOGGER.debug("请求头："+ header);
            LOGGER.debug("请求体："+ content);
            responseBody = HttpConnectionPoolUtil.request(url, content, header, base, client);
        }
        JSONObject result;
        if(responseBody != null){
            try{
                result = new JSONObject(responseBody);
            } catch (JSONException ignore){
                result = new JSONObject();
                result.put("value", responseBody);
            }
        } else {
            result = new JSONObject();
        }
        LOGGER.debug("调用平台API结果：" + result);
        return result;
    }

    /**
     * 清除设置的AEP模式
     */
    public static void removeIsAep(){
        IS_AEP.remove();
    }

    /**
     * 设置AEP模式
     */
    public static void setIsAep(Object isAep){
        if(isAep == null){
            isAep = 0;
        }
        IS_AEP.set(Integer.valueOf(String.valueOf(isAep)));
    }

    /**
     * 获取AEP模式
     */
    public static Integer getIsAep(){
        Integer b = IS_AEP.get();
        if (b == null){
            return 0;
        }
        return b;
    }
}
