package com.aote.webmeter.tools.iot;

import com.af.plugins.HttpConnectionPoolUtil;
import com.aote.redis.RedisUtil;
import com.aote.webmeter.enums.IOTBusinessTypeEnum;
import com.aote.webmeter.tools.WebMeterInfo;
import com.iotplatform.client.NorthApiException;
import com.iotplatform.constant.AuthConstant;
import com.iotplatform.constant.ExceptionEnum;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.log4j.Logger;
import org.json.JSONObject;

import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;
import java.io.IOException;
import java.io.InputStream;
import java.security.KeyStore;

/**
 * 电信IOT平台 应用接口鉴权
 * @author Mr.river
 */
public class AuthenticationTools {

	private static final Logger LOGGER = Logger.getLogger(AuthenticationTools.class);

	/**
	 * 发送请求的客户端单例
 	 */
	private static volatile CloseableHttpClient httpClient;

	/**
	 * 获取HTTP客户端单例
	 */
	public static CloseableHttpClient getHttpClient(){
		if (httpClient == null){
			//多线程下同时调用getHttpClient容易导致重复创建httpClient对象的问题,所以加上了同步锁
			synchronized (AuthenticationTools.class){
				if (httpClient == null){
					try {
						httpClient = HttpConnectionPoolUtil.getHttpClient(initSslConfig());
					} catch (NorthApiException e) {
						e.printStackTrace();
					}
				}
			}
		}
		return httpClient;
	}

	/**
	 * 获取TOKEN
	 * @return TOKEN值
	 */
	public static String getAuthToken() {
		int isAep = IotCommonTools.getIsAep();
		String appId;
		String secret;
		if(isAep == 1){
			appId = WebMeterInfo.getString("aepAPIKey");
			secret = WebMeterInfo.getString("aepSecret");
		} else if(isAep == 0){
			appId = WebMeterInfo.getString("telecomAPIKey");
			secret = WebMeterInfo.getString("telecomSecret");
		} else {
			appId = WebMeterInfo.getString("telecom2APIKey");
			secret = WebMeterInfo.getString("telecom2Secret");
		}
		String token = getToken(appId);
		if (token != null) {
			return token;
		} else {
			try {
				return getWebAuthToken(appId, secret);
			} catch (IOException e) {
				e.printStackTrace();
				return null;
			}
		}
	}

	/**
	 * 发起获取TOKEN的请求
	 *
	 * @return TOKEN信息
	 * @throws IOException IO异常
	 */
	public static synchronized String getWebAuthToken(String appId, String secret) throws IOException {
		RedisUtil redisUtil = RedisUtil.getInstance();
		final String[] newAccessToken = new String[1];
		redisUtil.lock(appId,()->{
			String token = getToken(appId);
			if (token == null) {
				JSONObject content = new JSONObject();
				content.put("appId", appId);
				content.put("secret", secret);

				JSONObject result;
				LOGGER.info(appId + ":进行鉴权操作");
				try {
					String refreshToken = getRefreshToken(appId);
					if(refreshToken != null){
						content.put("refreshToken", refreshToken);
						result = IotCommonTools.post(IOTBusinessTypeEnum.REFRESH_AUTH, new HttpPost(), content, null);
					} else {
						result = IotCommonTools.post(IOTBusinessTypeEnum.AUTH, new HttpPost(), content, null);
					}
					if(result.has("errorEntity")){
						throw new RuntimeException(result.toString());
					} else {
						newAccessToken[0] = result.getString("accessToken");
						redisUtil.set(appId+"_token", newAccessToken[0], result.getInt("expiresIn") - 20);
						if(refreshToken == null){
							String newRefreshToken = result.getString("refreshToken");
							redisUtil.set(appId+"_refreshToken", newRefreshToken, (3600 * 12) - 20);
						}
					}
				} catch (IOException e) {
					e.printStackTrace();
				}
			} else {
				newAccessToken[0] = token;
			}
		});
		return newAccessToken[0];
	}

	private static String getToken(String appId) {
		RedisUtil redisUtil = RedisUtil.getInstance();
		Object token = redisUtil.get(appId+"_token");
		if(token != null){
			LOGGER.info(appId + ":获取现有token");
			return String.valueOf(token);
		} else {
			return null;
		}
	}

	private static String getRefreshToken(String appId) {
		RedisUtil redisUtil = RedisUtil.getInstance();
		Object token = redisUtil.get(appId+"_refreshToken");
		if(token != null){
			LOGGER.info(appId + ":获取现有refresh_token");
			return String.valueOf(token);
		} else {
			return null;
		}
	}

	/**
	 * 初始化SSL证书
	 *
	 * @return SSL连接工厂
	 * @throws NorthApiException IOT平台请求异常
	 */
	protected static SSLConnectionSocketFactory initSslConfig() throws NorthApiException {
		try (InputStream isTrustCa = SignalDeliveryTools.class.getResourceAsStream("/ca.jks");
			 InputStream isSelfCert = SignalDeliveryTools.class.getResourceAsStream("/outgoing.CertwithKey.pkcs12")) {
			KeyStore selfCert = KeyStore.getInstance("pkcs12");
			selfCert.load(isSelfCert, AuthConstant.SELFCERTPWD.toCharArray());
			KeyManagerFactory kmf = KeyManagerFactory.getInstance("sunx509");
			kmf.init(selfCert, AuthConstant.SELFCERTPWD.toCharArray());
			KeyStore caCert = KeyStore.getInstance("jks");
			caCert.load(isTrustCa, AuthConstant.TRUSTCAPWD.toCharArray());
			TrustManagerFactory tmf = TrustManagerFactory.getInstance("sunx509");
			tmf.init(caCert);
			SSLContext sc = SSLContext.getInstance("TLS");
			sc.init(kmf.getKeyManagers(), tmf.getTrustManagers(), null);
			return new SSLConnectionSocketFactory(sc, NoopHostnameVerifier.INSTANCE);
		} catch (Exception var10) {
			throw new NorthApiException(ExceptionEnum.CLIENT_SSL_CONFIG_ERROR, var10.getMessage());
		}
	}
}
