package utils;

import com.af.plugins.UrlTools;
import com.aote.rs.mapper.WebException;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.json.JSONObject;

import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;

/**
 * REST风格请求插件
 */
public class RestTools {

    /**
     * 发送通用Http请求
     * @param path  请求路径
     * @param value 请求参数
     * @param headersStr    请求头
     * @param base  请求类型
     * @return  请求结果
     */
    public static String request(String path, String value, String headersStr, HttpEntityEnclosingRequestBase base){
        if (!path.startsWith("http")) {
            // 从系统配置中获得path对应的地址
            path = UrlTools.getUrl(path);
        }
        //设置请求地址
        base.setURI(URI.create(path));
        //设置请求体
        if(value != null && !"".equals(value)){
            StringEntity se = new StringEntity(value,StandardCharsets.UTF_8);
            base.setEntity(se);
        }
        //设置请求头
        setHeaders(headersStr, base);
        //发送请求
        HttpClient httpClient = HttpClientBuilder.create().build();
        try {
            HttpResponse response = httpClient.execute(base);
            int code = response.getStatusLine().getStatusCode();

            // 获取数据成功，返回数据
            if (code == 200) {
                return EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);
            } else {
                String data = response.getStatusLine().getReasonPhrase();
                // 返回错误码
                return "{status: " + code + ", data: '" + data + "'}";
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 发送通用Http请求
     * @param path  请求路径
     * @param value 请求参数
     * @param headersStr    请求头
     * @param base  请求类型
     * @return  请求结果
     */
    public static HttpEntity requestPostFile(String path, String value, String headersStr, HttpEntityEnclosingRequestBase base){
        if (!path.startsWith("http")) {
            // 从系统配置中获得path对应的地址
            path = UrlTools.getUrl(path);
        }
        //设置请求地址
        base.setURI(URI.create(path));
        //设置请求体
        if(value != null && !"".equals(value)){
            StringEntity se = new StringEntity(value,StandardCharsets.UTF_8);
            base.setEntity(se);
        }
        //设置请求头
        setHeaders(headersStr, base);
        // 发送请求
        HttpClient httpClient = HttpClientBuilder.create().build();
        try {
            HttpResponse response = httpClient.execute(base);
            int code = response.getStatusLine().getStatusCode();

            // 获取数据成功，返回数据
            if (code == 200) {
                return response.getEntity();
            }else {
                throw new WebException(code,"获取文件失败");
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 设置请求头，如果线程局部变量里有Token，把Token也发送过去
     * @param headersStr
     * @param base
     */
    private static void setHeaders(String headersStr, HttpEntityEnclosingRequestBase base) {
        // 设置token
        String token = ThreadResource.Token.get();
        if (token != null) {
            base.setHeader("Token", token);
        }

        // 设置其他请求头
        if (headersStr != null && !"".equals(headersStr)) {
            JSONObject headers = new JSONObject(headersStr);
            Iterator keys = headers.keys();
            while (keys.hasNext()) {
                String key = (String) keys.next();
                String val = headers.getString(key);
                base.setHeader(key, val);
            }
        }
    }

    /**
     * 发送post请求
     *
     * @param path  请求路径
     * @return String结果
     */
    public static String post(String path){
        return post(path,null,null);
    }

    /**
     * 发送带请求体JSON的post请求
     *
     * @param path    请求路径
     * @param value    请求参数
     * @return String结果
     */
    public static String post(String path, JSONObject value) {
        return post(path, value, null);
    }

    /**
     * 发送带请求体JSON的post请求
     *
     * @param path    请求路径
     * @param value    请求参数
     * @return String结果
     */
    public static HttpEntity postHttpEntity(String path, JSONObject value) {
        JSONObject headers = new JSONObject();
        headers.put("Content-Type", "application/json");
        return requestPostFile(path,value.toString(),headers.toString(),new HttpPost());
    }

    /**
     * 发送带请求体字符串的post请求
     *
     * @param path  请求路径
     * @param value 请求参数
     * @return String结果
     */
    public static String post(String path, String value) {
        JSONObject headers = new JSONObject();
        return request(path,value,headers.toString(),new HttpPost());
    }

    /**
     * 发送带请求体字符串的带请求头的post请求
     *
     * @param path  请求路径
     * @param value 请求参数
     * @return String结果
     */
    public static String postheader(String path, String value) {
        JSONObject headers = new JSONObject();
        headers.put("Content-Type", "application/json");
        return request(path,value,headers.toString(),new HttpPost());
    }

    /**
     * 发送带请求头和请求体的post请求
     *
     * @param path    请求路径
     * @param value    请求参数
     * @param headers    请求头
     * @return 请求结果
     */
    public static String post(String path, JSONObject value, JSONObject headers) {
        if(value != null){
            if(headers == null){
                headers = new JSONObject();
            }
            if(!headers.has("Content-Type")){
                //指定请求参数的数据格式是JSON。
                headers.put("Content-Type", "application/json");
            }
            return request(path,value.toString(),headers.toString(),new HttpPost());
        } else {
            if(headers == null){
                return request(path,"","",new HttpPost());
            }
            return request(path,"",headers.toString(),new HttpPost());
        }
    }
    /**
     * 获取标准格式的请求路径
     * @param protocolType 协议类型（http,https,ftp...）
     * @param url 请求ip,域名
     * @param port 端口，默认80
     * @param paramStr 需要拼接的路径
     * @return 标准格式的请求路径
     */
    public static String getStandardURL(String protocolType,String url,String port,String paramStr){
        if (port == null){
            port = "80";
        }
        url = protocolType+"://"+url+":"+port;
        if(paramStr != null){
            url += paramStr;
        }
        return url;
    }
}
