package com.aote.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.log4j.Logger;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Comment;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellAddress;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

/**
 * 数据导出到excel
 * Add template based excel export functionality
 */
public class ExcelUtil {

	private static Logger logger = Logger.getLogger(ExcelUtil.class);
	
	public final static String XLS = "xls";
	public final static String XLSX = "xlsx";

	private Workbook theBook;
	private Sheet lastSheet;
	private boolean hasTemplate;
	private String filePath;
	
	private int firstDataRow;
	private int firstDataCol;
	//private String[] shownCols;
	
	/**
	 * create a workbook
	 * @param templateName
	 * @throws Exception
	 */
	public void createBook(String templateName, String filePath) throws Exception {
		this.filePath = filePath;
		this.hasTemplate = (templateName != null && !templateName.isEmpty());
		//export based on excel template, clone a new workbook
		if(hasTemplate) {
			initWork(templateName);
		//otherwise, a plain one
		} else {
			theBook = filePath.endsWith(XLSX) ?  new XSSFWorkbook() : new HSSFWorkbook();
		}
	}
	
	/**
	 * create a new sheet
	 * @throws Exception
	 */
	public void createSheet() throws Exception {
		if(hasTemplate) {
			//always clone the first one which will end up being destroyed 
			lastSheet = theBook.cloneSheet(0);		
		} else {
			lastSheet = theBook.createSheet();
		}
	}
	
	/**
	 * init work for template export
	 * @param templateName
	 * @param filePath
	 * @throws Exception
	 */
	private void initWork(String templateName) throws Exception {
		InputStream templateStream = this.getClass().getClassLoader().getResourceAsStream(templateName);
			
		FileOutputStream fileOut = new FileOutputStream(handleFileName());
		
		byte[] bs = new byte[256];
		int n = 0;
		while(n != -1) {
			n = templateStream.read(bs);
			if(n != -1)
				fileOut.write(bs, 0, n);
		} 
		
		fileOut.close();
		templateStream.close();
		
		templateStream = new FileInputStream(filePath);
		theBook = templateName.endsWith(XLSX) ? new XSSFWorkbook(templateStream) : new HSSFWorkbook(templateStream);
		collectComments();
	}

	
	private String handleFileName() {
		String fileName = filePath.replace("/", File.separator).replace("\\\\", File.separator);
		File destDir = new File(filePath.substring(0, fileName.lastIndexOf(File.separator)));
		if (!destDir.exists()) {
			destDir.mkdirs();
		}
		return fileName;
	}

	/**
	 * Find out the first data row starting row and col index
	 * The first data row's first column must be commented and the only one be commented 
	 */
	private void collectComments() {
	    Map<CellAddress, Comment> comments = (Map<CellAddress, Comment>) theBook.getSheetAt(0).getCellComments();
	    
//	    if(comments.size() > 1) {
//	    	shownCols = new String[comments.size()];
//	    } else {
//	    	shownCols = new String[0];
//	    }
	    
	    for (Entry<CellAddress, ? extends Comment> e : comments.entrySet()) {
	      CellAddress loc = e.getKey();
	      if(firstDataRow < loc.getRow())
	    	  firstDataRow = loc.getRow();
	      if(firstDataCol < loc.getColumn())
	    	  firstDataCol = loc.getColumn();
	    }
	    
	}


	/**
	 * write data to sheet
	 * @param rows
	 * @param header
	 * @param footer 
	 */
	public void writeToSheet(List<List<Object>> rows, String[] header, String[][] footer) {
		if (rows == null || rows.size() == 0)
			return;
		int cCount = rows.get(0).size() + firstDataCol;
		
		//write header
		int startRow = 1;
		if(!hasTemplate) {
			writeHeader(header, rows.get(0).size());
		} else {
			startRow = firstDataRow;
		}
		int n = rows.size();
		
		//write data rows
		for (int i = 0; i < n; i++) {
			Row row = lastSheet.createRow(startRow);
			for (int j = firstDataCol; j < cCount; j++) {
				Object value = rows.get(i).get(j-firstDataCol);
				Cell cell = row.createCell(j);
				// 设置数据部分
				if (value != null) {
					if (value instanceof Boolean)
						cell.setCellValue((Boolean) value ? "是" : "否");
					if (value instanceof Date)
						cell.setCellValue(value + "");
					else
						cell.setCellValue(value == null ? "" : value + "");
				}
			}
			startRow++;
		}
		
		if(footer == null)
			return;
		//write the footer
		for(int i=0; i < footer.length; i++) {
			Row row = lastSheet.createRow(startRow);			
			for(int j=0; j<footer[i].length; j++) {
				String value = footer[i][j];
				Cell cell = row.createCell(j + firstDataCol);
				cell.setCellValue(value == null ? "" : value + "");
			}
			startRow++;
		}
	}
	
	/**
	 * write header
	 * @param header
	 * @param cCount
	 */
	private void writeHeader(String[] header, int cCount) {
		Row row = lastSheet.createRow(0);
		for (int j = 0; j < cCount; j++) {
			Object value = header[j];
			Cell cell = row.createCell(j);
			// 设置数据部分
			if (value != null) {
				cell.setCellValue(value == null ? "" : value + "");
				lastSheet.setColumnWidth(j, (value + "").getBytes().length * 256);
			}
		}
	}

	/**
	 * save the workbook
	 * @throws Exception
	 */
	public void saveBook() throws Exception {
		//delete the template sheet and wrap it up
		if(hasTemplate) {
			theBook.removeSheetAt(0);
			for(int i= 1; i<= theBook.getNumberOfSheets(); i++)
			theBook.setSheetName(i-1, "工作表" + i);
		}
		String fileName = handleFileName();
		theBook.write(new FileOutputStream(fileName));
		theBook.close();
	}

	/**
	 * 读入excel文件，解析后返回
	 * 
	 * @param file
	 * @throws IOException
	 */
	public static List<String[]> readExcel(InputStream is, String fileName) throws IOException {
		// 检查文件
		checkFile(is);
		// 获得Workbook工作薄对象
		Workbook workbook = getWorkBook(is, fileName);
		// 创建返回对象，把每行中的值作为一个数组，所有行作为一个集合返回
		List<String[]> list = new ArrayList<String[]>();
		if (workbook != null) {
			for (int sheetNum = 0; sheetNum < workbook.getNumberOfSheets(); sheetNum++) {
				// 获得当前sheet工作表
				Sheet sheet = workbook.getSheetAt(sheetNum);
				if (sheet == null) {
					continue;
				}
				// 获得当前sheet的开始行
				int firstRowNum = sheet.getFirstRowNum();
				// 获得当前sheet的结束行
				int lastRowNum = sheet.getLastRowNum();
				if (sheetNum == 0) {
					firstRowNum = firstRowNum - 1;
				}
				// 循环除了第一行的所有行
				// 列数以第一行为准
				int lastCellNum = 0;
				for (int rowNum = firstRowNum + 1; rowNum <= lastRowNum; rowNum++) {
					// 获得当前行
					Row row = sheet.getRow(rowNum);
					if (row == null) {
						continue;
					}
					// 获得当前行的开始列
					int firstCellNum = row.getFirstCellNum();
					// 获得当前行的列数
//					int lastCellNum = row.getPhysicalNumberOfCells();
					if(rowNum == 0) {
						lastCellNum = row.getPhysicalNumberOfCells();
					}
					String[] cells = new String[lastCellNum];
					// 循环当前行
					for (int cellNum = firstCellNum; cellNum < lastCellNum; cellNum++) {
						Cell cell = row.getCell(cellNum);
						cells[cellNum] = getCellValue(cell);
					}
					list.add(cells);
				}
			}
			workbook.close();
		}
		return list;
	}

	private static void checkFile(InputStream is) throws IOException {
		// 判断文件是否存在
		if (null == is) {
			logger.error("文件不存在！");
			throw new FileNotFoundException("文件不存在！");
		}
	}

	private static Workbook getWorkBook(InputStream is, String fileName) {
		//创建Workbook工作薄对象，表示整个excel  
        Workbook workbook = null;  
		try {
			// 获取excel文件的io流
			//InputStream is = file.getInputStream();
			// 根据文件后缀名不同(xls和xlsx)获得不同的Workbook实现类对象
			if (fileName.endsWith(XLS)) {
				// 2003
				workbook = new HSSFWorkbook(is);
			} else if (fileName.endsWith(XLSX)) {
				// 2007
				workbook = new XSSFWorkbook(is);
			}
		} catch (IOException e) {
			logger.info(e.getMessage());
		}
		return workbook;
	}

	private static String getCellValue(Cell cell) {
		String cellValue = "";
		if (cell == null) {
			return cellValue;
		}
		// 把数字当成String来读，避免出现1读成1.0的情况
		if (cell.getCellType() == Cell.CELL_TYPE_NUMERIC) {
			cell.setCellType(Cell.CELL_TYPE_STRING);
		}
		// 判断数据的类型
		switch (cell.getCellType()) {
		case Cell.CELL_TYPE_NUMERIC: // 数字
			cellValue = String.valueOf(cell.getNumericCellValue());
			break;
		case Cell.CELL_TYPE_STRING: // 字符串
			cellValue = String.valueOf(cell.getStringCellValue());
			break;
		case Cell.CELL_TYPE_BOOLEAN: // Boolean
			cellValue = String.valueOf(cell.getBooleanCellValue());
			break;
		case Cell.CELL_TYPE_FORMULA: // 公式
			cellValue = String.valueOf(cell.getCellFormula());
			break;
		case Cell.CELL_TYPE_BLANK: // 空值
			cellValue = "";
			break;
		case Cell.CELL_TYPE_ERROR: // 故障
			cellValue = "非法字符";
			break;
		default:
			cellValue = "未知类型";
			break;
		}
		return cellValue;
	}

}