package com.aote.sql;

import java.util.Iterator;
import java.util.Map;

import org.apache.log4j.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.aote.util.ExpressionHelper;
import com.aote.util.JsonHelper;
import com.aote.util.ResourceHelper;
import com.aote.util.SqlHelper;

@Component
public class SqlServer {
	static Logger log = Logger.getLogger(SqlServer.class);

	@Autowired
	public SessionFactory sessionFactory;
	
	public SessionFactory getSessionFactory() {
		return sessionFactory;
	}
	
	public Session assignedSession;

	public void setAssignedSession(Session session) {
		this.assignedSession = session;
	}


	/**
	 * 获取SQL语句的合计执行结果
	 * 
	 * @param name
	 *            : sql语句名
	 * @param str
	 *            : sql语句执行参数
	 * @return JSON格式Sql语句执行结果
	 * @throws JSONException
	 */
	public JSONObject queryTotal(String name, String str) throws Exception {

		// 获取参数，求和字段等内容
		JSONObject param = null;
		Object sums = null;
		if (str != null && !str.isEmpty()) {
			JSONObject json = new JSONObject(str);
			if (json.has("data")) {
				param = json.getJSONObject("data");
			}
			if (json.has("sums")) {
				sums = json.get("sums");
			}
		}
		Map<String, Object> params = JsonHelper.toMap(param);
		// 产生sql语句编译后的结果
		String sql = this.call(name, params);

		// 求和时，order by会导致sql错误，过滤掉order by部分。
		sql = filterOutOrderBy(sql, sums);
		Session session = (assignedSession==null ? sessionFactory.getCurrentSession() : assignedSession);
		JSONArray array = SqlHelper.query(session, sql);
		return array.getJSONObject(0);
	}

	/**
	 * 获取HQL语句的合计执行结果
	 * 
	 * @param name
	 *            : hql语句名
	 * @param str
	 *            : hql语句执行参数
	 * @return JSON格式hql语句执行结果
	 * @throws JSONException
	 */
	public JSONObject queryHqlTotal(String name, String str) throws Exception {
		// 获取参数，求和字段等内容
		JSONObject param = null;
		if (str != null && !str.isEmpty()) {
			JSONObject json = new JSONObject(str);
			if (json.has("data")) {
				param = json.getJSONObject("data");
			}
		}
		Map<String, Object> params = JsonHelper.toMap(param);
		// 产生sql语句编译后的结果
		String sql = this.call(name, params);
		Session session = (assignedSession==null ? sessionFactory.getCurrentSession() : assignedSession);
		JSONArray array = SqlHelper.hqlQuery(session, sql);
		return array.getJSONObject(0);
	}

	/**
	 * 执行sql分页查询
	 */
	public JSONArray query(String name, String str) throws Exception {
		return query(name, 1, 1000, str);
	}
	
	public JSONArray query(String name, String str, Integer pageSize) throws Exception {
		return pageSize <= 0  ? query(name, 1, 9999999, str) : query(name, 1, (Integer)pageSize, str);
	}
	
	// 参数为json对象
	public JSONArray query(String name, JSONObject params) throws Exception {
		Map<String, Object> map = JsonHelper.toMap(params);
		return query(name, 1, 1000, map);
	}
	
	public JSONArray query(String name, JSONObject params, Integer pageSize) throws Exception {
		Map<String, Object> map = JsonHelper.toMap(params);
		return pageSize <= 0 ? query(name, 1, 9999999, map) : query(name, 1, pageSize, map);
	}

	// 参数为map
	public JSONArray query(String name, Map<String, Object> params)
			throws Exception {
		return query(name, 1, 1000, params);
	}

	public JSONArray query(String name, Map<String, Object> params, Integer pageSize)
			throws Exception {
		return pageSize <= 0 ? query(name, 1, 9999999, params) : query(name, 1, pageSize, params);
	}
	
	public JSONArray query(String name, int pageNo, int pageSize, String str)
			throws Exception {
		// pageNo小于0， 纠正成1
		if (pageNo <= 0) {
			pageNo = 1;
		}

		// pageSize小于0，纠正成1, 解除1000限制
		if (pageSize < 1) {
			pageSize = 1000;
		}

		// 拿到json对象参数
		JSONObject param = null;
		if (str != null && !str.isEmpty()) {
			param = new JSONObject(str);
			param = param.getJSONObject("data");
		}
		Map<String, Object> params = JsonHelper.toMap(param);
		return query(name, pageNo, pageSize, params);
	}

	public JSONArray query(String name, int pageNo, int pageSize,
			Map<String, Object> params) throws Exception {
		// 产生SQL语句编译后的结果
		String sql = this.call(name, params);

		Session session = (assignedSession==null ? sessionFactory.getCurrentSession() : assignedSession);
		JSONArray array = SqlHelper.query(session, sql, pageNo - 1, pageSize);
		log.debug(array.toString());
		return array;
	}

	public JSONArray query(String sql) throws Exception{
		Session session = (assignedSession==null ? sessionFactory.getCurrentSession() : assignedSession);
		JSONArray array = SqlHelper.query(session, sql);
		log.debug(array.toString());
		return array;
	}
	
	// 执行sql语句
	public void run(String sql) {
		Session session = (assignedSession==null ? sessionFactory.getCurrentSession() : assignedSession);
		SqlHelper.bulkUpdate(session, sql);
	}

	// 执行sql语句，非hql 返回受影响行数
	public int runSQL(String sql) {
		Session session = (assignedSession==null ? sessionFactory.getCurrentSession() : assignedSession);
		return SqlHelper.bulkSQLUpdate(session, sql);
	}

	// 基于JSON对象的call函数
	public String call(String sqlName, JSONObject json) throws Exception {
		Map<String, Object> param = JsonHelper.toMap(json);
		return call(sqlName, param);
	}

	// 调用其它sql语句, 产生sql语句经过参数处理的结果串
	public String call(String sqlName, Map<String, Object> params) {
		// 获取原始sql语句
		String path = SqlMapper.getSql(sqlName);
		String sql = ResourceHelper.getString(path);

		// 获取编译后的sql语句
		sql = sql.replace("\r\n", "\n");
		sql = "$" + sql;
		// 把自身注册到执行环境中
		params.put("sql", this);
		sql = ExpressionHelper.run(sql, params).toString();

		return sql;
	}

	// 执行基本hql查询
	public JSONArray hqlQuery(String name, JSONObject params) throws Exception {
		Map<String, Object> map = JsonHelper.toMap(params);
		String sql = this.call(name, map);

		Session session = (assignedSession==null ? sessionFactory.getCurrentSession() : assignedSession);
		JSONArray array = SqlHelper.hqlQuery(session, sql);
		return array;

	}

	// 直接执行基本hql查询
	public JSONArray hqlQuery(String hql) throws Exception {
		Session session = (assignedSession==null ? sessionFactory.getCurrentSession() : assignedSession);
		JSONArray array = SqlHelper.hqlQuery(session, hql);
		return array;

	}

	// hql 分页查询入口
	public JSONArray hqlQuery(String name, int pageNo, int pageSize, String str)
			throws Exception {
		// pageNo小于0， 纠正成1
		if (pageNo <= 0) {
			pageNo = 1;
		}

		// pageSize小于0，纠正成1
		if (pageSize < 1 || pageSize > 1000) {
			pageSize = 1000;
		}

		// 拿到json对象参数
		JSONObject param = null;
		if (str != null && !str.isEmpty()) {
			param = new JSONObject(str);
			param = param.getJSONObject("data");
		}
		Map<String, Object> params = JsonHelper.toMap(param);
		return hqlQuery(name, pageNo, pageSize, params);
	}

	// 执行分页查询
	public JSONArray hqlQuery(String name, int pageNo, int pageSize,
			Map<String, Object> params) throws Exception {

		String sql = this.call(name, params);

		Session session = (assignedSession==null ? sessionFactory.getCurrentSession() : assignedSession);
		JSONArray array = SqlHelper
				.hqlQuery(session, sql, pageNo - 1, pageSize);
		log.debug(array.toString());
		return array;
	}

	// 过滤order by子句，产生求和结果
	private String filterOutOrderBy(String source, Object sums)
			throws Exception {
		int idx = source.toLowerCase().lastIndexOf("order by");
		String sql = "select ";
		// 如果有求和部分，产生求和部分的语句
		if (sums != null) {
			if (sums instanceof JSONArray) {
				JSONArray arraySums = (JSONArray) sums;
				for (int i = 0; i < arraySums.length(); i++) {
					String name = (String) arraySums.get(i);
					sql += "sum(" + name + ") " + name + ", ";
				}
			}else if (sums instanceof JSONObject) {
				JSONObject objSums = (JSONObject) sums;
				Iterator<String> keys = objSums.keys();
				while (keys.hasNext()) {
					String name = keys.next();
					String value = objSums.getString(name);
					sql += "sum(" + value + ") " + name + ", ";
				}
			}
		}
		if (idx != -1)
			sql += "count(*) n, 1 placeholder from ( "
					+ source.substring(0, idx) + ") t_";
		return sql;
	}

	/**
	 * 模拟平板端执行过程
	 */
	public JSONObject action(JSONObject params) {
		System.out.println(params.toString());
		JSONObject result = new JSONObject("{code: 200, data: {}}");
		return result;
	}
}
