package com.aote.sql;

import org.hibernate.transform.AliasedTupleSubsetResultTransformer;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Reader;
import java.sql.Clob;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.Map;

import com.aote.sensitive.SensitiveDataEnhancer;
import com.aote.sensitive.SensitiveFieldConfigLoader;
import com.aote.util.SubstitutionCipherUtil;

public class AliasTransformer extends AliasedTupleSubsetResultTransformer {

	public static final AliasTransformer INSTANCE = new AliasTransformer();
    private static final Logger log = LoggerFactory.getLogger(AliasTransformer.class);
    
    // ThreadLocal 方式处理 SimpleDateFormat 的线程安全问题
    private static final ThreadLocal<SimpleDateFormat> DATE_FORMAT_HOLDER = 
            ThreadLocal.withInitial(() -> new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"));

	/**
	 * Disallow instantiation of AliasToEntityMapResultTransformer.
	 */
	private AliasTransformer() {
	}

	@Override
	public Object transformTuple(Object[] tuple, String[] aliases) {
		Map<String, Object> result = new LinkedHashMap<>(tuple.length);
		Map<String, String> maskMap = SensitiveFieldConfigLoader.getSensitiveMaskMap();
		
		for (int i = 0; i < aliases.length; i++) {
			String alias = aliases[i];
			if (alias != null) {
				Object value = tuple[i];
                if (value == null) {
					value = JSONObject.NULL;
				} else if (value instanceof Date) {
					value = DATE_FORMAT_HOLDER.get().format(value);
				} else if (value instanceof Clob) {
					value = getClobString((Clob) value);
				} else if (value instanceof String) {
					// 脱敏字段处理：检查权限和加密状态决定返回的值
					String lowerAlias = alias.toLowerCase();
					if (maskMap.containsKey(lowerAlias)) {
						value = processSensitiveField(lowerAlias, (String) value, maskMap);
					}
				}
				result.put(alias.toLowerCase(), value);
			}
		}
		return result;
	}

	/**
	 * 处理敏感字段的脱敏逻辑
	 * 1. 如果是加密数据，先解密
	 * 2. 根据权限决定返回明文还是脱敏值
	 */
	private Object processSensitiveField(String fieldName, String encryptedValue, Map<String, String> maskMap) {
		String plainText = encryptedValue;

		// 第一步：检查是否已加密，若是则解密
		if (SubstitutionCipherUtil.isStrictCipher(encryptedValue)) {
			try {
				plainText = SubstitutionCipherUtil.decrypt(encryptedValue);

				if (AuthUtil.hasSensitiveViewPermission(fieldName)) {
					return plainText; // 有权限：返回明文
				} else {
					String strategy = maskMap.get(fieldName);
					return SensitiveDataEnhancer.applyMaskingStrategy(fieldName, plainText, strategy); // 无权限：返回脱敏值
				}

			} catch (Exception e) {
				// 解密失败时记录警告，保持原值不做处理
				log.warn("敏感字段解密异常: field={}, error={}", fieldName, e.getMessage());
				return encryptedValue; // 返回原始密文，不进行后续处理
			}
		}
		
		return encryptedValue;
	}

	@Override
	public boolean isTransformedValueATupleElement(String[] aliases, int tupleLength) {
		return false;
	}

	/**
	 * Serialization hook for ensuring singleton uniqueing.
	 *
	 * @return The singleton instance : {@link #INSTANCE}
	 */
	private Object readResolve() {
		return INSTANCE;
	}

	public static String getClobString(Clob c) {
		try {
			Reader reader = c.getCharacterStream();
			if (reader == null) {
				return null;
			}
			StringBuilder sb = new StringBuilder();
			char[] charArr = new char[4096];
			for (int i = reader.read(charArr); i > 0; i = reader.read(charArr)) {
				sb.append(charArr, 0, i);
			}
			return sb.toString();
		} catch (Exception e) {
			return "";
		}
	}
}
