package com.aote.util;

import com.aote.exception.FileNotFoundException;
import com.aote.redis.RedisUtil;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.ConcurrentHashMap;

@Component
public class ResourceHelper {

	public enum CACHE_MODE {
		//正常模式
		NORMAL,
		//DEBUG模式，不开启缓存
		DEBUG
	}

	public static CACHE_MODE mode = CACHE_MODE.DEBUG;

	//Logic源缓存
	private static final ConcurrentHashMap<String, String> CACHE_LOGIC_MAP;
	//SQL源缓存
	private static final ConcurrentHashMap<String, String> CACHE_SQL_MAP;
	//Path源缓存
	private static final ConcurrentHashMap<String, String> CACHE_PATH_MAP;

	static {
		CACHE_LOGIC_MAP = new ConcurrentHashMap<>(10);
		CACHE_SQL_MAP = new ConcurrentHashMap<>(10);
		CACHE_PATH_MAP = new ConcurrentHashMap<>(10);
	}

	/**
	 * 获取资源文件里，指定全路径文件的文本内容
	 *
	 * @param fileName
	 *            : 以'/'开始的资源文件的全路径文件名
	 * @return 文件的文本内容
	 */
	public static String getString(String fileName) throws FileNotFoundException,IOException {
		String result;
		InputStream inputStream = null;
		BufferedReader br = null;
		try {
			// 没有/,加一个/开头
			if (!fileName.startsWith("/")) {
				fileName = "/" + fileName;
			}
			inputStream = ResourceHelper.class.getResourceAsStream(fileName);
			if(null == inputStream){
				throw new FileNotFoundException(fileName + "文件不存在", fileName);
			}
			br = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));

			String line;
			StringBuilder sb = new StringBuilder();
			while ((line=br.readLine())!=null) {
				sb.append(line).append("\r\n");
			}
			result = sb.toString();
		} finally {
			if(null != inputStream){
				inputStream.close();
			}
			if(br != null) {
				br.close();
			}
		}
		return result;
	}

	/**
	 * 获取资源流
	 * @param fileName
	 *            : 以'/'开始的资源文件的全路径文件名
	 * @return 资源输入流
	 *
	 */
	public static InputStream getStream(String fileName) throws FileNotFoundException {
		String result;
		InputStream inputStream = null;
		BufferedReader br = null;
		// 没有/,加一个/开头
		if (!fileName.startsWith("/")) {
			fileName = "/" + fileName;
		}
		inputStream = ResourceHelper.class.getResourceAsStream(fileName);
		if(null == inputStream){
			throw new FileNotFoundException(fileName + "文件不存在", fileName);
		}
		return inputStream;
	}

	/**
	 * 从输入流里获取字符串内容
	 * @param input : 输入流
	 * @return 字符串内容
	 */
	public static String getString(InputStream input) throws Exception {
		BufferedReader br = new BufferedReader(new InputStreamReader(input, StandardCharsets.UTF_8));

		String line;
		StringBuilder sb = new StringBuilder();
		while ((line=br.readLine())!=null) {
			sb.append(line).append("\r\n");
		}

		return sb.toString();
	}


	/**
	 * 获取资源文件里，指定全路径文件的文本内容，并缓存
	 *
	 * @param fileName
	 *            : 以'/'开始的资源文件的全路径文件名
	 * @return 文件的文本内容
	 */
	public static Object getString(ResourceType resourceType, String alias, String fileName) throws FileNotFoundException,IOException {
		String source;
		switch (resourceType){
			case LOGIC:
				if(mode == CACHE_MODE.NORMAL && CACHE_LOGIC_MAP.containsKey(alias)){
					return CACHE_LOGIC_MAP.get(alias);
				} else {
					String sourceStr = getString(fileName);
					// 处理回车换行
					sourceStr = sourceStr.replace("\r\n", "\n");
					if(mode == CACHE_MODE.NORMAL){
						CACHE_LOGIC_MAP.put(alias, sourceStr);
					}
					return sourceStr;
				}
			case SQL:
				if(mode == CACHE_MODE.NORMAL && CACHE_SQL_MAP.containsKey(alias)){
					return CACHE_SQL_MAP.get(alias);
				} else {
					source = getString(fileName);
					// 获取编译后的sql语句
					source = source.replace("\r\n", "\n");
					source = "$" + source;
					if(mode == CACHE_MODE.NORMAL){
						CACHE_SQL_MAP.put(alias,source);
					}
					return source;
				}
			case PATH:
				if(mode == CACHE_MODE.NORMAL && CACHE_PATH_MAP.containsKey(alias)){
					return CACHE_PATH_MAP.get(alias);
				} else {
					source = getString(fileName);
					// 获取编译后的sql语句
					source = source.replace("\r\n", "\n");
					source = "$" + source;
					if(mode == CACHE_MODE.NORMAL){
						CACHE_PATH_MAP.put(alias,source);
					}
					return source;
				}
			default:
				throw new RuntimeException("错误的缓存数据类型");
		}
	}

	/**
	 * Logic缓存清理
	 */
	public static void clearLogicCache(){
		CACHE_LOGIC_MAP.clear();
	}

	/**
	 * SQL缓存清理
	 */
	public static void clearSqlCache(){
		CACHE_SQL_MAP.clear();
	}

	/**
	 * PATH缓存清理
	 */
	public static void clearPathCache(){
		CACHE_PATH_MAP.clear();
	}
}
