package com.aote.rs;

import com.aote.exception.ExceptionLogManage;
import com.aote.sql.SqlServer;
import com.aote.util.*;
import com.aote.util.other.AESUtil;
import org.apache.log4j.Logger;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import javax.ws.rs.*;

/**
 * 提供sql查询服务
 */
@Path("sql")
@Component
@Transactional
public class SqlService {
	static Logger log = Logger.getLogger(SqlService.class);

	@Autowired
	private SqlServer sqlServer;

	@Autowired
	private ExceptionLogManage exceptionLogManage;

	/**
	 * 清理SQL源缓存
	 * @return ok
	 */
	@GET
	@Path("/clear")
	public String clearSQLCache(){
		ResourceHelper.clearLogicCache();
		ResourceHelper.clearSqlCache();
		ResourceHelper.clearPathCache();
		return "ok";
	}

	/**
	 * 获取SQL语句的合计执行结果
	 *
	 * @param name
	 *            : sql语句名
	 * @param str
	 *            : sql语句执行参数
	 */
	@POST
	@Path("{name}/n")
	public String txgetTotalCnt(@PathParam("name") String name, String str)
			throws Exception {
		try {
			return sqlEncrypt(name, str, 1);
		} catch (Exception ex) {
			String stack = ExceptionLogManage.getErrorInfo(ExceptionHelper.stackToString(ex),2);
			log.error(stack);
			exceptionLogManage.writeError(ex,name,"/rs/sql/"+name+"/n",str);
			throw ex;
		}
	}

	/**
	 * 执行sql,对sql中的参数进行替换 pageNo - 页号，默认为1 pageSize - 每页个数，默认为1000
	 */
	@POST
	@Path("{name}")
	public String txExecute(@PathParam("name") String name,
							@QueryParam("pageNo") int pageNo,
							@QueryParam("pageSize") int pageSize, String str) throws Exception {
		try {
			return sqlEncrypt(name, str, 2, pageNo, pageSize);
		} catch (Exception ex) {
			String stack = ExceptionLogManage.getErrorInfo(ExceptionHelper.stackToString(ex),2);
			log.error(stack);
			exceptionLogManage.writeError(ex,name,"/rs/sql/"+name,str);
			throw ex;
		}
	}

	public String txQuery(String sql) throws Exception {
		try {
			return sqlEncrypt(sql, "{}", 6, 0, 0);
		} catch (Exception ex) {
			String stack = ExceptionLogManage.getErrorInfo(ExceptionHelper.stackToString(ex),2);
			log.error(stack);
			throw ex;
		}
	}

	/**
	 * 获取HQL语句的合计执行结果
	 *
	 * @param name
	 *            : hql语句名
	 * @param str
	 *            : hql语句执行参数
	 */
	@POST
	@Path("hql/{name}/n")
	public String txHqlgetTotalCnt(@PathParam("name") String name, String str)
			throws Exception {
		try {
			return sqlEncrypt(name, str, 3);
		} catch (Exception ex) {
			String stack = ExceptionLogManage.getErrorInfo(ExceptionHelper.stackToString(ex),2);
			log.error(stack);
			exceptionLogManage.writeError(ex,name,"/rs/sql/hql/"+name+"/n",str);
			throw ex;
		}
	}

	/**
	 * 执行hql,对hql中的参数进行替换 pageNo - 页号，默认为1 pageSize - 每页个数，默认为1000
	 */
	@POST
	@Path("hql/{name}")
	public String txHqlExecute(@PathParam("name") String name,
							   @QueryParam("pageNo") int pageNo,
							   @QueryParam("pageSize") int pageSize, String str) throws Exception {
		try {
			return sqlEncrypt(name, str, 4, pageNo, pageSize);
		} catch (Exception ex) {
			String stack = ExceptionLogManage.getErrorInfo(ExceptionHelper.stackToString(ex),2);
			log.error(stack);
			exceptionLogManage.writeError(ex,name,"/rs/sql/hql/"+name,str);
			throw ex;
		}
	}

	/**
	 * 执行hql,对hql中的参数进行替换 pageNo - 页号，默认为1 pageSize - 每页个数，默认为1000
	 * 取消大于1000时限制1000的条件
	 */
	@POST
	@Path("hql/{name}/all")
	public String txHqlExecuteAll(@PathParam("name") String name,
								  @QueryParam("pageNo") int pageNo,
								  @QueryParam("pageSize") int pageSize, String str) throws Exception {
		try {
			return sqlEncrypt(name, str, 5, pageNo, pageSize);
		} catch (Exception ex) {
			String stack = ExceptionLogManage.getErrorInfo(ExceptionHelper.stackToString(ex),2);
			log.error(stack);
			exceptionLogManage.writeError(ex,name,"/rs/sql/hql/"+name+"/all",str);
			throw ex;
		}
	}

	/**
	 * 封装sql加解密
	 * @param sqlName sql名
	 * @param param	参数
	 * @param method 区分调用
	 * @param value	接收分页的两个参数
	 * @return
	 */
	private String sqlEncrypt(String sqlName, String param, int method, int ... value)throws Exception{
		JSONObject paramJson = new JSONObject(param);
		return AESUtil.encryptData(paramJson,
			(data) -> {
				if (method == 1) {
					return sqlServer.queryTotal(sqlName, data).toString();
				} else if (method == 2) {
					return sqlServer.query(sqlName, value[0], value[1], data).toString();
				} else if (method == 3) {
					return sqlServer.queryHqlTotal(sqlName, data).toString();
				} else if (method == 4) {
					return sqlServer.hqlQuery(sqlName, value[0], value[1], data).toString();
				} else {
					return sqlServer.hqlQueryAll(sqlName, value[0], value[1], data).toString();
				}
			}
		);
	}
}
