package com.aote.rs;

import com.aote.entity.EntityServer;
import com.aote.rs.mapper.WebException;
import com.aote.sql.SqlServer;
import com.aote.util.ExcelUtil;
import com.aote.util.ExceptionHelper;
import com.aote.util.XMLReaderUtil;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import javax.sql.DataSource;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.nio.file.Files;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

/**
 * Excel import and export service 2016-11-16 1. rewrite and tidy up the code 2.
 * add template export support
 *
 * @author LGY
 *
 */

@Path("excel")
@Component
@Transactional
public class ExcelService {

	private static Logger log = Logger.getLogger(SqlService.class);

	@Autowired
	private SqlServer sqlServer;

	@Autowired
	private DataSource dataSource;

	@Autowired
	private EntityServer entityServer;

	public static final int MAX_ROWS_PER_SHEET = 500001;

	/**
	 * 上传excel
	 *
	 * @throws Exception
	 */
	@SuppressWarnings("resource")
	@POST
	@Path("import/{entityname}")
	@Produces(MediaType.APPLICATION_OCTET_STREAM)
	public String importExcel(@PathParam("entityname") String entityname,
			String param) throws Exception {
		System.out.println(param);
		JSONObject json = new JSONObject(param);
		String filepath = (String) json.get("filepath");
		String configpath = (String) json.get("configpath");

		InputStream is = Files.newInputStream(new File(filepath).toPath());
		filepath = filepath.replace("\\", "/");
		String fileName = filepath.split("/")[filepath.split("/").length - 1];

		if (!fileName.endsWith("xls") && !fileName.endsWith("xlsx")) {
			log.error(fileName + "不是excel文件");
			throw new WebException(500, fileName + "不是excel文件");
		}
		System.out.println(fileName);
		JSONArray ja = new JSONArray();
		List<String[]> data = ExcelUtil.readExcel(is, fileName);
		XMLReaderUtil read = new XMLReaderUtil();
		Object[] field = read.getColumn(configpath, data.get(0));
		for (int i = 1; i < data.size(); i++) {
			JSONObject jo = new JSONObject();

			// 若有子json对象，保存在子json中增加的excel下角标、
			int dataIndex = 0;
			for (int j = 0; j < field.length; j++) {
				if (field[j] instanceof JSONObject) {
					JSONObject children = (JSONObject) field[j];
					// 一般只有一个
					for (String s : children.keySet()) {
						JSONObject childrenjson = new JSONObject();
						if (children.get(s) instanceof Object[]) {
							Object[] obj = (Object[]) children.get(s);
							for (int z = 0; z < obj.length; z++) {
								childrenjson.put(obj[z].toString(),
										data.get(i)[j + dataIndex]);
								if (z != obj.length - 1) {
									dataIndex++;
								}
							}
						}
						jo.put(s, childrenjson);
					}
				} else {

					jo.put(field[j].toString(), data.get(i)[j + dataIndex]);
				}

			}
			ja.put(jo);
		}
		int len = ja.length();
		for (int i = 0; i < len; i++) {
			JSONObject jo = ja.getJSONObject(i);
			System.out.println(jo);
			entityServer.partialSave(entityname, jo);
		}
		return "OK";
	}

	/**
	 * 导出报表头格式的excel(表头比较特殊)
	 *
	 */
	@POST
	@Path("report/{namedSql}/{excelTemplate}")
	@Produces(MediaType.APPLICATION_OCTET_STREAM)
	public String exportReportExcel(@PathParam("namedSql") String name,
			@PathParam("excelTemplate") String excelTemplate, String body)
			throws Exception {
		excelTemplate = "excel/" + excelTemplate + ".xlsx";
		log.debug("sql:" + name + ", template:" + excelTemplate + ", data:"
				+ body);
		try {
			long tick = System.currentTimeMillis();

			String filePath = getFilePath();

			export(name, body, excelTemplate, filePath);

			tick = System.currentTimeMillis() - tick;
			log.fatal("进行Excel导出耗时：" + (tick / 1000.0) + "秒");

			log.fatal("文件名：" + filePath);
			return filePath;
		} catch (Exception e) {
			log.error(ExceptionHelper.stackToString(e));
			throw e;
		}
	}

	/**
	 * 导出excel jsonString parameter passed in supports 2 formats, format 1, { //
	 * variables for sql module data: {f_userid: '用户编号', f_username: '用户姓名',
	 * f_address: '用户地址'}, // header for excel columns field: {f_userid: '用户编号',
	 * f_username: '用户姓名', f_address: '用户地址', 'f_districtname': '小区'} } format
	 * 2, { // same as format 1 data: {f_userid: '用户编号', f_username: '用户姓名',
	 * f_address: '用户地址'}, //header from entity configuration field: {xmlPath:
	 * 'hibernate/handplan.hbm.xml'} }
	 */
	@POST
	@Path("{name}")
	@Produces(MediaType.APPLICATION_OCTET_STREAM)
	public String exportExcel(@PathParam("name") String name, String body)
			throws Exception {
		log.debug("sql:" + name + ", data:" + body);
		try {
			long tick = System.currentTimeMillis();

			String filePath = getFilePath();

			export(name, body, null, filePath);

			tick = System.currentTimeMillis() - tick;
			log.fatal("进行Excel导出耗时：" + (tick / 1000.0) + "秒");

			log.fatal("文件名：" + filePath);
			return filePath;
		} catch (Exception e) {
			log.error(ExceptionHelper.stackToString(e));
			throw e;
		}
	}

	/**
	 * Create random export file name
	 *
	 */
	private String getFilePath() {
		String excelFileName = UUID.randomUUID() + ".xlsx";
		String path = Objects.requireNonNull(ExcelUtil.class.getClassLoader().getResource("sql.xml"))
				.getPath();
		String rootPath = path.split("WEB-INF")[0];
		return rootPath + "excel/" + excelFileName;
	}

	/**
	 * Real export stuff
	 *
	 * @param name
	 * @param templateName
	 */
	private void export(String name, String body, String templateName,
			String filePath) throws Exception {

		JSONObject joParam = new JSONObject(body);

		String[][] footer = null;
		if (joParam.has("total")) {
			JSONArray totals = joParam.getJSONArray("total");
			footer = getFooter(totals);
		}

		// 获取原始sql语句
		JSONObject joVariable = joParam.getJSONObject("data");
		String sql = new SqlServer().call(name, joVariable);

		// open connection

		try (Connection connection = dataSource.getConnection(); PreparedStatement preparedStatement = connection.prepareStatement(sql); ResultSet rs = preparedStatement.executeQuery()) {
			// fetch data begins
			rs.setFetchSize(MAX_ROWS_PER_SHEET);
			if (templateName == null) {
				exportWithHedear(filePath, joParam, sql, rs, footer);
			} else {
				exportWithTemplate(templateName, sql, filePath, rs, footer);
			}
		}
	}

	/**
	 * get footer lines
	 *
	 * @param totals
	 */
	private String[][] getFooter(JSONArray totals) {
		int n = totals.length();
		String[][] footer = new String[n][];
		for (int i = 0; i < n; i++) {
			JSONObject jo = totals.getJSONObject(i);
			String[] names = JSONObject.getNames(jo);
			Arrays.sort(names, Comparator.comparingInt(Integer::parseInt));
			footer[i] = new String[names.length];
			for (int j = 0; j < names.length; j++) {
				if (jo.isNull(names[j])) {
                    footer[i][j] = null;
                } else {
                    footer[i][j] = jo.get(names[j]) + "";
                }
			}
		}
		return footer;
	}

	/**
	 * export with template
	 *
	 * @param templateName
	 * @param sql
	 * @param filePath
	 * @param footer
	 * @throws Exception
	 */
	private void exportWithTemplate(String templateName, String sql,
			String filePath, ResultSet rs, String[][] footer) throws Exception {
		// creating excel starts
		ExcelUtil eu = new ExcelUtil();
		eu.createBook(templateName, filePath);
		int nCol = rs.getMetaData().getColumnCount();
		int n = 1;
		List<List<Object>> rows = new ArrayList<>();
		while (rs.next()) {
			if (n % MAX_ROWS_PER_SHEET == 0) {
				eu.createSheet();
				n = 1;
				eu.writeToSheet(rows, null, footer);
				rows = new ArrayList<>();
			}
			List<Object> fieldList = new ArrayList<>();
			for (int i = 1; i <= nCol; i++) {
				Object object = rs.getObject(i);
				fieldList.add(object);
			}
			rows.add(fieldList);
			n++;
		}
		// not enough to reach the max row limit
		if (n > 1) {
			eu.createSheet();
			eu.writeToSheet(rows, null, footer);
		}
		eu.saveBook();
	}

	/**
	 * export with header info
	 *
	 * @param filePath
	 * @param joParam
	 * @param sql
	 * @param rs
	 * @param footer
	 * @throws Exception
	 * @throws SQLException
	 */
	private void exportWithHedear(String filePath, JSONObject joParam,
			String sql, ResultSet rs, String[][] footer) throws Exception,
			SQLException {
		boolean fromHbm = false;

		// find out the headers
		JSONObject joField = joParam.getJSONObject("field");
		Map<String, String> colsMap = new LinkedHashMap<>();
		if (joField.has("xmlPath")) {
			findOutHeaders(joField.getString("xmlPath"), sql, colsMap);
			fromHbm = true;
		} else {
			findOutHeaders(joField, colsMap);
		}
		String[] header = colsMap.values().toArray(new String[0]);
		String[] cols = colsMap.keySet().toArray(new String[0]);

		// creating excel starts
		ExcelUtil eu = new ExcelUtil();
		eu.createBook(null, filePath);

		int n = 1;
		List<List<Object>> rows = new ArrayList<>();
		while (rs.next()) {
			if (n % MAX_ROWS_PER_SHEET == 0) {
				eu.createSheet();
				n = 1;
				eu.writeToSheet(rows, header, footer);
				rows = new ArrayList<>();
			}
			List<Object> fieldList = new ArrayList<>();
			if (fromHbm) {
                fieldList.add(rs.getObject("id"));
            }
			for (int i = (fromHbm ? 1 : 0); i < colsMap.size(); i++) {
				Object object = rs.getObject(cols[i]);
				fieldList.add(object);
			}
			rows.add(fieldList);
			n++;
		}
		// not enough to reach the max row limit
		if (n > 1) {
			eu.createSheet();
			eu.writeToSheet(rows, header, footer);
		}
		eu.saveBook();
	}

	/**
	 * find out headers
	 *
	 * @param joField
	 * @param headers
	 */
	private void findOutHeaders(JSONObject joField, Map<String, String> headers) {
		for (String col : joField.keySet()) {
			headers.put(col, joField.getString(col));
		}
	}

	/**
	 * find out headers by parsing hbm configuration
	 *
	 * @param hbmFile
	 * @param sql
	 * @param headers
	 */
	private void findOutHeaders(String hbmFile, String sql,
			Map<String, String> headers) {
		LinkedHashMap<Object, Object> comment = new XMLReaderUtil()
				.getCommont(hbmFile);
		headers.put("id", "编号");
		for (Object o : comment.keySet()) {
			String col = (String) o;
			headers.put(col, comment.get(col) + "");
		}
	}

}
