package com.aote.logic;

import com.af.plugins.RedisTools;
import com.aote.entity.EntityServer;
import com.aote.exception.ExceptionLogManage;
import com.aote.exception.FileNotFoundException;
import com.aote.path.PathMapper;
import com.aote.path.PathServer;
import com.aote.redis.RedisUtil;
import com.aote.rs.LogicService;
import com.aote.rs.mapper.WebException;
import com.aote.sql.SqlMapper;
import com.aote.sql.SqlServer;
import com.aote.transaction.SessionPool;
import com.aote.transaction.ThreadSession;
import com.aote.util.ExceptionHelper;
import com.aote.util.ExpressionHelper;
import com.aote.util.ResourceHelper;
import org.apache.log4j.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.metadata.ClassMetadata;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionSynchronizationManager;
import redis.clients.jedis.exceptions.JedisConnectionException;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.*;

@Component
@Transactional
public class LogicServer {
	static Logger log = Logger.getLogger(LogicServer.class);

	@Autowired
	private EntityServer entityServer;

	@Autowired
	private SqlServer sqlServer;

	@Autowired
	private PathServer pathServer;

	@Autowired
	private RedisUtil redisUtil;

	@Autowired
	public SessionPool sessionPool;

	@Autowired
	public SessionFactory sessionFactory;

	@Autowired
	private ExceptionLogManage exceptionLogManage;

	// 安卓模拟时需要的上下文
	private JSONObject context = null;

	/**
	 * 执行业务逻辑处理过程
	 * @param name: 业务逻辑名
	 * @param str: 业务逻辑参数
	 * @return 业务逻辑执行结果
	 */
	public Object run(String name, String str) throws Exception {
		JSONObject param = null;
		// 把传递过来的参数，放到data里，以便跟entity，sql等对象区别开来
		// 判断传来的数据是否是XML格式
		str = str.trim();
		if (str.startsWith("<") && str.endsWith(">")) {
			// 传来数据为XML格式
			param = new JSONObject();
			param.put("xml", str);
		} else {
			// 传来数据为JSON格式
			Object json = new JSONTokener(str).nextValue();
			if (json instanceof JSONObject) {
				param = new JSONObject(str);
				// 有context部分，取context部分
				if (!param.isNull("context")) {
					context = param.getJSONObject("context");
				}
				if (!param.isNull("data")) {
					Object dataParam = param.get("data");
					if (dataParam instanceof JSONObject){
						param = (JSONObject) dataParam;
						param.put("standardData",new JSONObject(str).toString());
					}
				}
			} else if (json instanceof JSONArray) {
				param = new JSONObject();
				param.put("arr", new JSONArray(str));
			}
		}

		return run(name, param, context);
	}

	// 业务逻辑执行
	public Object run(String name, JSONObject param) throws Exception {
		return run(name, param, this.context);
	}

	// 业务逻辑带一个上下问对象，用于平板端测试
	public Object run(String name, JSONObject param, JSONObject context) throws Exception {
		String message = "维护信息:开始执行Logic["+name+"]，参数：" + param;
		log.debug(message);
		// 获取源程序内容
		String path = LogicMapper.getLogic(name);
		log.debug(name + "业务逻辑路径:" + path);
		if (path == null) {
			throw new RuntimeException("业务逻辑未找到: " + name);
		}
		// 执行结果
		Object result;
		long begin = System.currentTimeMillis();
		// 记录业务执行流程堆栈
		String reviewKey = LogicService.reviewKey.get();
		JSONObject reviewItem = new JSONObject();
		if(reviewKey != null) {
			JSONArray reviewArray = LogicService.reviewCache.get(reviewKey);
			if(reviewArray != null) {
				reviewItem.put("bTime", begin);
				reviewItem.put("name", name);
				reviewItem.put("type", "logic");
				reviewArray.put(reviewItem);
			}
		}
		try (InputStream source = ResourceHelper.getStream(path)){
			//JDK1.7后支持使用try-with-resource写法，IO流的构造写在try()中；
			//InputStream source = ResourceHelper.getStream(path);
			// 获取程序运行所需完整上下文
			Map<String, Object> maps = this.getContext(param, context);
			// 获取业务逻辑语言
			String language = LogicMapper.getAttr(name, "language");
			result = ExpressionHelper.run(source, maps, language, s->s);
			reviewItem.put("status", 0);
			return result;
		} catch (FileNotFoundException e) {
			throw new RuntimeException(path + ".文件无配置");
		} catch (Exception e){
			reviewItem.put("status", -1);
			throw e;
		} finally {
			long end = System.currentTimeMillis();
			long time = end - begin;
			String engMessage = "维护信息:执行Logic["+name+"]耗时:" + time;
			if(time >= 8000){
				log.error(engMessage);
			} else if(time >= 4000){
				log.warn(engMessage);
			} else {
				log.debug(engMessage);
			}
			reviewItem.put("eTime", end);
			reviewItem.put("constTime", time);
		}
	}

	/**
	 * web层运行业务逻辑入口
	 * @param logicName 业务逻辑名
	 * @param values 业务逻辑参数
	 * @return 运行结果
	 */
	public String runMain(String logicName, String values) throws Exception {
		try {
			Object result = this.run(logicName, values);
			if (result == null) {
				return "";
			}
			// 如果执行结果为Map，转换成JSON串
			if (result instanceof Map<?, ?>) {
				JSONObject json = new JSONObject((Map<?, ?>) result);
				return json.toString();
			}
			if (result instanceof JSONObject) {
				JSONObject json = (JSONObject) result;
				return json.toString();
			}
			return result.toString();
		} catch (Exception ex) {
			String stack = ExceptionLogManage.getErrorInfo(ExceptionHelper.stackToString(ex),1);
			log.error(stack);
			// 获取源程序路径
			String path = LogicMapper.getLogic(logicName);
			if(path == null){
				path = logicName;
			}
			exceptionLogManage.writeError(ex,path,"/rs/logic/"+logicName,values);
			WebException wex = this.getWebException(ex);
			if (wex != null)
			{
				throw wex;
			}
			throw ex;
		}
	}

	@Async("AsyncLogicGet")
	public void runAsync(String logicName, String values) throws Exception {
		runMain(logicName, values);
	}

	/**
	 * 执行业务逻辑内容
	 * @param source: 业务逻辑内容
	 * @param param
	 * @return
	 * @throws Exception
	 */
	public Object runLogic(String source, JSONObject param, JSONObject context) {
		// 处理回车换行
		source = source.replace("\r\n", "\n");
		// 执行源程序
		return ExpressionHelper.run(source, this.getContext(param, context));
	}

	// 生成业务逻辑调用所需上下文
	private Map<String, Object> getContext(JSONObject param, JSONObject context) {
		Map<String, Object> params = new HashMap<>();
		params.put("data", param);
		// context用于测试平板端业务逻辑
		if (context != null) {
			params.put("context", context);
		}
		// 附加entityServer, sqlServer等对象到参数中
		params.put("log", log);
		params.put("entity", entityServer);
		params.put("sql", sqlServer);
		params.put("path", pathServer);
		params.put("session", sessionPool.getSession());
		params.put("redis",redisUtil);
		// 把LogicServer自身注册进去
		params.put("logic", this);

		// 附加用户注册的对象到业务逻辑中
		Map<String, Object> plugins = PluginMapper.getPlugins();
		params.putAll(plugins);

		return params;
	}

	// 业务逻辑内部调用其它业务逻辑
	public Object noTranscRun(String name, JSONObject param) throws Exception {
		// 获取源程序内容
		String path = LogicMapper.getLogic(name);
		String source = ResourceHelper.getString("/logics/" + path);
		// 处理回车换行
		source = source.replace("\r\n", "\n");
		// 执行源程序
		Map<String, Object> params = new HashMap<>();
		params.put("data", param);
		// 附加entityServer, sqlServer等对象到参数中
		params.put("log", log);
		params.put("entity", entityServer);
		SqlServer sqlServer = new SqlServer();
		Session session = null;
		try {
			session = sessionFactory.openSession();
			sqlServer.setAssignedSession(session);
			params.put("sql", sqlServer);
			params.put("session", session);
			// 把LogicServer自身注册进去
			params.put("logic", this);

			// 附加用户注册的对象到业务逻辑中
			Map<String, Object> plugins = PluginMapper.getPlugins();
			for (String key : plugins.keySet()) {
				params.put(key, plugins.get(key));
			}
			return ExpressionHelper.run(source, params);
		} catch (Exception e) {
			if(session != null) {
				session.getTransaction().rollback();
			}
			throw e;
		}
		finally {
			if(session != null) {
				session.close();
			}
		}

	}

	/**
	 * 将来可以根据业务逻辑名称配置
	 */
	public void debug(String logic, Object msg) {
		// 获取业务逻辑配置信息
		String level = LogicMapper.getAttr(logic, "log");
		if (level != null) {
			log.debug(logic + ":" + msg);
		}
	}
	/**
	 * redis缓存业务逻辑组件
	 */
	public void redisCacheLogic() {
		if (ResourceHelper.class.getResourceAsStream("/config.json") == null) {
			log.warn("无config.json文件，不进行redis缓存");
			return;
		}
		String strConfig;
		try {
			strConfig = ResourceHelper.getString("config.json");
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
		JSONObject jsonObject = new JSONObject(strConfig);
		if (!jsonObject.has("logicRegister")) {
			log.warn("无模块镜像，不进行redis缓存");
			return;
		}
		//获取模块镜像IP和端口号
		JSONObject logicJsonObject = jsonObject.getJSONObject("logicRegister");
		String url = logicJsonObject.getString("url");
		if (url == null){
			log.warn("无url，不进行redis缓存");
			return;
		}

		try {
			if(logicJsonObject.get("redis")instanceof JSONObject){
				//如果是个json也就是一个redis地址
				JSONObject redisObject = logicJsonObject.getJSONObject("redis");
				//获取服务器的IP和redis额端口号
				String redisHost = redisObject.getString("host");
				int redisPort = redisObject.getInt("port");
				String redispwd = "";
				try {
					//确认redis是否有密码
					redispwd = redisObject.getString("redispwd");
				}catch (JSONException e){
					log.info("无redis密码");
				}
				//获取redis连接池
				RedisTools.getJedisPool(redisHost, redisPort);
				//获取模块逻辑名称
				Map<String, Map<String, String>> map = LogicMapper.getMap();
				if (map != null) {
					//遍历逻辑集合，将逻辑及对应IP和端口号存入服务器的redis缓存中
					for (String key : map.keySet()) {
						//判断config.json里面是否有redispwd
						if ("".equals(redispwd)){
							RedisTools.setValue(key, url);
						}else {
							RedisTools.setValue(key, url, redispwd);
						}
						log.info("logic加入redis缓存，key=" + key + ", value=" + url);
					}
				}
				// 对sql语句进行注册
				map = SqlMapper.getMap();
				if (map != null) {
					//遍历逻辑集合，将逻辑及对应IP和端口号存入服务器的redis缓存中
					for (String key : map.keySet()) {
						//判断config.json里面是否有redispwd
						if ("".equals(redispwd)){
							RedisTools.setValue(key, url);
						}else {
							RedisTools.setValue(key, url, redispwd);
						}
						log.info("sql加入redis缓存，key=" + key + ", value=" + url);
					}
				}
				// 对path进行注册
				map = PathMapper.getMap();
				if (map != null) {
					//遍历逻辑集合，将逻辑及对应IP和端口号存入服务器的redis缓存中
					for (String key : map.keySet()) {
						//判断config.json里面是否有redispwd
						if ("".equals(redispwd)){
							RedisTools.setValue(key, url);
						}else {
							RedisTools.setValue(key, url, redispwd);
						}
						log.info("path加入redis缓存，key=" + key + ", value=" + url);
					}
				}
				// 对实体进行注册
				Map<String, ClassMetadata> entities = (Map<String, ClassMetadata>) sessionFactory.getAllClassMetadata();
				for (String key : entities.keySet()) {
					//判断config.json里面是否有redispwd
					if ("".equals(redispwd)){
						RedisTools.setValue(key, url);
					}else {
						RedisTools.setValue(key, url, redispwd);
					}
					log.info("entity加入redis缓存，key=" + key + ", value=" + url);
				}
			}else {
				log.info("注册多个Redis地址-----");
				//如果是个jsonarry也就是一批redis地址
				JSONArray redisarry = logicJsonObject.getJSONArray("redis");
				//批量注册redis地址
				for (int i=0;i<redisarry.length();i++){
					log.info("【开始】注册一个Redis地址"+redisarry.getJSONObject(i));
					JSONObject redisObject = redisarry.getJSONObject(i);
					//获取服务器的IP和redis额端口号
					String redisHost = redisObject.getString("host");
					int redisPort = redisObject.getInt("port");
					String redispwd = "";
					try {
						//确认redis是否有密码
						redispwd = redisObject.getString("redispwd");
					}catch (JSONException e){
						log.info("无redis密码");
					}
					//获取redis连接池
					RedisTools.getJedisPool(redisHost, redisPort);
					//获取模块逻辑名称
					Map<String, Map<String, String>> map = LogicMapper.getMap();
					if (map != null) {
						//遍历逻辑集合，将逻辑及对应IP和端口号存入服务器的redis缓存中
						for (String key : map.keySet()) {
							//判断config.json里面是否有redispwd
							if ("".equals(redispwd)){
								RedisTools.setValue(key, url);
							}else {
								RedisTools.setValue(key, url, redispwd);
							}
							log.info("logic加入redis缓存，key=" + key + ", value=" + url);
						}
					}
					// 对sql语句进行注册
					map = SqlMapper.getMap();
					if (map != null) {
						//遍历逻辑集合，将逻辑及对应IP和端口号存入服务器的redis缓存中
						for (String key : map.keySet()) {
							//判断config.json里面是否有redispwd
							if ("".equals(redispwd)){
								RedisTools.setValue(key, url);
							}else {
								RedisTools.setValue(key, url, redispwd);
							}
							log.info("sql加入redis缓存，key=" + key + ", value=" + url);
						}
					}
					// 对path进行注册
					map = PathMapper.getMap();
					if (map != null) {
						//遍历逻辑集合，将逻辑及对应IP和端口号存入服务器的redis缓存中
						for (String key : map.keySet()) {
							//判断config.json里面是否有redispwd
							if ("".equals(redispwd)){
								RedisTools.setValue(key, url);
							}else {
								RedisTools.setValue(key, url, redispwd);
							}
							log.info("path加入redis缓存，key=" + key + ", value=" + url);
						}
					}
					// 对实体进行注册
					Map<String, ClassMetadata> entities = (Map<String, ClassMetadata>) sessionFactory.getAllClassMetadata();
					for (String key : entities.keySet()) {
						//判断config.json里面是否有redispwd
						if ("".equals(redispwd)){
							RedisTools.setValue(key, url);
						}else {
							RedisTools.setValue(key, url, redispwd);
						}
						log.info("entity加入redis缓存，key=" + key + ", value=" + url);
					}
					log.info("【结束】注册一个Redis地址"+redisarry.getJSONObject(i));
				}
			}
		} catch(JedisConnectionException e) {
			log.warn("无法连接redis服务器，未成功注册业务逻辑等组件！");
		}
	}

	// 找到WebException异常
	// return - 不存在，返回null
	private WebException getWebException(Exception ex) {
		Throwable cause = ex;
		while (cause != null)
		{
			if (cause instanceof WebException)
			{
				return (WebException)cause;
			}
			cause = cause.getCause();
		}
		return null;
	}
}
