package com.aote.util;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellAddress;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.stereotype.Component;

import java.io.*;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * 数据导出到excel
 * Add template based excel export functionality
 */
@Component
public class ExcelUtil {

    private static Logger logger = Logger.getLogger(ExcelUtil.class);

    public final static String XLS = "xls";
    public final static String XLSX = "xlsx";

    private Workbook theBook;
    private Sheet lastSheet;
    private boolean hasTemplate;
    private String filePath;

    private int firstDataRow;
    private int firstDataCol;
    //private String[] shownCols;

    /**
     * create a workbook
     *
     * @param templateName
     * @throws Exception
     */
    public void createBook(String templateName, String filePath) throws Exception {
        this.filePath = filePath;
        this.hasTemplate = (templateName != null && !templateName.isEmpty());
        //export based on excel template, clone a new workbook
        if (hasTemplate) {
            initWork(templateName);
            //otherwise, a plain one
        } else {
            theBook = filePath.endsWith(XLSX) ? new XSSFWorkbook() : new HSSFWorkbook();
        }
    }

    /**
     * 大量数据导出使用
     * @param filePath 文件路径
     * @param cacheInt  放在内存中的行数(有效防止OOM)
     * @param CompressTempFiles 临时文件是否压缩(当磁盘空间不足时,建议开启,但是导出性能会降低)
     * @throws Exception
     */
    public void createBigBook(String filePath, int cacheInt, boolean CompressTempFiles) {
        this.filePath = filePath;
        SXSSFWorkbook sheets = new SXSSFWorkbook(cacheInt);
        sheets.setCompressTempFiles(CompressTempFiles);
        theBook = sheets;
    }

    public Workbook getTheBook() {
        return theBook;
    }

    /**
     * create a new sheet
     */
    public void createSheet() throws Exception {
        if (hasTemplate) {
            //always clone the first one which will end up being destroyed
            lastSheet = theBook.cloneSheet(0);
        } else {
            lastSheet = theBook.createSheet();
        }
    }

    /**
     * init work for template export
     *
     * @param templateName
     */
    private void initWork(String templateName) throws Exception {
        InputStream templateStream = this.getClass().getClassLoader().getResourceAsStream(templateName);

        FileOutputStream fileOut = new FileOutputStream(handleFileName());

        byte[] bs = new byte[256];
        int n = 0;
        while (n != -1) {
            n = templateStream.read(bs);
            if (n != -1) {
                fileOut.write(bs, 0, n);
            }
        }

        fileOut.close();
        templateStream.close();

        templateStream = new FileInputStream(filePath);
        theBook = templateName.endsWith(XLSX) ? new XSSFWorkbook(templateStream) : new HSSFWorkbook(templateStream);
        collectComments();
    }


    private String handleFileName() {
        String fileName = filePath.replace("/", File.separator).replace("\\\\", File.separator);
        File destDir = new File(filePath.substring(fileName.indexOf(File.separator) + 1, fileName.lastIndexOf(File.separator)));
        if (!destDir.exists()) {
            destDir.mkdirs();
        }
        return filePath.substring(fileName.indexOf(File.separator) + 1, fileName.length());
    }

    /**
     * Find out the first data row starting row and col index
     * The first data row's first column must be commented and the only one be commented
     */
    private void collectComments() {
        Map<CellAddress, Comment> comments = (Map<CellAddress, Comment>) theBook.getSheetAt(0).getCellComments();

//	    if(comments.size() > 1) {
//	    	shownCols = new String[comments.size()];
//	    } else {
//	    	shownCols = new String[0];
//	    }

        for (Entry<CellAddress, ? extends Comment> e : comments.entrySet()) {
            CellAddress loc = e.getKey();
            if (firstDataRow < loc.getRow()) {
                firstDataRow = loc.getRow();
            }
            if (firstDataCol < loc.getColumn()) {
                firstDataCol = loc.getColumn();
            }
        }

    }


    /**
     * 读入excel文件，解析后返回
     */
    public static List<String[]> readExcel(InputStream is, String fileName) throws IOException {
        // 检查文件
        checkFile(is);
        // 获得Workbook工作薄对象
        Workbook workbook = getWorkBook(is, fileName);
        // 创建返回对象，把每行中的值作为一个数组，所有行作为一个集合返回
        List<String[]> list = new ArrayList<>();
        if (workbook != null) {
            for (int sheetNum = 0; sheetNum < workbook.getNumberOfSheets(); sheetNum++) {
                // 获得当前sheet工作表
                Sheet sheet = workbook.getSheetAt(sheetNum);
                if (sheet == null) {
                    continue;
                }
                // 获得当前sheet的开始行
                int firstRowNum = sheet.getFirstRowNum();
                // 获得当前sheet的结束行
                int lastRowNum = sheet.getLastRowNum();
                if (sheetNum == 0) {
                    firstRowNum = firstRowNum - 1;
                }
                // 循环除了第一行的所有行
                // 列数以第一行为准
                int lastCellNum = 0;
                for (int rowNum = firstRowNum + 1; rowNum <= lastRowNum; rowNum++) {
                    // 获得当前行
                    Row row = sheet.getRow(rowNum);
                    if (row == null || row.getPhysicalNumberOfCells()==0) {
                        continue;
                    }
                    // 获得当前行的开始列
                    int firstCellNum = row.getFirstCellNum();
                    // 获得当前行的列数
//					int lastCellNum = row.getPhysicalNumberOfCells();
                    if (rowNum == 0) {
                        lastCellNum = row.getPhysicalNumberOfCells();
                    }
                    String[] cells = new String[lastCellNum];
                    // 循环当前行
                    for (int cellNum = firstCellNum; cellNum < lastCellNum; cellNum++) {
                        Cell cell = row.getCell(cellNum);
                        cells[cellNum] = getCellValue(cell);
                    }
                    if (validateCells(cells)) {
                        list.add(cells);
                    }
                }
            }
            workbook.close();
        }
        return list;
    }

    /**
     * 读入excel文件，从第rowNum行开始解析后返回
     */
    public static List<String[]> readExcel(InputStream is, String fileName,int RowNumber) throws IOException {
        // 检查文件
        checkFile(is);
        // 获得Workbook工作薄对象
        Workbook workbook = getWorkBook(is, fileName);
        // 创建返回对象，把每行中的值作为一个数组，所有行作为一个集合返回
        List<String[]> list = new ArrayList<>();
        if (workbook != null) {
            for (int sheetNum = 0; sheetNum < workbook.getNumberOfSheets(); sheetNum++) {
                // 获得当前sheet工作表
                Sheet sheet = workbook.getSheetAt(sheetNum);
                if (sheet == null) {
                    continue;
                }
                // 获得当前sheet的开始行
                int firstRowNum = sheet.getFirstRowNum();
                // 获得当前sheet的结束行
                int lastRowNum = sheet.getLastRowNum();
                if (sheetNum == 0) {
                 // 从指定行开始
                 firstRowNum = firstRowNum+RowNumber - 1;
                }
                // 循环除了第一行的所有行
                // 列数以第一行为准
                int lastCellNum = 0;
                for (int rowNum = firstRowNum + 1; rowNum <= lastRowNum; rowNum++) {
                    // 获得当前行
                    Row row = sheet.getRow(rowNum);
                    if (row == null) {
                        continue;
                    }
                    // 获得当前行的开始列
                    int firstCellNum = row.getFirstCellNum();
                    // 获得当前行的列数
//					int lastCellNum = row.getPhysicalNumberOfCells();
                    if (rowNum == RowNumber) {
                        lastCellNum = row.getLastCellNum();
                    }
                    String[] cells = new String[lastCellNum];
                    // 循环当前行
                    for (int cellNum = firstCellNum; cellNum < lastCellNum; cellNum++) {
                        Cell cell = row.getCell(cellNum);
                        cells[cellNum] = getCellValue(cell);
                    }
                    if (validateCells(cells)) {
                        list.add(cells);
                    }
                }
            }
            workbook.close();
        }
        return list;
    }

    public static boolean validateCells (String[] cells) {
        for (String str: cells) {
            if (str != null && !"".equals(str)) {
                return true;
            }
        }
        return false;
    }

    /**
     * write data to sheet
     *
     * @param rows
     * @param header
     * @param footer
     */
    public void writeToSheet(List<List<Object>> rows, String[] header, String[][] footer) {
        if (rows == null || rows.size() == 0) {
            return;
        }
        int cCount = rows.get(0).size() + firstDataCol;

        //write header
        int startRow = 1;
        if (!hasTemplate) {
            writeHeader(header, rows.get(0).size());
        } else {
            startRow = firstDataRow;
        }
        int n = rows.size();

        //write data rows
        for (List<Object> row1 : rows) {
            Row row = lastSheet.createRow(startRow);
            for (int j = firstDataCol; j < cCount; j++) {
                Object value = row1.get(j - firstDataCol);
                Cell cell = row.createCell(j);
                // 设置数据部分
                if (value != null) {
                    if (value instanceof Boolean) {
                        cell.setCellValue((Boolean) value ? "是" : "否");
                    }else if (value instanceof BigDecimal) {
                        cell.setCellValue(((BigDecimal) value).doubleValue());
                    }else if (value instanceof Date) {
                        cell.setCellValue(value + "");
                    } else {
                        cell.setCellValue(value + "");
                    }
                }
            }
            startRow++;
        }

        if (footer == null) {
            return;
        }
        //write the footer
        for (int i = 0; i < footer.length; i++) {
            Row row = lastSheet.createRow(startRow);
            for (int j = 0; j < footer[i].length; j++) {
                String value = footer[i][j];
                Cell cell = row.createCell(j + firstDataCol);
                cell.setCellValue(value == null ? "" : value + "");
            }
            startRow++;
        }
    }

    /**
     * new write data to sheet
     *
     * @param rows
     * @param header
     * @param footer
     */
    public void writeToSheet(List<List<Object>> rows, String[] header, String[][] footer,String[][] newHeader) {
        if (rows == null || rows.size() == 0) {
            return;
        }
        int cCount = rows.get(0).size() + firstDataCol;

        //write header
        boolean hasNewHeader = (newHeader != null);
        int startRow = hasNewHeader ? newHeader.length + 1 : 1;
        if (!hasTemplate) {
            if (hasNewHeader) {
                writeHeader(header, rows.get(0).size(), newHeader);
            } else {
                writeHeader(header, rows.get(0).size());
            }
        } else {
            startRow = firstDataRow;
        }
        int n = rows.size();

        //write data rows
        for (List<Object> row1 : rows) {
            Row row = lastSheet.createRow(startRow);
            for (int j = firstDataCol; j < cCount; j++) {
                Object value = row1.get(j - firstDataCol);
                Cell cell = row.createCell(j);
                // 设置数据部分
                if (value != null) {
                    if (value instanceof Boolean) {
                        cell.setCellValue((Boolean) value ? "是" : "否");
                    }else if (value instanceof BigDecimal) {
                        cell.setCellValue(((BigDecimal) value).doubleValue());
                    }else if (value instanceof Date) {
                        cell.setCellValue(value + "");
                    } else {
                        cell.setCellValue(value + "");
                    }
                }
            }
            startRow++;
        }

        if (footer == null) {
            return;
        }
        //write the footer
        for (int i = 0; i < footer.length; i++) {
            Row row = lastSheet.createRow(startRow);
            for (int j = 0; j < footer[i].length; j++) {
                String value = footer[i][j];
                Cell cell = row.createCell(j + firstDataCol);
                cell.setCellValue(value == null ? "" : value + "");
                lastSheet.setColumnWidth(j, (value + "").getBytes().length * 256);
            }
            startRow++;
        }
    }

    /**
     * save the workbook
     */
    public void saveBook() throws Exception {
        //delete the template sheet and wrap it up
        if (hasTemplate) {
            theBook.removeSheetAt(0);
            for (int i = 1; i <= theBook.getNumberOfSheets(); i++) {
                theBook.setSheetName(i - 1, "工作表" + i);
            }
        }
        String fileName = handleFileName();
        theBook.write(new FileOutputStream(fileName));
        theBook.close();
    }

    /**
     * new write header
     *
     * @param header
     * @param cCount
     */
    private void writeHeader(String[] header, int cCount, String[][] newHeader) {
        // 添加新的自定义表头
        int n = newHeader.length;
        for (int i = 0; i < n; i++) {
            Row row = lastSheet.createRow(i);
            for (int j = 0; j < newHeader[i].length; j++) {
                String value = newHeader[i][j];
                Cell cell = row.createCell(j);
                cell.setCellValue(value == null ? "" : value + "");
                lastSheet.setColumnWidth(j, (value + "").getBytes().length * 256);
            }
        }
        Row row = lastSheet.createRow(n);
        for (int j = 0; j < cCount; j++) {
            Object value = header[j];
            Cell cell = row.createCell(j);
            // 设置数据部分
            if (value != null) {
                cell.setCellValue(value + "");
                lastSheet.setColumnWidth(j, (value + "").getBytes().length * 256);
            }
        }
    }

    /**
     * write header
     *
     * @param header
     * @param cCount
     */
    private void writeHeader(String[] header, int cCount) {
        Row row = lastSheet.createRow(0);
        for (int j = 0; j < cCount; j++) {
            Object value = header[j];
            Cell cell = row.createCell(j);
            // 设置数据部分
            if (value != null) {
                cell.setCellValue(value + "");
                lastSheet.setColumnWidth(j, (value + "").getBytes().length * 256);
            }
        }
    }

    private static void checkFile(InputStream is) throws IOException {
        // 判断文件是否存在
        if (null == is) {
            logger.error("文件不存在！");
            throw new FileNotFoundException("文件不存在！");
        }
    }

    private static Workbook getWorkBook(InputStream is, String fileName) {
        //创建Workbook工作薄对象，表示整个excel
        Workbook workbook = null;
        try {
            // 获取excel文件的io流
            //InputStream is = file.getInputStream();
            // 根据文件后缀名不同(xls和xlsx)获得不同的Workbook实现类对象
            if (fileName.endsWith(XLS)) {
                // 2003
                workbook = new HSSFWorkbook(is);
            } else if (fileName.endsWith(XLSX)) {
                // 2007
                workbook = new XSSFWorkbook(is);
            }
        } catch (IOException e) {
            logger.info(e.getMessage());
        }
        return workbook;
    }

    private static String getCellValue(Cell cell) {
        String cellValue = "";
        if (cell == null) {
            return cellValue;
        }
        // 把数字当成String来读，避免出现1读成1.0的情况
        if (cell.getCellType() == Cell.CELL_TYPE_NUMERIC) {
            cell.setCellType(Cell.CELL_TYPE_STRING);
        }
        // 判断数据的类型
        switch (cell.getCellType()) {
            case Cell.CELL_TYPE_NUMERIC: // 数字
                cellValue = String.valueOf(cell.getNumericCellValue());
                break;
            case Cell.CELL_TYPE_STRING: // 字符串
                cellValue = String.valueOf(cell.getStringCellValue());
                break;
            case Cell.CELL_TYPE_BOOLEAN: // Boolean
                cellValue = String.valueOf(cell.getBooleanCellValue());
                break;
            case Cell.CELL_TYPE_FORMULA: // 公式
                cellValue = String.valueOf(cell.getCellFormula());
                break;
            case Cell.CELL_TYPE_BLANK: // 空值
                cellValue = "";
                break;
            case Cell.CELL_TYPE_ERROR: // 故障
                cellValue = "非法字符";
                break;
            default:
                cellValue = "未知类型";
                break;
        }
        return cellValue;
    }

    public static Integer getStrType(String value) {
        if (StringUtils.isBlank(value)) {
            return null;
        }
        //判断是否是整数包括正整数和负整数
        if (value.matches("\\-?[1-9]\\d+")) {
            return 1;//整数
        }
        //判断是否是正负小数
        if (value.matches("\\-?[1-9]\\d+(\\.\\d+)?")) {
            return 2;//数值类型
        }
        //判断是否是百分数
        if (value.matches("\\-?[1-9]\\d+(\\.\\d+)?%")) {
            return 3;//百分数
        }
        return 0;//其他类型
    }

    /**
     * 数据导出到excel
     * @param data   表格数据
     * @param header 表头行
     * @param outSrc 表名称以及路径
     * @return
     */
    public static boolean dataToExcel(String[] header, List<List<String>> data, String outSrc) {
        //校验数据  暂时不做

        // 创建excel
        HSSFWorkbook wk = new HSSFWorkbook();
        // 创建一张工作表
        HSSFSheet sheet = wk.createSheet();
        // 设置列宽
        sheet.setColumnWidth(0, 5000);
        //创建表头
        HSSFRow row = sheet.createRow(0);
        HSSFCell cell = null;
        for (int i = 0; i < header.length; i++) {
            cell = row.createCell(i);
            cell.setCellValue(header[i]);
        }
        //导数据
        for (int i = 0; i < data.size(); i++) {
            row = sheet.createRow(i + 1);
            List<String> rowData = data.get(i);
            for (int i1 = 0; i1 < rowData.size(); i1++) {
                cell = row.createCell(i1);
                switch (ExcelUtil.getStrType(rowData.get(i1))) {
                    case 1:
                        cell.setCellValue(Integer.parseInt(rowData.get(i1)));
                        break;
                    case 2:
                        cell.setCellValue(Double.parseDouble(rowData.get(i1)));
                        break;
                    default:
                        cell.setCellValue(rowData.get(i1));
                }
            }
        }
        try {
            //保存工作簿
            wk.write(new FileOutputStream(new File(outSrc)));
            // 关闭工作薄
            wk.close();
        } catch (IOException e) {
            e.printStackTrace();
            return false;
        }
        return true;
    }
}