package com.aote.rs;

import com.af.plugins.AESUtil;
import com.af.plugins.RSAUtil;
import com.aote.exception.ExceptionLogManage;
import com.aote.logic.LogicMapper;
import com.aote.logic.LogicServer;
import com.aote.rs.mapper.WebException;
import com.aote.transaction.SessionPool;
import com.aote.util.*;
import javassist.NotFoundException;
import net.sf.json.JSON;
import net.sf.json.xml.XMLSerializer;
import org.apache.log4j.Logger;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.annotation.PostConstruct;
import javax.inject.Singleton;
import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import java.util.Enumeration;
import java.util.Map;

@Path("logic")
@Singleton
@Component
public class LogicService {
	static Logger log = Logger.getLogger(LogicService.class);

	@Autowired
	private SessionPool sessionPool;

	@Autowired
	private LogicServer logicServer;

	@Autowired
	private ExceptionLogManage exceptionLogManage;

	/**
	 * 执行业务逻辑(GET方式)
	 *
	 * @param logicName 业务逻辑名称
	 * @param request   HttpServletRequest对象
	 * @return 执行结果
	 */
	@GET
	@Transactional
	@Path("{logic}")
	public String xtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) throws Exception {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		return xtSave(logicName, params.toString(), request);
	}

	/**
	 * 异步执行业务逻辑(GET方式)
	 * https://stackoverflow.com/questions/31401640/spring-xml-equivalent-of-enableasync
	 *
	 * @param logicName 业务逻辑名称
	 * @param request   HttpServletRequest对象
	 * @return 执行结果
	 */
	@GET
	@Transactional
	@Async("AsyncLogicGet")
	@Path("async/{logic}")
	public String asyncXtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) throws Exception {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		return xtSave(logicName, params.toString(), request);
	}

	/**
	 * 异步执行业务逻辑(POST方式)
	 * https://stackoverflow.com/questions/31401640/spring-xml-equivalent-of-enableasync
	 *
	 * @param logicName 业务逻辑名称
	 * @param values JSON资源字符串
	 */
	@POST
	@Async("AsyncLogicPost")
	@Transactional
	@Path("async/{logic}")
	public void asyncXtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request)
			throws Exception {
		xtSave(logicName,values,request);
	}

	/**
	 * 执行调试用的业务逻辑(GET方式)，自动回滚
	 *
	 * @param logicName 业务逻辑名称
	 * @param request   HttpServletRequest对象
	 * @return 执行结果
	 */
	@GET
	@Transactional
	@Path("debug/{logic}")
	public String debugXtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) throws Exception {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		return debugXtSave(logicName, params.toString(),request);
	}

	/**
	 * 执行调试用的业务逻辑(POST方式)，自动回滚
	 *
	 * @param logicName 业务逻辑名称
	 * @param values JSON资源字符串
	 * @return 执行结果
	 */
	@POST
	@Transactional
	@Path("debug/{logic}")
	public String debugXtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request)
			throws Exception {
		String result = xtSave(logicName,values,request);
		//事务回滚
		TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
		return result;
	}

	@Transactional
	public String xtSave(String logicName, String values) throws Exception {
		return xtSave(logicName,values,null);
	}

	/**
	 * 运行业务逻辑
	 * @param logicName 业务逻辑名
	 * @param values 业务逻辑参数
	 * @return 运行结果
	 */
	private String run(String logicName, String values) throws Exception {
		try {
			//如果有action，执行action处理过程，返回sessionid
			JSONObject jsonObject = new JSONObject(values);
			if (jsonObject.has("action")){
				int id = judgeAction(jsonObject);
				return id + "";
			}

			Object result = logicServer.run(logicName, values);
			return resultToString(result);
		} catch (Exception ex) {
			log.error(ExceptionHelper.stackToString(ex));
			WebException wex = this.getWebException(ex);
			if (wex != null)
			{
				throw wex;
			}
			throw ex;
		}
	}

	@POST
	@Path("noTransc/{logic}")
	public String save(@PathParam("logic") String logicName, String values)
			throws Exception {
		log.info("logic:" + logicName + ", values:" + values);
		try {
			Object result = logicServer.noTranscRun(logicName, values);
			if (result == null) {
                return "";
			}
			// 如果执行结果为Map，转换成JSON串
			if (result instanceof Map<?, ?>) {
				Map<String, Object> map = (Map<String, Object>)result;
				JSONObject json = (JSONObject) new JsonTransfer().MapToJson(map);
				if (!json.isNull("xml")) {
					return (String) json.get("xml");
				}
				return json.toString();
			}
			if (result instanceof JSONObject) {
				JSONObject json = (JSONObject) result;
				if (!json.isNull("xml")) {
					return json.getString("xml");
				}
				return json.toString();
			}
			return result.toString();
		} catch (Exception ex) {
			String stack = ExceptionLogManage.getLogicErrorInfo(ExceptionHelper.stackToString(ex));
			log.error(stack);
			// 获取源程序路径
			String path = LogicMapper.getLogic(logicName);
			if(path == null){
				path = logicName;
			}
			exceptionLogManage.writeError(ex,path,"/rs/logic/"+logicName);
			WebException wex = this.getWebException(ex);
			if (wex != null)
			{
				throw wex;
			}
			throw ex;
		}
	}

	@POST
	@Path("xml/{logic}")
	@Consumes({"application/xml"})
	public String xmlXtSave(@PathParam("logic") String logicName, @Context  HttpServletRequest request) throws Exception {

		ServletInputStream inputStream = request.getInputStream();
		StringBuilder content = new StringBuilder();
		byte[] b = new byte[1024];
		int lens;
		while ((lens = inputStream.read(b)) > 0) {
			content.append(new String(b, 0, lens));
		}
		String strcont = content.toString();// 内容
		JSON json;
		try {
			XMLSerializer xmlSerializer = new XMLSerializer();
			json = xmlSerializer.read(strcont);
		}catch (Exception e){
			throw new RuntimeException("传入的数据不是XML格式");
		}
		String jsonString = new JSONObject().put("data",json).toString();
		log.debug("调用XML接口，Logic："+logicName+"  参数："+json);
		return run(logicName,jsonString);
	}

	/**
	 * 执行业务逻辑(POST方式)
	 *
	 * @param logicName 业务逻辑名称
	 * @param values JSON资源字符串
	 * @return 执行结果
	 */
	@POST
	@Transactional
	@Path("{logic}")
	public String xtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request)
			throws Exception {

		JSONObject paramJson;
		log.info("logic:" + logicName + ", values:" + values);
		// 如果内容以"$"开始，说明传过来的是加密数据
		if (values.startsWith("$")) {
			values = RSAUtil.descrypt(values.substring(1));
			paramJson = new JSONObject(values);
		}else if(!values.startsWith("{")){
			paramJson = new JSONObject();
			paramJson.put("str",values);
			//加入请求头数据
			if(request != null){
				Enumeration headerNames = request.getHeaderNames();
				if(headerNames != null){
					JSONObject header = new JSONObject();
					while (headerNames.hasMoreElements()){
						String temp = headerNames.nextElement().toString();
						header.put(temp, request.getHeader(temp));
					}
					paramJson.put("header",header);
				}
				values = paramJson.toString();
			}
		} else {
			paramJson = new JSONObject(values);
		}

		JSONObject configJson = JsonHelper.getConfig();
		// 加密验证
		Util.checkEncry(paramJson, configJson);
		//sql注入标志位
		boolean sqlInject=false;
		if(configJson.has("sqlInject")){
			sqlInject = configJson.getBoolean("sqlInject");
		}
		//如果不需要加密，进入default正常执行
		if (!paramJson.has("aoteEncrypt")){
			paramJson.put("aoteEncrypt", "default");
		}
		//判断解密方式
		if ("AES".equals(paramJson.get("aoteEncrypt").toString())) {// 数据解密
			String data = paramJson.getString("data");
			String encryKey = configJson.getString("encryKey");
			String decryptData = AESUtil.decrypt(data, encryKey);
			//验证参数是否合法
			if (sqlInject) {
				ParamFilter.checkSqlJsonStr(decryptData);
			}
			return AESUtil.encrypt(run(logicName, decryptData), encryKey);
		}//验证参数是否合法
		if (sqlInject) {
			ParamFilter.checkSqlJsonStr(values);
		}
		return run(logicName, values);
	}


	//服务器启动时加载模块至redis
	@PostConstruct
	public void redisCacheLogic() throws Exception {
		logicServer.redisCacheLogic();
	}

	// 找到WebException异常
	// return - 不存在，返回null
	private WebException getWebException(Exception ex)
	{
		Throwable cause = ex;
		while (cause != null)
		{
			if (cause instanceof WebException)
			{
				return (WebException)cause;
			}
			cause = cause.getCause();
		}

		return null;
	}

	/**
	 * 执行action中的动作
	 * @param jsonObject {action: 'actionname', id: 'sessionid'}
	 * @return 启动事务的id
	 * @throws NotFoundException
	 */
	private int judgeAction(JSONObject jsonObject)throws NotFoundException {
		int sessionid;
		//不为空则，获取action，并判断action请求为何种类型
		String actObj = (String)jsonObject.get("action");
		if("begin".equals(actObj)) {
			//如果为begin类型，openSession，并获取sessionid；
			sessionid = sessionPool.createSession();
			return sessionid;
		} else {
			//否则直接获取id
			sessionid = jsonObject.getInt("id");
		}
		return sessionid;
	}


	/**
	 * 转换执行结果
	 */
	private String resultToString(Object result) throws Exception {
		if (result == null) {
			return "";
		}
		// 如果执行结果为Map，转换成JSON串
		if (result instanceof Map<?, ?>) {
			Map<String, Object> map = (Map<String, Object>)result;
			JSONObject json = (JSONObject) new JsonTransfer().MapToJson(map);
			if (!json.isNull(" xml")) {
				String jsStr = (String) json.get("xml");
				return (String) json.get("xml");
			}
			return json.toString();
		}
		if (result instanceof JSONObject) {
			JSONObject json = (JSONObject) result;
			if (!json.isNull("xml")) {
				String JSONstr = json.getString("xml");
				return json.getString("xml");
			}
			return json.toString();
		}
		return result.toString();
	}
}
