package com.aote.transaction;

import com.af.plugins.RestTools;
import com.aote.ThreadResource;
import javassist.NotFoundException;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * session池，用于自己对hibernate的session进行管理，特别是前台控制事务的情况。
 */
@Component
public class SessionPool {
    @Autowired
    private SessionFactory factory;

    @Autowired
    private ThreadSession tSession;

    // 缓存的session，包括session及其事务
    private InnerSession[] sessions = new InnerSession[100];

    public static SessionPool getInstance() {
        return instance;
    }

    private static SessionPool instance;

    private SessionPool() {
        instance = this;
    }

    /**
     *  创建session，把新建的session添加到缓存中
     * @return
     */
    public int createSession() throws NotFoundException {
        Session session = this.factory.openSession();
        Transaction trans = session.beginTransaction();
        InnerSession innerSession = new InnerSession(session, trans);

        // 找到空闲位置，把session添加进去
        synchronized (this.sessions) {
            for(int i = 0; i < sessions.length; i++) {
                if (sessions[i] == null) {
                    sessions[i] = innerSession;
                    return i;
                }
            }
        }

        throw new NotFoundException("没有空闲位置");
    }

    /**
     * 创建后继session
     * @param i 要创建后继session的sessionid
     * @param url 要创建的后继session的url地址
     */
    public int createCallSession(int i, String url) {
        return this.sessions[i].createCallSession(url);
    }

    /**
     * 获取第i项session
     * @param i 第i项session
     * @return session
     */
    public Session getSession(int i) {
        return this.sessions[i].session;
    }

    /**
     * 提交第i项session，把session从缓存中移除
     * @param i 第i项session
     */
    public void commit(int i) {
        InnerSession session = this.sessions[i];
        session.transaction.commit();
        // 提交后继事务
        session.commitAll();
        synchronized (this.sessions) {
            this.sessions[i] = null;
        }
    }

    /**
     * 回滚第i项session，把session从缓存中移除
     * @param i
     */
    public void rollback(int i) {
        InnerSession session = this.sessions[i];
        session.transaction.rollback();
        // 回滚后继事务
        session.rollbackAll();
        synchronized (this.sessions) {
            this.sessions[i] = null;
        }
    }

    /**
     * 保存session及事务，方便事务提交或回滚
     */
    private class InnerSession {
        public Session session;
        public Transaction transaction;

        // 调用的session列表
        private List<CallSession> chanSessions = new ArrayList<>();

        public InnerSession(Session session, Transaction transaction) {
            this.session = session;
            this.transaction = transaction;
        }

        // 产生后继事务
        public int createCallSession(String url) {
            // 发送申请事务请求
            String id = RestTools.post(url, "{action:'begin'}");
            int result = Integer.parseInt(id);
            CallSession callSession = new CallSession(url, result);
            // 保存事务
            this.chanSessions.add(callSession);

            return result;
        }

        // 提交所有后继事务
        public void commitAll() {
            for(CallSession callSession : chanSessions) {
                // 发送提交事务的请求
                RestTools.post(callSession.url,  "{action:'commit', id: "+ callSession.id + "}");
            }
        }

        // 回滚所有后继事务
        public void rollbackAll() {
            for(CallSession callSession : chanSessions) {
                // 发送回滚事务的请求
                RestTools.post(callSession.url,  "{action:'rollback', id: "+ callSession.id + "}");
            }
        }
    }

    /**
     * 获得当前线程session，如果线程局部变量有sessionid，从sessionpool中取，否则，从sessionfactory中取。
     * @return
     */
    public Session getSession() {
        //如果线程中有session，直接取线程中的
        Session threadSession = tSession.get();
        if(threadSession != null){
            return threadSession;
        }
        // 如果请求头有session，根据请求头里的sessionid获得对应session，放到事务管理器中
        Integer sessionId = ThreadResource.SessionId.get();
        if (sessionId != null) {
            int i = sessionId;
            return this.getSession(i);
        }
        return factory.getCurrentSession();
    }
    /**
     * 后续调用的session
     */
    private class CallSession {

        // 访问路径
        public String url;
        // 访问的sessionid
        public int id;

        public CallSession(String url, int id) {
            this.url = url;
            this.id = id;
        }
    }
}
