package com.aote.redis;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.Map;
import java.util.Set;

/**
 * Redis业务接口
 */
public interface RedisService {

    /**
     * 判断是否含有该key的数据
     * @param key 数据key
     */
    Boolean has(String key);

    /**
     * 存储一条key/value数据
     * @param key 数据key
     * @param value 数据value
     */
    void set(String key, Object value);

    /**
     * 存储一条key/value数据（带过期时间）
     * @param key 数据key
     * @param value 数据value
     * @param timeoutSec 过期时间
     */
    void set(String key, Object value, long timeoutSec);

    /**
     * 存储一条hash数据
     * @param hashKey hashKey
     * @param value jsonObject对象
     */
    void setHash(String hashKey, JSONObject value);

    /**
     * 存储一条hash数据
     * @param hashKey hashKey
     * @param value map对象
     */
    void setHash(String hashKey, Map<String, Object> value);

    /**
     * 获取一条json格式的hash数据
     * @param hashKey hashKey
     * @return jsonObject对象
     */
    JSONObject getHash(String hashKey);

    /**
     * 获取指定hash中key的数据
     * @param hashKey hashKey
     * @param key 数据key
     * @return 数据value
     */
    Object getHash(String hashKey,String key);

    /**
     * 向指定hash中存储一条key/value数据
     * @param hashKey hashKey
     * @param key 数据key
     * @param value 数据value
     */
    void setHashKey(String hashKey,String key,Object value);

    /**
     * 删除指定hash中key的数据
     * @param hashKey hashKey
     * @param key 数据key
     */
    void deleteHashKey(String hashKey,String key);

    /**
     * 判断指定hash中是否含有该key的数据
     * @param hashKey hashKey
     * @param key 数据key
     */
    Boolean hasHashKey(String hashKey,String key);

    /**
     * 获取一条key/value数据
     * @param key 数据key
     * @return 数据value
     */
    Object get(String key);

    /**
     * 删除一条key/value数据
     * @param key 数据key
     */
    void delete(String key);

    /**
     * 批量删除指定key集合的数据
     * @param keys 数据key集合
     */
    void deleteList(Set<String> keys);
    void deleteList(JSONArray keys);
    void deleteList(JSONObject keys);

    /**
     * 获取锁
     * @param key 需要获取锁的key
     * @param timeoutSec 超时时间，单位：秒，超过此时间后锁自动释放
     * @return 是否成功获取锁
     */
    Boolean lock(String key, long timeoutSec);

    /**
     * 获取hash锁
     * @param hashKey 需要获取锁的hashKey
     * @param key 需要获取锁的key
     * @param timeoutSec 超时时间，单位：秒，超过此时间后锁自动释放
     * @return 是否成功获取锁
     */
    Boolean hashLock(String hashKey, String key, long timeoutSec);

    /**
     * 释放锁
     * @param key 需要释放锁的key
     */
    void unLock(String key);

    void hashUnLock(String hashKey, String key);

    /**
     * 重命名数据key的名称
     *
     * @param oldKey 旧key名称
     * @param newKey 新key名称
     */
    void rename(String oldKey, String newKey);
}