package com.aote.excel.core;

import cn.hutool.core.util.StrUtil;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.excel.exception.ExcelAnalysisException;
import com.alibaba.excel.exception.ExcelCommonException;
import com.alibaba.excel.exception.ExcelDataConvertException;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import static cn.hutool.core.lang.Console.log;

/**
 * 导入监听
 *
 * @author Eraser
 */
public class DefautExcelListener<T> extends AnalysisEventListener<T> implements ExcelListener<T> {
    /**
     * 配置文件表头备注-字段数据
     */
    private final Map<String, Object> columMap;
    /**
     * 导入回执
     */
    private final ExcelResult<T> excelResult;
    /**
     * 是否Validator检验，默认为是
     */
    private final Boolean isValidate;
    /**
     * 表头数据（存储所有的表头数据）
     */
    private Map<Integer, String> headMap;

    public DefautExcelListener(boolean isValidate, Map<String, Object> columMap) {
        this.excelResult = new DefautExcelResult<>();
        this.isValidate = isValidate;
        this.columMap = columMap;

    }

    /**
     * 这里会一行行的返回头
     */
    @Override
    public void invokeHeadMap(Map<Integer, String> headMap, AnalysisContext context) {
        //存储全部表头数据
        this.headMap = headMap;
        //验证导入表头与系统表头是否对应
        Set<String> headList = columMap.keySet();
        for (String value : headMap.values()) {
            if (!headList.contains(value)) {
                throw new ExcelCommonException("表头数据无法匹配");
            }
        }
        log("解析到一条表头数据: {}", headMap.toString());

    }

    /**
     * 处理异常
     *
     * @param exception ExcelDataConvertException
     * @param context   Excel 上下文
     */
    @Override
    public void onException(Exception exception, AnalysisContext context) {
        String errMsg = null;
        if (exception instanceof ExcelDataConvertException) {
            // 如果是某一个单元格的转换异常 能获取到具体行号
            Integer rowIndex = ((ExcelDataConvertException)exception).getRowIndex();
            Integer columnIndex = ((ExcelDataConvertException)exception).getColumnIndex();
            errMsg = StrUtil.format("第{}行-第{}列-表头{}: 解析异常<br/>",
                    rowIndex + 1, columnIndex + 1, headMap.get(columnIndex));
        }
        if (exception instanceof ExcelCommonException) {
            errMsg = exception.getMessage();
        }
        excelResult.getErrorList().add(errMsg);
        throw new ExcelAnalysisException(errMsg);
    }

    /**
     * 这个每一条数据解析都会来调用
     */
    @Override
    public void invoke(T data, AnalysisContext context) {
        Map<Integer, Object> map = (Map<Integer, Object>) data;
        if (isValidate) {
            //
            if (map.size() > columMap.size()) {
                excelResult.getErrorList().add("表格内容数据长度超出表头长度!");
                throw new ExcelCommonException("表格内容数据长度超出表头长度!");
            }
        }
        Map<String, Object> dataMap = new HashMap<>(map.size());
        map.forEach((k, v) -> dataMap.put((columMap.get(headMap.get(k))).toString(), v));
        log("解析到一条内容数据: {}", dataMap.toString());
        excelResult.getList().add(dataMap);

    }

    /**
     * 所有数据解析完成了 都会来调用
     */
    @Override
    public void doAfterAllAnalysed(AnalysisContext context) {
        // 这里也要保存数据，确保最后遗留的数据也存储到数据库
        log("所有数据解析完成！");
    }

    public Map<Integer, String> getHeadMap() {
        return headMap;
    }

    @Override
    public ExcelResult<T> getExcelResult() {
        return excelResult;
    }
}


