package com.aote.logic;

import com.alibaba.fastjson.JSON;
import com.aote.config.SystemConfig;
import com.aote.entity.EntityServer;
import com.aote.exception.ExceptionLogManage;
import com.aote.exception.FileNotFoundException;
import com.aote.redis.RedisService;
import com.aote.redis.RedisUtil;
import com.aote.rs.LogicService;
import com.aote.rs.mapper.WebException;
import com.aote.sql.SqlMapper;
import com.aote.sql.SqlServer;
import com.aote.transaction.SessionPool;
import com.aote.util.ExceptionHelper;
import com.aote.util.ExpressionHelper;
import com.aote.util.ResourceHelper;
import com.aote.util.ResourceType;
import org.apache.log4j.Logger;
import org.hibernate.SessionFactory;
import org.hibernate.metadata.ClassMetadata;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicReference;

@Component
@Transactional
public class LogicServer {
	static Logger log = Logger.getLogger(LogicServer.class);

	@Autowired
	private EntityServer entityServer;

	@Autowired
	private SqlServer sqlServer;

	@Autowired
	private SessionPool sessionPool;


	// 执行业务逻辑处理过程
	public Object run(String name, String str) {
		JSONObject param = null;
		// 把传递过来的参数，放到data里，以便跟entity，sql等对象区别开来
		// 判断传来的数据是否是XML格式
		str = str.trim();
		if (str.startsWith("<") && str.endsWith(">")) {
			// 传来数据为XML格式
			param = new JSONObject();
			param.put("xml", str);
		} else {
			// 传来数据为JSON格式
			Object json = new JSONTokener(str).nextValue();
			if (json instanceof JSONObject) {
				param = new JSONObject(str);
				if (!param.isNull("data")) {
					Object dataParam = param.get("data");
					if (dataParam instanceof JSONObject){
						param = (JSONObject) dataParam;
						param.put("standardData",new JSONObject(str).toString());
					}
				}
			} else if (json instanceof JSONArray) {
				param = new JSONObject();
				param.put("arr", new JSONArray(str));
			}
		}

		return run(name, param);
	}

	// 业务逻辑带一个上下问对象，用于平板端测试
	public Object run(String name, JSONObject param) {
		boolean isEnableCache = LogicMapper.isEnableCache(name);
		String message = "维护信息:开始执行Logic["+name+"]，缓存："+isEnableCache+"，参数：" + param;
		log.debug(message);
		if(isEnableCache){
			String paramsStr = new TreeMap<>(param.toMap()).toString();
			RedisUtil redisUtil = RedisUtil.getInstance();
			String cacheKey = "DataCache@" + name + "-" + paramsStr;
			Object result = redisUtil.get(cacheKey);
			if(result != null){
				log.debug("维护信息:Logic["+name+"]命中缓存");
				return result;
			}
			AtomicReference<Object> returnResult = new AtomicReference<>();
			String cacheLockKey = "DataCacheLock@" + name + "-" + paramsStr;
			redisUtil.lock(cacheLockKey, 60, 300, ()->{
				Object newResult = redisUtil.get(cacheKey);
				if(newResult == null){
					newResult = runLogic(name, param);
					if(newResult != null){
						// 默认缓存3小时
						long cacheTime = LogicMapper.getCacheTime(name);
						if(cacheTime > 0) {
							redisUtil.set(cacheKey, newResult, cacheTime);
							log.debug("维护信息:Logic["+name+"]缓存成功，存活时间：" + cacheTime + "秒");
						}
					}
				} else {
					log.debug("维护信息:Logic["+name+"]命中缓存");
				}
				returnResult.set(newResult);
			});
			return returnResult.get();
		}
		return runLogic(name, param);
	}

	private Object runLogic(String name, JSONObject param){
		// 获取源程序内容
		String path = LogicMapper.getLogic(name);
		if (path == null) {
			throw new RuntimeException("业务逻辑未找到: " + name);
		}

		String source;
		try {
			source = (String) ResourceHelper.getString(ResourceType.LOGIC,name,path);
		} catch (FileNotFoundException e) {
			throw new RuntimeException(path + ".文件无配置");
		} catch (IOException io) {
			throw new RuntimeException(io);
		}
		// 执行源程序
		Map<String, Object> params = new HashMap<>();
		params.put("data", param);
		// 附加entityServer, sqlServer等对象到参数中
		params.put("log", log);
		params.put("entity", entityServer);
		params.put("sql", sqlServer);
		params.put("session", sessionPool.getSession());
		params.put("redis",RedisUtil.getInstance());
		// 把LogicServer自身注册进去
		params.put("logic", this);

		// 附加用户注册的对象到业务逻辑中
		Map<String, Object> plugins = PluginMapper.getPlugins();
		params.putAll(plugins);
		// 执行结果
		Object result;
		long begin = System.currentTimeMillis();
		// 记录业务执行流程堆栈
		Long reviewKey = LogicService.reviewKey.get();
		JSONObject reviewItem = new JSONObject();
		if(reviewKey != null) {
			JSONArray reviewArray = LogicService.reviewCache.get(reviewKey);
			if(reviewArray != null) {
				reviewItem.put("bTime", begin);
				reviewItem.put("name", name);
				reviewItem.put("type", "logic");
				reviewArray.put(reviewItem);
			}
		}
		try {
			result = ExpressionHelper.run(source, params);
			reviewItem.put("status", 0);
			return result;
		} catch (Exception e){
			reviewItem.put("status", -1);
			throw e;
		} finally {
			long end = System.currentTimeMillis();
			long time = end - begin;
			String engMessage = "维护信息:执行Logic["+name+"]耗时:" + time;
			if(time >= 8000){
				log.error(engMessage);
			} else if(time >= 4000){
				log.warn(engMessage);
			} else {
				log.debug(engMessage);
			}
			reviewItem.put("eTime", end);
			reviewItem.put("constTime", time);
		}
	}

	/**
	 * web层运行业务逻辑入口
	 * @param logicName 业务逻辑名
	 * @param values 业务逻辑参数
	 * @return 运行结果
	 */
	public String runMain(String logicName, String values) {
		Object result = this.run(logicName, values);
		if (result == null) {
			return "";
		}
		// 如果执行结果为Map，转换成JSON串
		if (result instanceof Map<?, ?>) {
			JSONObject json = new JSONObject((Map<?, ?>) result);
			return json.toString();
		}
		if (result instanceof JSONObject) {
			JSONObject json = (JSONObject) result;
			return json.toString();
		}
		return result.toString();
	}

	@Async("AsyncLogicGet")
	public void runAsync(String logicName, String values){
		runMain(logicName, values);
	}

	/**
	 * redis缓存业务逻辑组件
	 */
	public void redisCacheLogic() {
		if (SystemConfig.Redis.hasHostName() && SystemConfig.Redis.hasRegisterUrl()){
			RedisUtil redisUtil = RedisUtil.getInstance();
			String url = SystemConfig.Redis.getRegisterUrl();
			//获取模块逻辑名称
			Map<String, Map<String, String>> map = LogicMapper.getMap();
			if (map != null) {
				//遍历逻辑集合，将逻辑及对应IP和端口号存入服务器的redis缓存中
				for (String key : map.keySet()) {
					redisUtil.set(key, url);
					log.info("logic加入redis缓存，key=" + key + ", value=" + url);
				}
			}
			// 对sql语句进行注册
			map = SqlMapper.getMap();
			if (map != null) {
				//遍历逻辑集合，将逻辑及对应IP和端口号存入服务器的redis缓存中
				for (String key : map.keySet()) {
					redisUtil.set(key, url);
					log.info("sql加入redis缓存，key=" + key + ", value=" + url);
				}
			}
		}
	}
}
