package com.aote.module;

import com.af.expression.Program;
import com.aote.exception.FileNotFoundException;
import com.aote.logic.LogicMapper;
import com.aote.sql.SqlMapper;
import com.aote.util.ResourceHelper;
import com.aote.util.ResourceType;
import org.apache.log4j.Logger;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 业务资源检测工具
 *
 * @author Mrriver
 * @date 2020/6/23
 */
@Component
public class ResCheck {

    private static final Logger LOGGER = Logger.getLogger(ResCheck.class);

    private static final List<String> ERROR_ARRAY = new ArrayList<>();

    @PostConstruct
    public void checkAll(){
        // 获取所有引入的表具模块名
        Map<String, Map<String, String>> map = ModuleMapper.getMap();
        int GrammarErrorNums = 0;
        int GrammarErrorSQLNums = 0;
        for (String module : map.keySet()) {
            String modulePath = map.get(module).get("path");
            String logicUrl = (modulePath == null ? module + "/logic.xml" : modulePath + "/"
                    + module + "/logic.xml");
            String sqlUrl = (modulePath == null ? module + "/sql.xml" : modulePath + "/"
                    + module + "/sql.xml");
            //语法检测
            GrammarErrorNums = GrammarErrorNums + checkLogicGrammar(module, logicUrl);
            GrammarErrorSQLNums = GrammarErrorSQLNums + checkSqlResource(module, sqlUrl);
        }
        boolean throwError = false;
        String msg = "";
        if(GrammarErrorNums > 0){
            throwError = true;
            msg = "Logic强制检查未通过，共有"+ GrammarErrorNums+"项错误，如果您在查看tomcat日志，请转到应用日志查看错误详情";
        }
        if(GrammarErrorSQLNums > 0){
            throwError = true;
            if(!msg.equals("")){
                msg = msg + "\n";
            }
            msg = msg + "SQL强制检查未通过，共有"+ GrammarErrorSQLNums+"项错误，如果您在查看tomcat日志，请转到应用日志查看错误详情";
        }
        if(throwError){
            StringBuilder errorWelcome = new StringBuilder("\n/***\n" +
                    " * _ooOoo_\n" +
                    " * o8888888o\n" +
                    " * 88\" . \"88\n" +
                    " * (| -_- |)\n" +
                    " *  O\\ = /O\n" +
                    " * ___/`---'\\____\n" +
                    " * .   ' \\\\| |// `.\n" +
                    " * / \\\\||| : |||// \\\n" +
                    " * / _||||| -:- |||||- \\\n" +
                    " * | | \\\\\\ - /// | |\n" +
                    " * | \\_| ''\\---/'' | |\n" +
                    " * \\ .-\\__ `-` ___/-. /\n" +
                    " * ___`. .' /--.--\\ `. . __\n" +
                    " * .\"\" '< `.___\\_<|>_/___.' >'\"\".\n" +
                    " * | | : `- \\`.;`\\ _ /`;.`/ - ` : | |\n" +
                    " * \\ \\ `-. \\_ __\\ /__ _/ .-` / /\n" +
                    " * ======`-.____`-.___\\_____/___.-`____.-'======\n" +
                    " * `=---='\n" +
                    " * .............................................\n" +
                    " * 佛曰：bug泛滥，我已瘫痪！\n" +
                    " * " + msg + "\n");
            for(int i = 0; i < ERROR_ARRAY.size();i++){
                errorWelcome.append(" * ").append(i).append(". ").append(ERROR_ARRAY.get(i)).append("\n");
            }
            errorWelcome.append(" */");
            LOGGER.error(errorWelcome);
            throw new RuntimeException(msg);
        } else {
            String successWelcome = "\n/***\n" +
                " *                    _ooOoo_\n" +
                " *                   o8888888o\n" +
                " *                   88\" . \"88\n" +
                " *                   (| -_- |)\n" +
                " *                    O\\ = /O\n" +
                " *                ____/`---'\\____\n" +
                " *              .   ' \\\\| |// `.\n" +
                " *               / \\\\||| : |||// \\\n" +
                " *             / _||||| -:- |||||- \\\n" +
                " *               | | \\\\\\ - /// | |\n" +
                " *             | \\_| ''\\---/'' | |\n" +
                " *              \\ .-\\__ `-` ___/-. /\n" +
                " *           ___`. .' /--.--\\ `. . __\n" +
                " *        .\"\" '< `.___\\_<|>_/___.' >'\"\".\n" +
                " *       | | : `- \\`.;`\\ _ /`;.`/ - ` : | |\n" +
                " *         \\ \\ `-. \\_ __\\ /__ _/ .-` / /\n" +
                " * ======`-.____`-.___\\_____/___.-`____.-'======\n" +
                " *                    `=---='\n" +
                " *\n" +
                " * .............................................\n" +
                " *          佛祖保佑             永无BUG\n" +
                " */";
            LOGGER.info(successWelcome);
        }
    }

    /**
     * Sql资源映射检测
     */
    private int checkSqlResource(String module, String sqlUrl) {
        InputStream input = LogicMapper.class.getClassLoader()
                .getResourceAsStream(sqlUrl);
        int errorNums = 0;
        if (input != null) {
            SAXReader reader = new SAXReader();
            Document document = null;
            try {
                document = reader.read(input);
            } catch (DocumentException e) {
                e.printStackTrace();
            }
            assert document != null;
            Element root = document.getRootElement();
            List<Element> sqls = root.elements("sql");
            LOGGER.debug("检查【" + module + "】模块下Sql资源映射，共" + sqls.size() + "个...");
            AtomicInteger i = new AtomicInteger(1);
            AtomicInteger success = new AtomicInteger(0);
            AtomicInteger error = new AtomicInteger(0);
            sqls.forEach((it) -> {
                String index = "(" + i + "/" + sqls.size() + ")";
                String alias = it.attribute("alias").getValue();
                // 获取源程序内容
                String path = SqlMapper.getSql(alias);
                if (path == null) {
                    ERROR_ARRAY.add("Sql【" + alias + "】未配置" + index);
                    error.set(error.get() + 1);
                } else {
                    try {
                        ResourceHelper.getString(ResourceType.SQL, alias, path);
                        success.set(success.get() + 1);
                    } catch (FileNotFoundException e) {
                        ERROR_ARRAY.add("Sql【" + alias + "】未找到文件：" + path);
                        error.set(error.get() + 1);
                    } catch (IOException io) {
                        ERROR_ARRAY.add("Sql【" + alias + "】IO异常：" + path);
                        error.set(error.get() + 1);
                    }
                }
                i.set(i.get() + 1);
            });
            LOGGER.debug("【" + module + "】模块语法检查完成，共" + sqls.size() + "个,正确" + success.get() + "个,错误" + error.get() + "个");
            errorNums = errorNums + error.get();
        }
        return errorNums;
    }

    /**
     * Logic语法检测
     */
    private int checkLogicGrammar(String module, String logicUrl) {
        InputStream input = LogicMapper.class.getClassLoader()
                .getResourceAsStream(logicUrl);
        int errorNums = 0;
        if (input != null) {
            SAXReader reader = new SAXReader();
            Document document = null;
            try {
                document = reader.read(input);
            } catch (DocumentException e) {
                e.printStackTrace();
            }
            assert document != null;
            Element root = document.getRootElement();
            List<Element> logics = root.elements("logic");
            LOGGER.debug("检查【"+module+"】模块下Logic语法，共"+logics.size()+"个...");
            AtomicInteger i = new AtomicInteger(1);
            AtomicInteger success = new AtomicInteger(0);
            AtomicInteger error = new AtomicInteger(0);
            ResourceHelper.CACHE_MODE metaMode = ResourceHelper.mode;
            ResourceHelper.mode = ResourceHelper.CACHE_MODE.DEBUG;
            logics.forEach((it) -> {
                String index = "("+i+"/"+logics.size()+")";
                String alias = it.attribute("alias").getValue();
                // 获取源程序内容
                String path = LogicMapper.getLogic(alias);
                if (path == null) {
                    ERROR_ARRAY.add("Logic【"+alias+"】未配置"+index);
                    error.set(error.get() + 1);
                } else {
                    String source;
                    try {
                        source = (String) ResourceHelper.getString(ResourceType.LOGIC, alias, path);
                        Program prog = new Program(source);
                        // 解析
                        try {
                            prog.parse();
                            success.set(success.get() + 1);
                        } catch (RuntimeException e){
                            String message = e.getMessage();
                            ERROR_ARRAY.add("Logic【"+alias+"】编译错误："+message);
                            error.set(error.get() + 1);
                        }
                    } catch (FileNotFoundException e) {
                        ERROR_ARRAY.add("Logic【"+alias+"】未找到文件："+path);
                        error.set(error.get() + 1);
                    } catch (IOException io) {
                        ERROR_ARRAY.add("Logic【"+alias+"】IO异常："+path);
                        error.set(error.get() + 1);
                    }
                }
                i.set(i.get() + 1);
            });
            ResourceHelper.mode = metaMode;
            LOGGER.debug("【"+module+"】模块语法检查完成，共"+logics.size()+"个,正确"+success.get()+"个,错误"+error.get()+"个");
            errorNums = errorNums + error.get();
        }
        return errorNums;
    }
}
