package com.aote.util;

import cn.hutool.cache.Cache;
import cn.hutool.cache.CacheUtil;
import com.aote.exception.FileNotFoundException;
import com.aote.queryparams.QueryParamsMapper;
import com.aote.redis.RedisUtil;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;

@Component
public class ResourceHelper {

	public enum CACHE_MODE {
		//正常模式
		NORMAL,
		//DEBUG模式，不开启缓存
		DEBUG
	}

	public static CACHE_MODE mode = CACHE_MODE.NORMAL;

	//Logic源缓存
	private static final Cache<String, String> LOGIC_CACHE = CacheUtil.newLRUCache(100);
	//SQL源缓存
	private static final Cache<String, String> SQL_CACHE = CacheUtil.newLRUCache(100);

	@Autowired
	private QueryParamsMapper queryParamsMapper;

	/**
	 * 获取资源文件里，指定全路径文件的文本内容
	 *
	 * @param fileName
	 *            : 以'/'开始的资源文件的全路径文件名
	 * @return 文件的文本内容
	 */
	public static String getString(String fileName) throws FileNotFoundException,IOException {
		String result;
		InputStream inputStream = null;
		BufferedReader br = null;
		try {
			// 没有/,加一个/开头
			if (!fileName.startsWith("/")) {
				fileName = "/" + fileName;
			}
			inputStream = ResourceHelper.class.getResourceAsStream(fileName);
			if(null == inputStream){
				throw new FileNotFoundException(fileName + "文件不存在", fileName);
			}
			br = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));

			String line;
			StringBuilder sb = new StringBuilder();
			while ((line=br.readLine())!=null) {
				sb.append(line).append("\r\n");
			}
			result = sb.toString();
		} finally {
			if(null != inputStream){
				inputStream.close();
			}
			if(br != null) {
				br.close();
			}
		}
		return result;
	}

	/**
	 * 获取资源文件里，指定全路径文件的文本内容，并缓存
	 *
	 * @param fileName
	 *            : 以'/'开始的资源文件的全路径文件名
	 * @return 文件的文本内容
	 */
	public static Object getString(ResourceType resourceType, String alias, String fileName) throws FileNotFoundException,IOException {
		String sourceStr = null;
		switch (resourceType){
			case LOGIC:
				if(mode == CACHE_MODE.NORMAL){
					sourceStr = LOGIC_CACHE.get(alias);
				}
				if(sourceStr == null){
					sourceStr = getString(fileName);
					// 处理回车换行
					sourceStr = sourceStr.replace("\r\n", "\n");
					if(mode == CACHE_MODE.NORMAL){
						LOGIC_CACHE.put(alias, sourceStr);
					}
				}
				return sourceStr;
			case SQL:
				if(mode == CACHE_MODE.NORMAL){
					sourceStr = SQL_CACHE.get(alias);
				}
				if(sourceStr == null){
					sourceStr = getString(fileName);
					// 获取编译后的sql语句
					sourceStr = "$" + sourceStr.replace("\r\n", "\n");
					if(mode == CACHE_MODE.NORMAL){
						SQL_CACHE.put(alias, sourceStr);
					}
				}
				return sourceStr;
			case QUERY_PARAMS:
				RedisUtil redisUtil = RedisUtil.getInstance();
				Object compileSource = redisUtil.get(QueryParamsMapper.QUERY_PARAMS_PREFIX + alias);
				if (compileSource != null){
					return new JSONObject(compileSource.toString());
				} else {
					throw new RuntimeException("未找到名为"+alias+"的查询配置");
				}
			default:
				throw new RuntimeException("错误的缓存数据类型");
		}
	}

	/**
	 * Logic缓存清理
	 */
	public static void clearLogicCache(){
		LOGIC_CACHE.clear();
	}

	/**
	 * SQL缓存清理
	 */
	public static void clearSqlCache(){
		SQL_CACHE.clear();
	}

	/**
	 * 查询数据源缓存清理
	 */
	public void clearQueryParamsCache(){
		RedisUtil redisUtil = RedisUtil.getInstance();
		if(redisUtil != null) {
			redisUtil.lock(QueryParamsMapper.QUERY_PARAMS_IS_LOADED_KEY, () -> redisUtil.set(QueryParamsMapper.QUERY_PARAMS_IS_LOADED_KEY, QueryParamsMapper.NO_LOAD_FLAG));
			queryParamsMapper.load();
		}
	}
}
