package com.aote.rs;

import com.aote.exception.ExceptionLogManage;
import com.aote.logic.LogicMapper;
import com.aote.logic.LogicServer;
import com.aote.mq.RocketMQProducer;
import com.aote.rs.mapper.WebException;
import com.aote.util.ExceptionHelper;
import com.aote.util.ResourceHelper;
import org.apache.log4j.Logger;
import org.apache.rocketmq.common.message.Message;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.inject.Singleton;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import java.nio.charset.StandardCharsets;
import java.util.Enumeration;
import java.util.Map;

@Path("logic")
@Singleton
@Component
public class LogicService {
	static Logger log = Logger.getLogger(LogicService.class);

	@Autowired
	private LogicServer logicServer;

	@Autowired
	@Qualifier("rocketMQProducer")
	private RocketMQProducer producer;

	@Autowired
	private ExceptionLogManage exceptionLogManage;

	/**
	 * 清理Logic源缓存
	 * @return ok
	 */
	@GET
	@Path("/clear")
	public String clearLogicCache(){
		ResourceHelper.clearLogicCache();
		return "ok";
	}

	/**
	 * 清理所有缓存
	 * @return ok
	 */
	@GET
	@Path("/clearall")
	public String clearAllCache(){
		ResourceHelper.clearLogicCache();
		ResourceHelper.clearSqlCache();
		ResourceHelper.clearPathCache();
		ResourceHelper.clearQueryParamsCache();
		return "ok";
	}

	/**
	 * 执行业务逻辑(GET方式)
	 *
	 * @param logicName 业务逻辑名称
	 * @param request   HttpServletRequest对象
	 * @return 执行结果
	 */
	@GET
	@Path("{logic}")
	public Response xtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		return xtSave(logicName, params.toString(),request);
	}

	/**
	 * 执行业务逻辑(POST方式)
	 *
	 * @param logicName 业务逻辑名称
	 * @param values JSON资源字符串
	 * @return 执行结果
	 */
	@POST
	@Path("{logic}")
	public Response xtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request) {
		log.info("logic:" + logicName + ", values:" + values);
		values = getValues(values,request);

		String result = this.run(logicName, values);
		Response.ResponseBuilder builder = Response.ok();
		if (result.startsWith("{")) {
			JSONObject paramJson = new JSONObject(result);
			if (paramJson.has("header") && paramJson.has("content")) {
				Map<String, Object> paramHeader = paramJson.getJSONObject("header").toMap();

				for (Map.Entry<String, Object> stringObjectEntry : paramHeader.entrySet()) {
					builder.header(stringObjectEntry.getKey(), String.valueOf(stringObjectEntry.getValue()));
				}
				return builder.type("text/plain").entity(String.valueOf(paramJson.get("content"))).build();
			}
		}
		return builder.entity(result).build();
	}

	@GET
	@Path("mq/{logic}")
	public Response mqXtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) throws Exception {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		return mqXtSave(logicName, params.toString(),request);
	}

	@POST
	@Path("mq/{logic}")
	public Response mqXtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request)
			throws Exception {
		log.info("logic:" + logicName + ", values:" + values);
		values = getValues(values,request);
		Message msg = new Message("logic",logicName,values.getBytes(StandardCharsets.UTF_8));
		// 发送单向消息
		producer.getDefaultMQProducer().sendOneway(msg);
		return Response.ok().build();
	}

	/**
	 * 异步执行业务逻辑(GET方式)
	 * https://stackoverflow.com/questions/31401640/spring-xml-equivalent-of-enableasync
	 *
	 * @param logicName 业务逻辑名称
	 * @param request   HttpServletRequest对象
	 * @return 执行结果
	 */
	@GET
	@Path("async/{logic}")
	public Response asyncXtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		try {
			asyncXtSaveContent(logicName, params.toString(), request);
		} catch (Exception ignore){}
		//移动平台推送特殊处理
		if(params.has("msg")){
			return Response.ok(params.getJSONArray("msg").get(0)).build();
		}
		return Response.ok().build();
	}

	/**
	 * 异步执行业务逻辑(POST方式)
	 * https://stackoverflow.com/questions/31401640/spring-xml-equivalent-of-enableasync
	 *
	 * @param logicName 业务逻辑名称
	 * @param values JSON资源字符串
	 * @return 执行结果
	 */
	@POST
	@Path("async/{logic}")
	public Response asyncXtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request) {
		try {
			asyncXtSaveContent(logicName, values, request);
		} catch (Exception ignore){}
		return Response.ok().build();
	}

	@Async("AsyncLogicGet")
	public void asyncXtSaveContent(String logicName, String values, HttpServletRequest request){
		xtSave(logicName,values,request);
	}

	public String xtSave(String logicName, String values) {
		return xtSave(logicName,values,null).getEntity().toString();
	}

	private String getValues(String values,HttpServletRequest request){
		if (!values.startsWith("{")){
			JSONObject paramJson = new JSONObject();
			paramJson.put("str",values);
			//加入请求头数据
			Enumeration headerNames = request.getHeaderNames();
			if(headerNames != null){
				JSONObject header = new JSONObject();
				while (headerNames.hasMoreElements()){
					String temp = headerNames.nextElement().toString();
					header.put(temp, request.getHeader(temp));
				}
				paramJson.put("header",header);
			}
			values = paramJson.toString();
		}
		return values;
	}

	/**
	 * 运行业务逻辑
	 * @param logicName 业务逻辑名
	 * @param values 业务逻辑参数
	 * @return 运行结果
	 */
	private String run(String logicName, String values) {
		try {
			Object result = logicServer.run(logicName, values);
			if (result == null) {
                return "";
			}
			// 如果执行结果为Map，转换成JSON串
			if (result instanceof Map<?, ?>) {
				JSONObject json = new JSONObject((Map<?, ?>) result);
				return json.toString();
			}
			if (result instanceof JSONObject) {
				JSONObject json = (JSONObject) result;
				return json.toString();
			}
			return result.toString();
		} catch (Exception ex) {
			String stack = ExceptionLogManage.getErrorInfo(ExceptionHelper.stackToString(ex),1);
			log.error(stack);
			// 获取源程序路径
			String path = LogicMapper.getLogic(logicName);
			if(path == null){
				path = logicName;
			}
			exceptionLogManage.writeError(ex,path,"/rs/logic/"+logicName,values);
			WebException wex = this.getWebException(ex);
			if (wex != null)
			{
				throw wex;
			}
			throw ex;
		}
	}

	// 找到WebException异常
	// return - 不存在，返回null
	private WebException getWebException(Exception ex)
	{
		Throwable cause = ex;
		while (cause != null)
		{
			if (cause instanceof WebException)
			{
				return (WebException)cause;
			}
			cause = cause.getCause();
		}

		return null;
	}

	//服务器启动时加载模块至redis
	@PostConstruct
	public void redisCacheLogic() throws Exception {
		logicServer.redisCacheLogic();
	}
}
