package com.aote.util;

import com.aote.exception.FileNotFoundException;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;

import java.util.concurrent.ConcurrentHashMap;

public class ResourceHelper {

	//Logic源缓存
	private static final ConcurrentHashMap<String, String> CACHE_LOGIC_MAP;
	//SQL源缓存
	private static final ConcurrentHashMap<String, String> CACHE_SQL_MAP;
	//Path源缓存
	private static final ConcurrentHashMap<String, String> CACHE_PATH_MAP;

	static {
		CACHE_LOGIC_MAP = new ConcurrentHashMap<>(30);
		CACHE_SQL_MAP = new ConcurrentHashMap<>(30);
		CACHE_PATH_MAP = new ConcurrentHashMap<>(30);
	}

	/**
	 * 获取资源文件里，指定全路径文件的文本内容
	 *
	 * @param fileName
	 *            : 以'/'开始的资源文件的全路径文件名
	 * @return 文件的文本内容
	 */
	public static String getString(String fileName) throws FileNotFoundException,IOException {
		String result;
		InputStream inputStream = null;
		BufferedReader br = null;
		try {
			// 没有/,加一个/开头
			if (!fileName.startsWith("/")) {
				fileName = "/" + fileName;
			}
			inputStream = ResourceHelper.class.getResourceAsStream(fileName);
			if(null == inputStream){
				throw new FileNotFoundException(fileName + "文件不存在", fileName);
			}
			br = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));

			String line;
			StringBuilder sb = new StringBuilder();
			while ((line=br.readLine())!=null) {
				sb.append(line).append("\r\n");
			}
			result = sb.toString();
		} finally {
			try {
				if(null != inputStream){
					inputStream.close();
				}
				if(br != null) {
					br.close();
				}
			} catch (IOException e) {
				throw e;
			}
		}
		return result;
	}

	/**
	 * 获取资源文件里，指定全路径文件的文本内容，并缓存
	 *
	 * @param fileName
	 *            : 以'/'开始的资源文件的全路径文件名
	 * @return 文件的文本内容
	 */
	public static String getString(ResourceType resourceType, String alias, String fileName) throws FileNotFoundException,IOException {
		String source;
		switch (resourceType){
			case LOGIC:
				if(CACHE_LOGIC_MAP.containsKey(alias)){
					source = CACHE_LOGIC_MAP.get(alias);
				} else {
					source = getString(fileName);
					// 处理回车换行
					source = source.replace("\r\n", "\n");
					CACHE_LOGIC_MAP.put(alias,source);
				}
				break;
			case SQL:
				if(CACHE_SQL_MAP.containsKey(alias)){
					source = CACHE_SQL_MAP.get(alias);
				} else {
					source = getString(fileName);
					// 获取编译后的sql语句
					source = source.replace("\r\n", "\n");
					source = "$" + source;
					CACHE_SQL_MAP.put(alias,source);
				}
				break;
			case PATH:
				if(CACHE_PATH_MAP.containsKey(alias)){
					source = CACHE_PATH_MAP.get(alias);
				} else {
					source = getString(fileName);
					// 获取编译后的sql语句
					source = source.replace("\r\n", "\n");
					source = "$" + source;
					CACHE_PATH_MAP.put(alias,source);
				}
				break;
			default:
				throw new RuntimeException("错误的缓存数据类型");
		}
		return source;
	}

	/**
	 * Logic缓存清理
	 */
	public static void clearLogicCache(){
		CACHE_LOGIC_MAP.clear();
	}

	/**
	 * SQL缓存清理
	 */
	public static void clearSqlCache(){
		CACHE_SQL_MAP.clear();
	}

	/**
	 * PATH缓存清理
	 */
	public static void clearPathCache(){
		CACHE_PATH_MAP.clear();
	}
}
