package com.aote.path;

import com.aote.entity.EntityServer;
import com.aote.exception.FileNotFoundException;
import com.aote.transaction.SessionPool;
import com.aote.util.ExpressionHelper;
import com.aote.util.JsonHelper;
import com.aote.util.ResourceHelper;
import com.aote.util.ResourceType;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.HashMap;
import java.util.Map;

@Component
public class PathServer {
	static Logger log = Logger.getLogger(PathServer.class);

	@Autowired
	public SessionFactory sessionFactory;

	@Autowired
	public SessionPool sessionPool;

	public SessionFactory getSessionFactory() {
		return sessionFactory;
	}

	public Session assignedSession;

	public void setAssignedSession(Session session) {
		this.assignedSession = session;
	}

	/**
	 * 执行一条path查询
	 */
	public JSONArray query(String name, JSONObject params) {
		return query(name, params.toString());
	}

	/**
	 * 执行一条path查询
	 * @param name path名称
	 * @param str 参数
	 */
	public JSONArray query(String name, String str) {
		try {
			// 获取参数
			JSONObject param = null;
			if (str != null && !str.isEmpty()) {
				JSONObject json = new JSONObject(str);
				if (json.has("data")) {
					param = json.getJSONObject("data");
				}
			}
			Map<String, Object> params = JsonHelper.toMap(param);
			// 产生path语句编译后的结果
			String path = this.call(name, params);

			Session session = (assignedSession==null ? sessionPool.getSession() : assignedSession);
			return PathHelper.query(session, path);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 获取path查询的总和
	 * @param name path查询
	 * @param str path参数
	 * @return 查询结果
	 * @throws Exception
	 */
	public JSONObject queryTotal(String name, String str) throws Exception {
		// 获取参数
		JSONObject param = null;
		if (str != null && !str.isEmpty()) {
			JSONObject json = new JSONObject(str);
			if (json.has("data")) {
				param = json.getJSONObject("data");
			}
		}
		Map<String, Object> params = JsonHelper.toMap(param);
		// 产生path语句编译后的结果
		String path = this.call(name, params);

		// 对path进行处理，只查最上层的总和
		path = filterOutOrderBy(path);
		Session session = (assignedSession==null ? sessionFactory.getCurrentSession() : assignedSession);
		JSONArray array = PathHelper.query(session, path);
		return array.getJSONObject(0);
	}

	public JSONArray query(String name, int pageNo, int pageSize, String str)
			throws Exception {
		// pageNo小于0， 纠正成1
		if (pageNo <= 0) {
			pageNo = 1;
		}

		// pageSize小于0，纠正成1, 解除1000限制
		if (pageSize < 1) {
			pageSize = 1000;
		}

		// 拿到json对象参数
		JSONObject param = null;
		if (str != null && !str.isEmpty()) {
			param = new JSONObject(str);
			param = param.getJSONObject("data");
		}
		Map<String, Object> params = JsonHelper.toMap(param);
		return query(name, pageNo, pageSize, params);
	}

	public JSONArray query(String name, int pageNo, int pageSize,
			Map<String, Object> params) throws Exception {
		// 产生SQL语句编译后的结果
		String sql = this.call(name, params);
		Session session = (assignedSession==null ? sessionFactory.getCurrentSession() : assignedSession);
		String tableName=getTableName(sql);         //获取表名
		HashMap<String, HashMap<String, Object>> metaMap = EntityServer.metaMap;//得到实例map
		String tableId=(String) metaMap.get(tableName).get("idColName"); //获取表名的主键名字
		//拼接查询表主键值得sql语句path
		String findId=searchIdSql(sql,tableName,tableId);
		try{
			JSONArray idArray = PathHelper.query(session, findId,pageNo - 1, pageSize);   //查询主键值
			log.info(idArray.toString());
			String res=getIdList(idArray, tableId);					//得到主键id数组
			JSONArray array = new JSONArray();
			if(!("".equals(res))){
				String replace="["+tableId+" in ("+res+")].(\n";    //拼接字符串
				String sqlRes=querySql(sql, replace);		        //替换
				array = PathHelper.query(session, sqlRes);
				return array;
			}
			return array;
		}catch(RuntimeException e){
			log.error("path: " + name + "出错，原因 ：" + e.getMessage());
			throw e;
		}
	}
	/**
	 * 得到id数组“1,2,3”
	 * @param idArray
	 * @param tableId
	 */
	public String getIdList(JSONArray idArray,String tableId) {
		StringBuilder result = new StringBuilder();
		if(idArray.length()>0){
			for(int j=0;j<idArray.length();j++){
				JSONObject job = idArray.getJSONObject(j);
				if(job.get(tableId) instanceof String){
					result.append("'").append(job.get(tableId).toString()).append("'");
				}else{
					result.append(job.get(tableId).toString());
				}
				if(j<idArray.length()-1){
					result.append(",");
					}

			}
		}
		return result.toString();
	}
	/**
	 * 返回拼接查询sql语句
	 * @param sql
	 * @param tableName
	 * @param tableId
	 */
	public String searchIdSql(String sql,String tableName,String tableId) {
		String resSql;
		BufferedReader reader = new BufferedReader(new StringReader(sql));
		try {
			//如果有排序，先排序
			if(sql.contains("order(")){
				int startflag=sql.indexOf("order(");
				int endflag=sql.indexOf(")", startflag);
				//获取排序字符串
				String orderbystr=sql.substring(startflag, endflag+1);
				String line = reader.readLine();             //取第一行数据
				reader.close();
				resSql=StringUtils.substringBeforeLast(line, ".")+".("+tableId+","+orderbystr+")";
			}else{
				String line = reader.readLine();             //取第一行数据
				reader.close();
				resSql=StringUtils.substringBeforeLast(line, ".")+".("+tableId+")";
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return resSql;
	}
	/**
	 * 返回拼接查询sql语句
	 * @param sql
	 * @param replaceStr
	 * @return
	 */
	public String querySql(String sql,String replaceStr) {
		StringBuilder resSql;
		BufferedReader reader = new BufferedReader(new StringReader(sql));
		try {
			String line = reader.readLine();             //取第一行数据
			resSql = new StringBuilder(line.replaceFirst("\\.\\(", replaceStr));
			while((line = reader.readLine())!=null){
				resSql.append("\n").append(line);
	        }
			reader.close();
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return resSql.toString();
	}
	/**
	 * 获取表名
	 * @param sql  sql语句
	 * @return   表名
	 */
	public String getTableName(String sql) {
		String tableName;
		BufferedReader reader = new BufferedReader(new StringReader(sql));
		try {
			String line = reader.readLine();             //取第一行数据
			reader.close();
			if (!line.contains("[")) {
				tableName=StringUtils.substringBefore(line, ".(");    //不含“[”截取“.(”之前
			}else{
				tableName=StringUtils.substringBefore(line, "[");     //含有“[”之前
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return tableName;
	}

	// 调用其它sql语句, 产生sql语句经过参数处理的结果串
	public String call(String name, Map<String, Object> params) {
		// 获取原始sql语句
		String path = PathMapper.getPath(name);
		if (path == null) {
			throw new RuntimeException("path未注册，name=" + name);
		}
		String sql;
		try {
			sql = ResourceHelper.getString(ResourceType.PATH,name,path);
		} catch (FileNotFoundException e) {
			throw new RuntimeException(path + ".文件无配置");
		} catch (IOException io) {
			throw new RuntimeException(io);
		}
		// 把自身注册到执行环境中
		params.put("sql", this);
		sql = ExpressionHelper.run(sql, params).toString();

		return sql;
	}

	// 对path路径进行过滤，改变成查总数的形式
	private String filterOutOrderBy(String source) {
		// 读取第一行数据
		BufferedReader reader = new BufferedReader(new StringReader(source));
		try {
			String line = reader.readLine();
			reader.close();
			// 去掉最后的'(', 添加'.count(*)'求和
			int index = line.lastIndexOf('.');
			return line.substring(0, index) + ".(count(*) n, group(null))";
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	// 执行业务逻辑中内嵌的path语句
	public JSONArray query(String sql) {
		Session session = (assignedSession==null ? sessionPool.getSession() : assignedSession);
		try {
			return PathHelper.query(session, sql);
		} catch(RuntimeException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 模拟平板端执行过程
	 */
	public JSONObject action(JSONObject params) {
		String path = params.getString("alias");
		JSONObject data = params.getJSONObject("data");
		JSONObject param = new JSONObject();
		param.put("data", data);
		JSONArray array = this.query(path, param);
		JSONObject result = new JSONObject("{code: 200}");
		result.put("data", array);
		return result;
	}
}
