package com.aote.logic;

import com.af.plugins.RedisTools;
import com.aote.entity.EntityServer;
import com.aote.exception.FileNotFoundException;
import com.aote.path.PathMapper;
import com.aote.path.PathServer;
import com.aote.redis.RedisUtil;
import com.aote.sql.SqlMapper;
import com.aote.sql.SqlServer;
import com.aote.transaction.SessionPool;
import com.aote.transaction.ThreadSession;
import com.aote.util.ExpressionHelper;
import com.aote.util.ResourceHelper;
import org.apache.log4j.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.metadata.ClassMetadata;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionSynchronizationManager;
import redis.clients.jedis.exceptions.JedisConnectionException;

import java.io.IOException;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.*;

@Component
@Transactional
public class LogicServer {
	static Logger log = Logger.getLogger(LogicServer.class);

	@Autowired
	private EntityServer entityServer;

	@Autowired
	private SqlServer sqlServer;

	@Autowired
	private PathServer pathServer;

	@Autowired
	private RedisUtil redisUtil;

	@Autowired
	public SessionPool sessionPool;

	@Autowired
	public SessionFactory sessionFactory;

	@Autowired
	private ThreadPoolTaskExecutor executorService;

	@Autowired
	private ThreadSession tSession;

	//异步调用Logic所用连接池
	private static ExecutorService logicExecutor = Executors.newCachedThreadPool();

	// 安卓模拟时需要的上下文
	private JSONObject context = null;

	// 执行业务逻辑处理过程
	public Object run(String name, String str) throws Exception {
		JSONObject param = null;
		// 把传递过来的参数，放到data里，以便跟entity，sql等对象区别开来
		// 判断传来的数据是否是XML格式
		str = str.trim();
		if (str.startsWith("<") && str.endsWith(">")) {
			// 传来数据为XML格式
			param = new JSONObject();
			param.put("xml", str);
		} else {
			// 传来数据为JSON格式
			Object json = new JSONTokener(str).nextValue();
			if (json instanceof JSONObject) {
				param = new JSONObject(str);
				// 有context部分，取context部分
				if (!param.isNull("context")) {
					context = param.getJSONObject("context");
				}
				if (!param.isNull("data")) {
					Object dataParam = param.get("data");
					if (dataParam instanceof String){
						param = new JSONObject(param.getString("data"));
					} else {
						param = param.getJSONObject("data");
					}
					param.put("standardData",new JSONObject(str).toString());
				}
			} else if (json instanceof JSONArray) {
				param = new JSONObject();
				param.put("arr", new JSONArray(str));
			}
		}

		return run(name, param, context);
	}

	// 业务逻辑带一个上下问对象，用于平板端测试
	public Object run(String name, JSONObject param, JSONObject context) {
		// 获取源程序内容
		String path = LogicMapper.getLogic(name);
		if (path == null) {
			throw new RuntimeException("业务逻辑未找到: " + name);
		}

		String source = null;
		try {
			source = ResourceHelper.getString(path);
		} catch (FileNotFoundException e) {
			throw new RuntimeException(path + ".文件无配置");
		} catch (IOException io) {
			throw new RuntimeException(io);
		}
		// 处理回车换行
		source = source.replace("\r\n", "\n");
		// 执行源程序
		Map<String, Object> params = new HashMap<>();
		params.put("data", param);
		// context用于测试平板端业务逻辑
		if (context != null) {
			params.put("context", context);
		}
		// 附加entityServer, sqlServer等对象到参数中
		params.put("log", this);
		params.put("entity", entityServer);
		params.put("sql", sqlServer);
		params.put("path", pathServer);
		params.put("session", sessionPool.getSession());
		params.put("redis",redisUtil);
		// 把LogicServer自身注册进去
		params.put("logic", this);

		// 附加用户注册的对象到业务逻辑中
		Map<String, Object> plugins = PluginMapper.getPlugins();
		for (String key : plugins.keySet()) {
			params.put(key, plugins.get(key));
		}
		return ExpressionHelper.run(source, params);
	}

	// 业务逻辑内部调用其它业务逻辑
	public Object run(String name, JSONObject param) throws Exception {
		return run(name, param, this.context);
	}

	/*
	 *异步调用第三方税务相关处理接口
	 */

	public Object runAsync(String name,JSONObject param) throws Exception{
		//信号量：同时允许处理的最大并发数，这里指定为10
		final Semaphore semaphore = new Semaphore(executorService.getCorePoolSize());

		/*
		 * 线程池的submit方法允许有返回值，可以用Future接口的实现类FutureTask的对象接收
		 * 线程执行完成后，调用futureTask的get()方法获取call方法的返回值
		 */
		Future<Object> futureTask=executorService.submit(new Callable<Object>() {
			Session session = sessionFactory.openSession();

			@Override
			public Object call() throws Exception {
				TransactionSynchronizationManager.bindResource(sessionFactory, session);
				log.info("开始请求第三方接口（线程池处理）");
				semaphore.acquire();
				Object result=run(name,param);
				semaphore.release();
				log.info("已处理完成（线程池处理）");
				TransactionSynchronizationManager.unbindResourceIfPossible(sessionFactory);
				if (session != null) {
	                session.close();
	            }
				return result;
			}

		});

		return futureTask.get();
	}

	/**
	 * 异步调用Logic（无回调）
	 */
	public void simpleRunAsync(String name, String params) {
		new Thread(() -> {
			try {
				run(name,params);
			} catch (Exception e) {
				log.error("出现异常：",e);
				e.printStackTrace();
			}
		}).start();
	}

	public void simpleRunAsync(String name, JSONObject params){
		simpleRunAsync(name,params.toString());
	}

	/*
	 *异步调用第三方税务相关处理接口
	 */
//	public Object runAsynchronous(String name,JSONObject param) throws Exception{
//
//		new Thread(()->{
//			log.info("开始请求第三方接口");
//			try {
//				//调用业务逻辑
//				Object result=run(name,param);
//				// TODO: 2018/6/19
//				//如果后续业务需要处理状态的反馈
//				//while定时循环请求发票entity实例对应的对于持久层的查询逻辑，
//				//条件是status='已处理'对应的状态码
//				//如果返回值大于零，则说明已处理，则break当前循环
//				//可设置sleep时间为30秒
//				//之前所有说明需要在单例模式下进行
//				log.info("已处理完成");
//			} catch (Exception e) {
//				e.printStackTrace();
//				log.error(e);
//			}
//		}).start();
//
//		return null;
//	}

	// 执行业务逻辑处理过程
	public Object noTranscRun(String name, String str) throws Exception {
		JSONObject param = null;
		// 把传递过来的参数，放到data里，以便跟entity，sql等对象区别开来
		// 判断传来的数据是否是XML格式
		str = str.trim();
		if (str.startsWith("<") && str.endsWith(">")) {
			// 传来数据为XML格式
			param = new JSONObject();
			param.put("xml", str);
		} else {
			// 传来数据为JSON格式
			Object json = new JSONTokener(str).nextValue();
			if (json instanceof JSONObject) {
				param = new JSONObject(str);
				if (!param.isNull("data")) {
					param = param.getJSONObject("data");
				}
			} else if (json instanceof JSONArray) {
				param = new JSONObject();
				param.put("arr", new JSONArray(str));
			}
		}
		return noTranscRun(name, param);
	}

	// 业务逻辑内部调用其它业务逻辑
	public Object noTranscRun(String name, JSONObject param) throws Exception {
		// 获取源程序内容
		String path = LogicMapper.getLogic(name);
		String source = ResourceHelper.getString("/logics/" + path);
		// 处理回车换行
		source = source.replace("\r\n", "\n");
		// 执行源程序
		Map<String, Object> params = new HashMap<>();
		params.put("data", param);
		// 附加entityServer, sqlServer等对象到参数中
		params.put("log", log);
		params.put("entity", entityServer);
		SqlServer sqlServer = new SqlServer();
		Session session = null;
		try {
			session = sessionFactory.openSession();
			sqlServer.setAssignedSession(session);
			params.put("sql", sqlServer);
			params.put("session", session);
			// 把LogicServer自身注册进去
			params.put("logic", this);

			// 附加用户注册的对象到业务逻辑中
			Map<String, Object> plugins = PluginMapper.getPlugins();
			for (String key : plugins.keySet()) {
				params.put(key, plugins.get(key));
			}
			return ExpressionHelper.run(source, params);
		} catch (Exception e) {
			if(session != null) {
				session.getTransaction().rollback();
			}
			throw e;
		}
		finally {
			if(session != null) {
				session.close();
			}
		}

	}

	public void fatal(Object msg){log.fatal(msg);}

	public void error(Object msg){log.error(msg);}

	public void warn(Object msg){log.warn(msg);}

	public void debug(Object msg) {
		log.debug(msg);
	}

	public void info(Object msg){log.info(msg);}

	public void trace(Object msg){log.trace(msg);}

	/**
	 * 将来可以根据业务逻辑名称配置
	 */
	public void debug(String logic, Object msg) {
		// 获取业务逻辑配置信息
		String level = LogicMapper.getAttr(logic, "log");
		if (level != null) {
			log.debug(logic + ":" + msg);
		}
	}
	/**
	 * readis缓存业务逻辑组件
	 */
	public void redisCacheLogic() throws Exception{
		if (ResourceHelper.class.getResourceAsStream("/config.json") == null) {
			log.warn("无config.json文件，不进行redis缓存");
			return;
		}
		String strConfig = ResourceHelper.getString("config.json");
		JSONObject jsonObject = new JSONObject(strConfig);
		if (!jsonObject.has("logicRegister")) {
			log.warn("无模块镜像，不进行redis缓存");
			return;
		}
		//获取模块镜像IP和端口号
		JSONObject logicJsonObject = jsonObject.getJSONObject("logicRegister");
		String url = logicJsonObject.getString("url");
		if (url == null){
			log.warn("无url，不进行redis缓存");
			return;
		}

		try {
			JSONObject redisObject = logicJsonObject.getJSONObject("redis");
			//获取服务器的IP和redis额端口号
			String redisHost = redisObject.getString("host");
			int redisPort = redisObject.getInt("port");
			String redispwd = "";
			try {
				//确认redis是否有密码
				redispwd = redisObject.getString("redispwd");
			}catch (JSONException e){
				log.info("无redis密码");
			}
			//获取redis连接池
			RedisTools.getJedisPool(redisHost, redisPort);
			//获取模块逻辑名称
			Map<String, Map<String, String>> map = LogicMapper.getMap();
			if (map != null) {
				//遍历逻辑集合，将逻辑及对应IP和端口号存入服务器的redis缓存中
				for (String key : map.keySet()) {
					//判断config.json里面是否有redispwd
					if ("".equals(redispwd)){
						RedisTools.setValue(key, url);
					}else {
						RedisTools.setValue(key, url, redispwd);
					}
					log.info("logic加入redis缓存，key=" + key + ", value=" + url);
				}
			}
			// 对sql语句进行注册
			map = SqlMapper.getMap();
			if (map != null) {
				//遍历逻辑集合，将逻辑及对应IP和端口号存入服务器的redis缓存中
				for (String key : map.keySet()) {
					//判断config.json里面是否有redispwd
					if ("".equals(redispwd)){
						RedisTools.setValue(key, url);
					}else {
						RedisTools.setValue(key, url, redispwd);
					}
					log.info("sql加入redis缓存，key=" + key + ", value=" + url);
				}
			}
			// 对path进行注册
			map = PathMapper.getMap();
			if (map != null) {
				//遍历逻辑集合，将逻辑及对应IP和端口号存入服务器的redis缓存中
				for (String key : map.keySet()) {
					//判断config.json里面是否有redispwd
					if ("".equals(redispwd)){
						RedisTools.setValue(key, url);
					}else {
						RedisTools.setValue(key, url, redispwd);
					}
					log.info("path加入redis缓存，key=" + key + ", value=" + url);
				}
			}
			// 对实体进行注册
			Map<String, ClassMetadata> entities = (Map<String, ClassMetadata>) sessionFactory.getAllClassMetadata();
			for (String key : entities.keySet()) {
				//判断config.json里面是否有redispwd
				if ("".equals(redispwd)){
					RedisTools.setValue(key, url);
				}else {
					RedisTools.setValue(key, url, redispwd);
				}
				log.info("entity加入redis缓存，key=" + key + ", value=" + url);
			}
		} catch(JedisConnectionException e) {
			log.warn("无法连接redis服务器，未成功注册业务逻辑等组件！");
		}
	}
}
