package com.aote.file;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.aote.config.SystemConfig;
import com.aote.entity.EntityServer;
import com.aote.rs.enums.ResUploadModeEnum;
import com.aote.sql.SqlServer;
import com.aote.util.other.Base64Utils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;
import org.glassfish.jersey.media.multipart.FormDataContentDisposition;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.transaction.Transactional;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * 文件处理服务
 * @author Mr.river
 */
@Component
@Transactional
public class FileServer {
    private static final Logger LOGGER = Logger.getLogger(FileServer.class);
    @Autowired
    private SqlServer sqlServer;
    @Autowired
    private EntityServer entityServer;

    /**
     * 文件上传结果
     */
    public static class FileUploadResult {

        /**
         * 是否成功
         */
        private final Boolean isSuccess;
        /**
         * 响应数据
         */
        private final String data;

        private FileUploadResult(boolean isSuccess, String data){
            this.isSuccess = isSuccess;
            this.data = data;
        }


        public Boolean getSuccess() {
            return isSuccess;
        }

        public String getData() {
            return data;
        }
    }

    /**
     * 文件上传
     *
     * @param fileInputStream 文件输入流
     * @param resUploadMode   资源上传模式
     * @param f_stock_id      文件所属仓库id
     * @param filename        文件名 前台展示需要
     * @param filesize        文件大小 前台统计需要
     * @param f_operator      操作人
     * @return 响应
     */
    public FileUploadResult uploadFile(InputStream fileInputStream, String resUploadMode, int f_stock_id, String filename,
                                       BigDecimal filesize, String f_operator, String pathKey, FormDataContentDisposition disposition,
                                       String formType
    ) throws Exception {
        ResUploadModeEnum resUploadTypeEnum = ResUploadModeEnum.toType(resUploadMode);
        if (resUploadTypeEnum == null) {
            return new FileUploadResult(false, "不支持该资源上传类型：" + resUploadMode);
        }
        // 查询出仓库信息
        String sql = "select * from t_files_manager where id = " + f_stock_id;
        JSONArray stock = new JSONArray(sqlServer.querySQL(sql).toList());
        // 获取根目录
        String fileRootPath = stock.getJSONObject(0).getString("f_root_path");
        // 获取阔扩展目录
        String extraPathKey = null;
        if (pathKey != null) {
            JSONArray extraPathKeys =
                    new JSONArray(
                            JSONArray.parseArray(stock.getJSONObject(0).getString("f_lower_path_json")).stream()
                                    .filter(item -> Objects.equals(((JSONObject) item).getString("label"), pathKey))
                                    .collect(Collectors.toList()));
            if (extraPathKeys.size() > 0) {
                extraPathKey = extraPathKeys.getJSONObject(0).getString("value");
            }
        }
        FileUploadResult result;
        // 根据不同的文件上传模式，上传文件
        switch (resUploadTypeEnum) {
            case SERVER:
            default:
                String fileName = uploadFileToServer(fileInputStream, f_stock_id, fileRootPath, extraPathKey, filename, filesize, f_operator, formType);
                result = new FileUploadResult(true, fileName);
                break;
            case BASE64:
                String base64 = uploadFileToBase64(fileInputStream);
                result = new FileUploadResult(true, base64);
                break;
            case OSS:
                String ossResult = uploadFileToOSS(fileInputStream, disposition);
                result = new FileUploadResult(true, ossResult);
                break;
        }
        return result;
    }


    /**
     * 上传文件到服务器
     * @param fileInputStream 文件输入流
     * @param disposition 描述
     * @param extraPathKey 文件上传扩展路径Key
     * @return 结果
     */
    public String uploadFileToServer(InputStream fileInputStream,
                                        FormDataContentDisposition disposition,
                                        String extraPathKey) throws IOException {
        // 获取文件后缀名
        String fileSuffix = disposition.getFileName().substring(disposition.getFileName().lastIndexOf('.') + 1);
        // 获取根目录
        String path = SystemConfig.Resources.getFileRootPath();
        // 如果有扩展目录则追加
        if (extraPathKey != null) {
            path += SystemConfig.Resources.getExtraPath(extraPathKey);
        }
        // 获取当前日期和UUID，生成最终的文件上传路径：fileRootPath\{extraPath}\yyyyMMdd\UUID_filename
        Date date = new Date();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");
        String formatDate = sdf.format(date);
        path += "\\" + formatDate;
        String uuid = UUID.randomUUID().toString().replace("-","");
        // 生成文件名
        String fileName = uuid + "." + fileSuffix;
        path = path + "\\" + fileName;
        File fileDir = new File(path);
        // 写入文件
        FileUtils.copyToFile(fileInputStream, fileDir);
        return fileName;
    }

    /**
     * 上传文件到服务器
     *
     * @param fileInputStream 文件输入流
     * @param f_stock_id      仓库id
     * @param path            仓库根目录
     * @param extraPathKey    文件上传扩展路径Key
     * @param filename        文件名称
     * @param filesize        文件大小 m为单位
     * @param f_operator      操作人
     * @return 结果 t_files 的id
     */
    public String uploadFileToServer(InputStream fileInputStream, int f_stock_id, String path,
                                     String extraPathKey, String filename, BigDecimal filesize, String f_operator, String formType)
            throws Exception {
        // TODO 保存文件
        // 获取文件后缀名
        String fileSuffix = filename.substring(filename.lastIndexOf('.') + 1);
        // 如果有扩展目录则追加
        if (extraPathKey != null) {
            path += File.separator + extraPathKey;
        }
        // 获取当前日期
        String formatDate = new SimpleDateFormat("yyyyMMdd").format(new Date());
        // 获取UUID
        String uuid = UUID.randomUUID().toString().replace("-", "");
        // 生成最终的文件上传路径：path\{extraPath}\yyyyMMdd\UUID_filename
        path += File.separator + formatDate;
        path = path + File.separator + uuid + "." + fileSuffix;
        System.out.println(path);
        File fileDir = new File(path);
        // TODO 组装存储t_files记录数据
        org.json.JSONObject files_entity = new org.json.JSONObject();
        String f_downloadpath =  path.replace("\\", "/").substring(path.indexOf(":") + 1);
        files_entity.put("f_filename", filename);
        files_entity.put("f_stock_id", f_stock_id);
        files_entity.put("f_filetype", fileSuffix);
        files_entity.put("f_downloadpath",f_downloadpath);
        files_entity.put("f_realpath", path);
        files_entity.put("f_file_size", filesize);
        files_entity.put("f_operator", f_operator);
        files_entity.put("f_form_type", formType);
        // 写入文件
        FileUtils.copyToFile(fileInputStream, fileDir);
        // 保存记录
        files_entity.put("id",JSONObject.parseObject(entityServer.partialSave("t_files", files_entity)).getString("id"));
        return files_entity.toString();
    }


    /**
     * 上传文件为base64串
     * @param fileInputStream 文件输入流
     * @return 结果
     */
    public String uploadFileToBase64(InputStream fileInputStream) throws IOException {
        return Base64Utils.encode(IOUtils.toByteArray(fileInputStream));
    }

    /**
     * 上传文件到OSS
     * @param fileInputStream 文件输入流
     * @param disposition 描述
     * @return 结果
     */
    public String uploadFileToOSS(InputStream fileInputStream,
                                  FormDataContentDisposition disposition) {
        throw new RuntimeException("暂未实现OSS对象存储模式");
    }
}
