package com.aote.file;

import com.aote.config.SystemConfig;
import com.aote.rs.enums.ResUploadModeEnum;
import com.aote.util.other.Base64Utils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;
import org.glassfish.jersey.media.multipart.FormDataContentDisposition;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.UUID;

/**
 * 文件处理服务
 * @author Mr.river
 */
@Component
public class FileServer {
    private static final Logger LOGGER = Logger.getLogger(FileServer.class);

    /**
     * 文件上传结果
     */
    public static class FileUploadResult {

        /**
         * 是否成功
         */
        private final Boolean isSuccess;
        /**
         * 响应数据
         */
        private final String data;

        private FileUploadResult(boolean isSuccess, String data){
            this.isSuccess = isSuccess;
            this.data = data;
        }


        public Boolean getSuccess() {
            return isSuccess;
        }

        public String getData() {
            return data;
        }
    }

    /**
     * 文件上传
     * @param fileInputStream 文件输入流
     * @param disposition 描述
     * @param resUploadMode 资源上传模式
     * @param extraPathKey 文件上传扩展路径Key
     * @return 响应
     */
    public FileUploadResult uploadFile(InputStream fileInputStream,
                             FormDataContentDisposition disposition,
                             String resUploadMode,
                             String extraPathKey) throws IOException {
        ResUploadModeEnum resUploadTypeEnum = ResUploadModeEnum.toType(resUploadMode);
        if(resUploadTypeEnum == null) {
            return new FileUploadResult(false, "不支持该资源上传类型：" + resUploadMode);
        }
        FileUploadResult result;
        // 根据不同的文件上传模式，上传文件
        switch (resUploadTypeEnum) {
            case SERVER:
            default:
                String fileName = uploadFileToServer(fileInputStream, disposition, extraPathKey);
                result = new FileUploadResult(true, fileName);
                break;
            case BASE64:
                String base64 = uploadFileToBase64(fileInputStream);
                result = new FileUploadResult(true, base64);
                break;
            case OSS:
                String ossResult = uploadFileToOSS(fileInputStream, disposition);
                result = new FileUploadResult(true, ossResult);
                break;
        }
        return result;
    }


    /**
     * 上传文件到服务器
     * @param fileInputStream 文件输入流
     * @param disposition 描述
     * @param extraPathKey 文件上传扩展路径Key
     * @return 结果
     */
    public String uploadFileToServer(InputStream fileInputStream,
                                        FormDataContentDisposition disposition,
                                        String extraPathKey) throws IOException {
        // 获取文件后缀名
        String fileSuffix = disposition.getFileName().substring(disposition.getFileName().lastIndexOf('.') + 1);
        // 获取根目录
        String path = SystemConfig.Resources.getFileRootPath();
        // 如果有扩展目录则追加
        if (extraPathKey != null) {
            path += SystemConfig.Resources.getExtraPath(extraPathKey);
        }
        // 获取当前日期和UUID，生成最终的文件上传路径：fileRootPath\{extraPath}\yyyyMMdd\UUID_filename
        Date date = new Date();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");
        String formatDate = sdf.format(date);
        path += "\\" + formatDate;
        String uuid = UUID.randomUUID().toString().replace("-","");
        // 生成文件名
        String fileName = uuid + "." + fileSuffix;
        path = path + "\\" + fileName;
        File fileDir = new File(path);
        // 写入文件
        FileUtils.copyToFile(fileInputStream, fileDir);
        return fileName;
    }

    /**
     * 上传文件为base64串
     * @param fileInputStream 文件输入流
     * @return 结果
     */
    public String uploadFileToBase64(InputStream fileInputStream) throws IOException {
        return Base64Utils.encode(IOUtils.toByteArray(fileInputStream));
    }

    /**
     * 上传文件到OSS
     * @param fileInputStream 文件输入流
     * @param disposition 描述
     * @return 结果
     */
    public String uploadFileToOSS(InputStream fileInputStream,
                                  FormDataContentDisposition disposition) {
        throw new RuntimeException("暂未实现OSS对象存储模式");
    }
}
