package com.aote.rs;

import com.aote.sql.SqlMapper;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 通用业务批处理服务
 * 用于避免批量业务处理时间长导致整个事务过大的问题
 *
 * @author Mrriver
 * @date 2022/8/16
 */
@Path("ReadMeterBusiness")
@Component
public class ReadMeterBatchBusinessService {
    /**
     * 日志对象
     */
    private static final Logger LOGGER = Logger.getLogger(ReadMeterBatchBusinessService.class);
    /**
     * SQL查询支持
     */
    @Autowired
    private SqlService sqlService;
    /**
     * Logic执行支持
     */
    @Autowired
    private LogicService logicService;

    /**
     * 核心数
     */
    private static final Integer processors = Runtime.getRuntime().availableProcessors();

    /**
     * 线程池
     */
    private static final ExecutorService executorService = Executors.newFixedThreadPool(processors + 1);

    /**
     * 根据已有结果集进行业务轮询
     *
     * @param logicName LOGIC名称
     * @param values    LOGIC参数，condition为结果集
     * @param request   HttpServletRequest
     */
    @POST
    @Path("batchRun/{logicName}")
    public Response batchRunByResult(@PathParam("logicName") String logicName, String values, @Context HttpServletRequest request) {
        JSONObject params = new JSONObject(values);
        JSONArray sqlResult = params.getJSONArray("params");
        doBusinessBySqlResult(logicName, values, sqlResult, request);
        return Response.ok().build();
    }

    /**
     * 根据SQL名称和参数进行业务轮询
     *
     * @param logicName LOGIC名称
     * @param sqlName   SQL名称
     * @param values    包含了LOGIC参数和SQL参数
     * @param request   HttpServletRequest
     */
    @POST
    @Path("batchRun/{logicName}/{sqlName}")
    public Response batchRunByName(@PathParam("logicName") String logicName, @PathParam("sqlName") String sqlName, String values, @Context HttpServletRequest request) {
        JSONObject params = new JSONObject(values);
        String logicParams = String.valueOf(params.get("logicParams"));
        String sqlParams = String.valueOf(params.get("sqlParams"));
        doBusinessBySqlName(logicName, logicParams, sqlName, sqlParams, request);
        return Response.ok().build();
    }


    /**
     * 根据已有SQL结果集进行业务轮询
     *
     * @param logicName LOGIC名称
     * @param values    LOGIC参数
     * @param sqlResult SQL结果集
     * @param request   HttpServletRequest
     */
    private void doBusinessBySqlResult(String logicName, String values, JSONArray sqlResult, HttpServletRequest request) {
        doBusiness(logicName, values, sqlResult, null, null, request);
    }

    /**
     * 根据SQL名称和参数进行业务轮询
     *
     * @param logicName LOGIC名称
     * @param values    LOGIC参数
     * @param sqlName   SQL名称
     * @param sqlParams SQL参数
     * @param request   HttpServletRequest
     */
    private void doBusinessBySqlName(String logicName, String values, String sqlName, String sqlParams, HttpServletRequest request) {
        doBusiness(logicName, values, null, sqlName, sqlParams, request);
    }

    /**
     * 执行单个业务
     *
     * @param logicName Logic名称
     */
    private void doBusiness(String logicName, String values, JSONArray sqlResult, String sqlName, String sqlParams, HttpServletRequest request) {
        //组织执行数据
        JSONObject data = new JSONObject(values);
        //开始执行业务
        LOGGER.info("======开始执行批量[" + logicName + "]业务流程======");
        //轮询执行依据
        JSONArray resultArray;
        if (sqlName != null) {
            // 通过SQL名称和参数执行SQL，然后将得到的SQL结果集作为轮询依据
            resultArray = getSqlResult(sqlName, sqlParams);
        } else {
            // 传入的SQL结果集作为轮询依据
            resultArray = sqlResult;
        }
        if (resultArray != null && resultArray.length() > 0) {
            JSONObject countData = new JSONObject();
            countData.put("total", resultArray.length());
            countData.put("index", 1);
            countData.put("errorCount", 0);

            AtomicInteger ai = new AtomicInteger(1);
            resultArray.forEach(item -> executorService.execute(() -> {
                try {
                    //追加请求数据
                    data.put("object", item);
                    //执行Logic
                    runLogic(logicName, data.getJSONObject("object"), request, countData);
                    LOGGER.info("======业务流程[" + logicName + "]批量执行中：" +
                            "总数：" + countData.getInt("total") + ", 当前：" + ai.getAndIncrement() + "=======");
                } catch (Exception e) {
                    LOGGER.error(e);
                }
            }));
            LOGGER.info("======业务流程[" + logicName + "]批量执行结束，错误数：" + countData.getInt("errorCount") + "=======");
        } else {
            LOGGER.info("======业务流程[" + logicName + "]批量执行未执行：没有需要轮询的数据");
        }
    }

    /**
     * 获取业务执行所需SQL的结果集
     *
     * @param sqlName SQL名称
     * @param sqlData SQL请求参数
     * @return 结果集
     */
    private JSONArray getSqlResult(String sqlName, String sqlData) {
        if (SqlMapper.getSql(sqlName) == null) {
            LOGGER.info("未找到名为【" + sqlName + "】的SQL映射文件");
            return null;
        }
        //查询条件
        try {
            JSONObject params = new JSONObject();
            params.put("data", sqlData);
            return new JSONArray(sqlService.txExecute(sqlName, 1, 9999999, params.toString()));
        } catch (Exception e) {
            LOGGER.error("出现异常：", e);
            return null;
        }
    }

    private void runLogic(String logicName, JSONObject data, HttpServletRequest request, JSONObject countData) {
        try {
            logicService.xtSave(logicName, data.toString(), request);
        } catch (Exception e) {
            int errorCount = countData.getInt("errorCount") + 1;
            countData.put("errorCount", errorCount);
        }
    }
}
