package com.aote.plugins.impexp.exportfile;

import com.af.plugins.timeoutReturn.TimeCache;
import com.aote.ThreadResource;
import com.aote.module.ModuleMapper;
import com.aote.path.PathServer;
import com.aote.sql.SqlServer;
import com.aote.util.ExcelUtil;
import com.aote.util.ResourceHelper;
import com.aote.util.SqlHelper;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.commons.configuration.reloading.FileChangedReloadingStrategy;
import org.apache.log4j.Logger;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.hibernate.Session;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import javax.sql.DataSource;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

/**
 * 导出Excel文件
 */
@Component
public class ExportFile implements IExportFile {

    // json形式的配置内容
    private JSONObject map = null;
    private static final int MAX_ROWS_PER_SHEET = 50001;
    private static final String defaultCondition = "{orderitem: 'id',condition: '1=1'}";
    static Logger log = Logger.getLogger(ExportFile.class);
    @Autowired
    private DataSource dataSource;
    @Autowired
    private PathServer pathServer;
    @Autowired
    private SqlServer sqlServer;
    @Autowired
    private TimeCache timeCache;

    /**
     * exportFileConfig导出参数
     */
    public static PropertiesConfiguration exportFileConfig;

    static {
        try {
            exportFileConfig = new PropertiesConfiguration("ExportFileConfig.properties");
            // 自动重新加载
            exportFileConfig.setReloadingStrategy(new FileChangedReloadingStrategy());
        } catch (ConfigurationException e) {
            e.printStackTrace();
        }
    }

    /**
     * 获取request对象
     * @return 返回request对象
     */
    public static HttpServletRequest getRequest() {
        return ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
    }



    /**
     * 导出数据
     * @param condition  导出时的前台传递的数据
     * @param configName 导出对应的配置内容
     * @return
     * @throws Exception
     */
    @Override
    public JSONArray export(String condition, String configName) throws Exception {
        // 读取对应目录下的配置文件
        this.map = this.getConfig(configName);
        if (map == null) {
            return noTemplate(condition);
        } else {
            Set<String> templateNames = this.map.keySet();
            if (templateNames.size() == 0) {
                throw new Exception("此导出需要模板，请检查impexp/exp/路径下的相关配置！！");
            }
            return hasTemplate(templateNames, condition);
        }
    }


    @Override
    public JSONArray export(String condition, String configName, Session session) throws Exception {
        // 读取对应目录下的配置文件
        this.map = this.getConfig(configName);
        if (map == null) {
            return noTemplate(condition, session);
        } else {
            Set<String> templateNames = this.map.keySet();
            if (templateNames.size() == 0) {
                throw new Exception("此导出需要模板，请检查impexp/exp/路径下的相关配置！！");
            }
            return hasTemplate(templateNames, condition);
        }
    }

    @Override
    public double getExportspeed() {
        return 0;
    }


    /**
     * 获取配置文件
     * @param configName 配置文件名
     * @return 配置文件
     */
    private JSONObject getConfig(String configName) {
        if (configName == null || "".equals(configName)) {
            return null;
        }
        // 从线程变量里获取登录用户目录名
        String dir = ThreadResource.ComponentDir.get();
        // 如果登录用户不为空，
        if (dir != null) {
            return null;
        }

        //模块中的路径
        Map<String, Map<String, String>> map = ModuleMapper.getMap();
        for (String key : map.keySet()) {
            try {
                String configStr = ResourceHelper.getString(key + "/impexp/exp/" + configName);
                JSONObject result = new JSONObject(configStr);
                return result;
            } catch (Exception e) {
                // 未找到文件处理
                continue;
            }
        }
        throw new RuntimeException("导出的配置文件未找到！" + configName);
    }

    /**
     *  获取模板文件
     * @param configName
     * @return
     */
//	private JSONObject getTemplate(String configName) {
//
//	}

    /**
     * 无模板导出
     * @param condition: 前台传入的条件等内容，格式为：{templateName: 要保存的excel文件名}
     * @return
     * @throws Exception
     */
    private JSONArray noTemplate(String condition) throws Exception {
        String uuid = new JSONObject(condition).getString("uuid");
        timeCache.setCahce(uuid, new JSONObject());
        JSONArray result = new JSONArray();
        JSONObject json = new JSONObject();
        long tick = System.currentTimeMillis();
        // 从前台传入的内容中，获取excel文件名，templateName为excel文件名
        String filepath = getFilePath(new JSONObject(condition).getString("templateName") + ".xlsx");
        json.put("filename", filepath);
        result.put(json);

        if (new JSONObject(condition).has("sqlName")) {
            export(new JSONObject(condition).getString("sqlName"), condition, null, filepath);
        } else if (new JSONObject(condition).has("pathName")) {
            exportpath(new JSONObject(condition).getString("pathName"), condition, null, filepath);
        }

        tick = System.currentTimeMillis() - tick;
        // 存入导出文件路径到redis
        JSONObject cache = timeCache.getCache(uuid);
        cache.put("fileName", result);
        timeCache.setCahce(uuid, cache);
        log.debug("最终: " + timeCache.getCache(uuid));
        System.out.println("进行Excel导出耗时：" + (tick / 1000.0) + "秒");
        System.out.println("文件名：" + result);
        return result;
    }

    private JSONArray noTemplate(String condition, Session session) throws Exception {
        JSONArray result = new JSONArray();
        JSONObject json = new JSONObject();
        long tick = System.currentTimeMillis();

        // 从前台传入的内容中，获取excel文件名，templateName为excel文件名
        String filepath = getFilePath(new JSONObject(condition).getString("templateName") + ".xlsx");
        json.put("filename", filepath);
        result.put(json);
        //
        if (new JSONObject(condition).has("sqlName")) {
            export(new JSONObject(condition).getString("sqlName"), condition, null, filepath, session);
        } else if (new JSONObject(condition).has("pathName")) {
            exportpath(new JSONObject(condition).getString("pathName"), condition, null, filepath);
        }

        tick = System.currentTimeMillis() - tick;


        System.out.println("进行Excel导出耗时：" + (tick / 1000.0) + "秒");
        System.out.println("文件名：" + result);
        return result;
    }

    /**
     * 有模板内容的导出
     * @param templateNames: 模板文件名
     * @param condition:     前台传入的数据
     * @return
     * @throws Exception
     */
    private JSONArray hasTemplate(Set<String> templateNames, String condition)
            throws Exception {
        JSONArray result = new JSONArray();

        int i = 0;
        // 可以有多个模板，对每一个模板进行处理
        for (String template : templateNames) {
            String sqlName = map.getString(template);
            String filepath = getFilePath(template);
            JSONObject json = new JSONObject();
            json.put("filename", filepath);
            result.put(json);
            // 模板名 template 判断各公司有无特殊模板
            template = "excel/" + template;
            // sql名 sqlName
            long tick = System.currentTimeMillis();

            if (i == 0) {
                export(sqlName, condition, template, filepath);
            } else {
                export(sqlName, defaultCondition, template, filepath);
            }

            tick = System.currentTimeMillis() - tick;
            System.out.println("进行Excel导出耗时：" + (tick / 1000.0) + "秒");
            System.out.println("文件名：" + result);
        }

        return result;
    }

    /**
     * 取得导出文件名称
     * @param name
     * @return
     */
    private String getFilePath(String name) {
        String excelFileName = (name == null ? UUID.randomUUID() + ".xlsx" : name);
        String path = ExcelUtil.class.getClassLoader().getResource("config.json")
                .getPath();
        String rootPath = path.split("WEB-INF")[0];
        String filePath = rootPath + "excel/" + excelFileName;
        return filePath;
    }

    /**
     * Real export stuff
     * @param name
     * @param templateName
     * @throws Exception
     */
    private void exportpath(String name, String body, String templateName,
                            String filePath) throws Exception {

        JSONObject joParam = new JSONObject(body);

        String[][] footer = null;
        if (joParam.has("total")) {
            JSONArray totals = joParam.getJSONArray("total");
            footer = getFooter(totals);
        }

        // 获取原始sql语句
        JSONObject joVariable = joParam.getJSONObject("data");
        //获取到结果集
        JSONArray result = pathServer.query(name, joParam.toString());
        try {
            exportWithHedearpath(filePath, joParam, result, footer);
        } catch (Exception e) {
            log.error("导出文件: " + name + "出错，原因 ：" + e.getMessage());
        }
    }


    /**
     * Real export stuff
     * @param name
     * @param templateName
     * @throws Exception
     */

    private void export(String name, String body, String templateName,
                        String filePath) throws Exception {

        JSONObject joParam = new JSONObject(body);

        String[][] footer = null;
        if (joParam.has("total")) {
            JSONArray totals = joParam.getJSONArray("total");
            footer = getFooter(totals);
        }

        // 获取原始sql语句
        JSONObject joVariable = joParam.getJSONObject("data");
        // 参数转map
        Map<String, Object> params = new HashMap<>();
        Iterator<String> iterator = joVariable.keys();
        while (iterator.hasNext()) {
            String key = (String) iterator.next();
            Object value = joVariable.get(key);
            params.put(key, value);
        }
        String sql = sqlServer.call(name, params);
        // 数据源方言兼容
        sql = SqlHelper.DATA_SOURCE_DIALECT.getSql(sql);
        log.debug("导出执行sql:\n" +  sql);
        // open connection
        Connection connection = null;
        PreparedStatement preparedStatement = null;
        ResultSet rs = null;
        try {
            // fetch data begins
            connection = dataSource.getConnection();
            preparedStatement = connection.prepareStatement(sql, ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_READ_ONLY);
            rs = preparedStatement.executeQuery();
            rs.setFetchSize(MAX_ROWS_PER_SHEET);
            if (templateName == null) {
                exportWithHedear(filePath, joParam, sql, rs, footer);
            } else {
                exportWithTemplate(templateName, filePath, rs, footer);
            }
        } finally {
            if (rs != null) {
                rs.close();
            }
            if (preparedStatement != null) {
                preparedStatement.close();
            }
            if (connection != null) {
                connection.close();
            }
        }
    }

    /**
     * Real export stuff
     * @param name
     * @param templateName
     * @throws Exception
     */
    private void export(String name, String body, String templateName,
                        String filePath, Session session) throws Exception {

        JSONObject joParam = new JSONObject(body);

        String[][] footer = null;
        if (joParam.has("total")) {
            JSONArray totals = joParam.getJSONArray("total");
            footer = getFooter(totals);
        }

        // 获取原始sql语句
        JSONObject joVariable = joParam.getJSONObject("data");
        String sql = new SqlServer().call(name, joVariable);
        JSONArray array = SqlHelper.query(session, sql);

        if (templateName == null) {
            exportWithHedear(filePath, joParam, sql, array, footer);
        }
    }

    /**
     * get footer lines
     * @param totals
     * @return
     */
    private String[][] getFooter(JSONArray totals) {
        int n = totals.length();
        String[][] footer = new String[n][];
        for (int i = 0; i < n; i++) {
            JSONObject jo = totals.getJSONObject(i);
            String[] names = JSONObject.getNames(jo);
            Arrays.sort(names, new Comparator<String>() {
                @Override
                public int compare(String o1, String o2) {
                    return Integer.parseInt(o1) - Integer.parseInt(o2);
                }
            });
            footer[i] = new String[names.length];
            for (int j = 0; j < names.length; j++) {
                if (jo.isNull(names[j])) {
                    footer[i][j] = null;
                } else {
                    footer[i][j] = jo.get(names[j]) + "";
                }
            }
        }
        return footer;
    }

    /**
     * export with template
     * @param templateName
     * @param filePath
     * @param footer
     * @throws Exception
     */
    private void exportWithTemplate(String templateName, String filePath, ResultSet rs, String[][] footer) throws Exception {
        // creating excel starts
        ExcelUtil eu = new ExcelUtil();
        eu.createBook(templateName, filePath);
        int nCol = rs.getMetaData().getColumnCount();
        int n = 1;
        List<List<Object>> rows = new ArrayList<List<Object>>();
        while (rs.next()) {
            if (n % MAX_ROWS_PER_SHEET == 0) {
                eu.createSheet();
                n = 1;
                eu.writeToSheet(rows, null, footer);
                rows = new ArrayList<List<Object>>();
            }
            List<Object> fieldList = new ArrayList<Object>();
            for (int i = 1; i <= nCol; i++) {
                Object object = rs.getObject(i);
                fieldList.add(object);
            }
            rows.add(fieldList);
            n++;
        }
        // not enough to reach the max row limit
        if (n > 1) {
            eu.createSheet();
            eu.writeToSheet(rows, null, footer);
        }
        eu.saveBook();
    }

    /**
     * export with header info
     * @param filePath
     * @param joParam
     * @param footer
     * @throws Exception
     * @throws SQLException
     */
    private void exportWithHedearpath(String filePath, JSONObject joParam,
                                      JSONArray jsonarry, String[][] footer) throws Exception,
            SQLException {
        JSONArray joField = joParam.getJSONArray("field");
        Map<String, String> colsMap = new LinkedHashMap<String, String>();
        findOutHeaders(joField, colsMap);
        String[] header = colsMap.values().toArray(new String[colsMap.size()]);
        String[] cols = colsMap.keySet().toArray(new String[colsMap.size()]);
        // creating excel starts
        ExcelUtil eu = new ExcelUtil();
        eu.createBook(null, filePath);
        log.error("数据长度是: " + jsonarry.length());
        int n = 1;
        List<List<Object>> rows = new ArrayList<List<Object>>();
        while (n <= jsonarry.length()) {
            if (n % MAX_ROWS_PER_SHEET == 0) {
                eu.createSheet();
                n = 1;
                eu.writeToSheet(rows, header, footer);
                rows = new ArrayList<List<Object>>();
            }
            List<Object> fieldList = new ArrayList<Object>();
            for (int i = 0; i < colsMap.size(); i++) {
                Object object = this.getSObject(jsonarry.getJSONObject(n - 1), cols[i]);
                fieldList.add(object);
            }
            rows.add(fieldList);
            n++;
        }
        // not enough to reach the max row limit
        if (n > 1) {
            eu.createSheet();
            eu.writeToSheet(rows, header, footer);
        }
        eu.saveBook();
    }

    /**
     * @param obj
     * @param filestr
     * @return
     */
    private Object getSObject(JSONObject obj, String filestr) {
        Object result = null;
        try {
            //得到关系字符串
            String[] strArray = filestr.split("\\.");
            int i = 0;
            result = obj;
            while (i < strArray.length) {
                result = this.getSValue((JSONObject) result, strArray[i]);
                i++;
            }
        } catch (Exception e) {
            log.error("解析对象出错: " + obj.toString() + "出错，" + filestr + "原因 ：" + e.getMessage());
        }
        if (result instanceof JSONObject) {
            result = "";
        }
        return result;
    }

    private Object getSValue(JSONObject obj, String res) {
        Object result = null;
        try {
            if (obj != null) {
                String copy = res;
                if (copy.indexOf("[") > -1 && copy.indexOf("]") > -1) {
                    String intflag = copy.substring(res.indexOf("[") + 1, copy.lastIndexOf("]"));
                    if (intflag.length() > 0) {
                        String prop = res.substring(0, res.indexOf("["));
                        if (obj.has(prop)) {
                            result = obj.getJSONArray(prop).get(Integer.parseInt(intflag));
                        } else {
                            result = null;
                        }
                    }
                } else {
                    result = obj.get(res);
                }
            }
        } catch (Exception e) {
            log.error("解析对象出错: " + obj.toString() + "出错，" + res + "原因 ：" + e.getMessage());
        }
        return result;
    }

    /**
     * export with header info
     *
     * @param filePath
     * @param joParam
     * @param sql
     * @param rs
     * @param footer
     */
    private void exportWithHedear(String filePath, JSONObject joParam, String sql, ResultSet rs, String[][] footer) {
        ExcelUtil eu = null;
        try {
            boolean fromHbm = false;
            boolean isSum = false;

            // 新的添加表头
            String[][] newHeader = null;
            if (joParam.has("header")) {
                JSONArray headerArray = joParam.getJSONArray("header");
                int n = headerArray.length();
                // 创建多维数组
                newHeader = new String[n][];
                // 往数组添加数据
                jsonToTwoArr(newHeader, headerArray, n);
            }

            // 新的添加表尾
            if (joParam.has("footer")) {
                JSONArray footerArray = joParam.getJSONArray("footer");
                int n = footerArray.length();
                // 创建多维数组
                String[][] footerth = new String[n][];
                // 往数组添加数据
                jsonToTwoArr(footerth, footerArray, n);
                // 进行替换
                footer = footerth;
            }

            // find out the headers
            JSONArray joField = joParam.getJSONArray("field");
            JSONArray sumName;
            if (joParam.has("sumName")) {
                Object object = joParam.get("sumName");
                if (object instanceof JSONArray) {
                    sumName = (JSONArray) object;
                    isSum = true;
                }
            }

            List<Object> sumNames = null;
            Object[] sum = null;
            List<Object> sums;

            if (isSum) {
                sumName = joParam.getJSONArray("sumName");
                sumNames = sumName.toList();
            }

            Map<String, String> colsMap = new LinkedHashMap<String, String>();
            findOutHeaders(joField, colsMap);
            String[] header = colsMap.values().toArray(new String[colsMap.size()]);
            String[] cols = colsMap.keySet().toArray(new String[colsMap.size()]);

            // creating excel starts
            eu = new ExcelUtil();
            int cacheInt = exportFileConfig.getInt("cacheInt", 1000);
            boolean compressTempFiles = exportFileConfig.getBoolean("CompressTempFiles", false);
            log.debug("SXSSFWorkbook导出配置: " + cacheInt + ", " + compressTempFiles);
            eu.createBigBook(filePath, cacheInt, compressTempFiles);

            int n = 1;
            List<List<Object>> rows = new ArrayList<List<Object>>();
            if (isSum) {
                sum = new BigDecimal[colsMap.size()];
            }
            rs.last();
            int curcount = 1;
            int totalCount = rs.getRow();
            log.debug("导出总行数: " + totalCount);
            rs.beforeFirst();
            while (rs.next()) {
                if (n % MAX_ROWS_PER_SHEET == 0) {
                    eu.createSheet();
                    n = 1;
                    eu.writeToSheet(rows, header, footer, newHeader);
                    rows = new ArrayList<List<Object>>();
                }
                List<Object> fieldList = new ArrayList<Object>();
                if (fromHbm) {
                    fieldList.add(rs.getObject("id"));
                }
                for (int i = (fromHbm ? 1 : 0); i < colsMap.size(); i++) {
                    Object object = rs.getObject(cols[i]);
                    if (sumNames != null && sum != null) {
                        if (sumNames.contains(cols[i])) {
                            if (sum[i] == null) {
                                BigDecimal num2 = new BigDecimal(String.valueOf(object));
                                sum[i] = num2;
                            } else {
                                BigDecimal num1 = new BigDecimal(String.valueOf(sum[i]));
                                BigDecimal num2 = new BigDecimal(String.valueOf(object));
                                sum[i] = num1.add(num2);
                            }
                        } else {
                            sum[i] = null;
                        }
                    }

                    fieldList.add(object);
                }
                rows.add(fieldList);

                double dPercent = (double) curcount / totalCount;
                int percent = (int) (dPercent * 100);
                // 进度存入redis
                String uuid = joParam.getString("uuid");
                JSONObject cache = timeCache.getCache(uuid);
                cache.put("percent", percent);
                timeCache.setCahce(uuid, cache);
                n++;
                curcount++;
            }
            if (sum != null) {
                sums = Arrays.asList(sum);
                rows.add(sums);
            }
            // not enough to reach the max row limit
            if (n > 1) {
                eu.createSheet();
                eu.writeToSheet(rows, header, footer, newHeader);
            }
            eu.saveBook();
        } catch (Exception e) {
            try {
                String uuid = joParam.getString("uuid");
                JSONObject cache = timeCache.getCache(uuid);
                cache.put("error", e.getMessage());
                timeCache.setCahce(uuid, cache);
            } catch (Exception exception) {
                log.debug("error存入redis异常", exception);
            }
            log.debug("导出excel异常: ", e);
        } finally {
            if (eu != null) {
                Workbook theBook = eu.getTheBook();
                // 删除临时文件
                if (theBook instanceof SXSSFWorkbook) {
                    log.debug("删除临时文件");
                    ((SXSSFWorkbook) theBook).dispose();
                }
            }
        }
    }

    private void jsonToTwoArr(String[][] newHeader, JSONArray headerArray, int n) {
        for (int i = 0; i < n; i++) {
            JSONArray array = headerArray.getJSONArray(i);
            int p = array.length();
            newHeader[i] = new String[p];
            for (int j = 0; j < p; j++) {
                newHeader[i][j] = String.valueOf(array.get(j));
            }
        }
    }

    private void exportWithHedear(String filePath, JSONObject joParam,
                                  String sql, JSONArray array, String[][] footer) throws Exception,
            SQLException {
        boolean fromHbm = false;
        boolean isSum = false;

        // find out the headers
        JSONArray joField = joParam.getJSONArray("field");
        JSONArray sumName;
        if (joParam.has("sumName")) {
            Object object = joParam.get("sumName");
            if (object instanceof JSONArray) {
                sumName = (JSONArray) object;
                isSum = true;
            }
        }

        List<Object> sumNames = null;
        Object[] sum = null;
        List<Object> sums;

        if (isSum) {
            sumName = joParam.getJSONArray("sumName");
            sumNames = sumName.toList();
        }

        Map<String, String> colsMap = new LinkedHashMap<String, String>();
        findOutHeaders(joField, colsMap);
        String[] header = colsMap.values().toArray(new String[colsMap.size()]);
        String[] cols = colsMap.keySet().toArray(new String[colsMap.size()]);

        // creating excel starts
        ExcelUtil eu = new ExcelUtil();
        eu.createBook(null, filePath);

        int n = 1;
        List<List<Object>> rows = new ArrayList<List<Object>>();
        if (isSum) {
            sum = new BigDecimal[colsMap.size()];
        }
        for (Object item : array) {
            JSONObject objectItem = (JSONObject) item;
            if (n % MAX_ROWS_PER_SHEET == 0) {
                eu.createSheet();
                n = 1;
                eu.writeToSheet(rows, header, footer);
                rows = new ArrayList<List<Object>>();
            }
            List<Object> fieldList = new ArrayList<Object>();
            if (fromHbm) {
                fieldList.add(objectItem.get("id"));
            }
            for (int i = (fromHbm ? 1 : 0); i < colsMap.size(); i++) {
                Object object = objectItem.get(cols[i]);
                if (sumNames != null && sum != null) {
                    if (sumNames.contains(cols[i])) {
                        if (sum[i] == null) {
                            BigDecimal num2 = new BigDecimal(String.valueOf(object));
                            sum[i] = num2;
                        } else {
                            BigDecimal num1 = new BigDecimal(String.valueOf(sum[i]));
                            BigDecimal num2 = new BigDecimal(String.valueOf(object));
                            sum[i] = num1.add(num2);
                        }
                    } else sum[i] = null;
                }

                fieldList.add(object);
            }
            rows.add(fieldList);
            n++;
        }
        if (sum != null) {
            sums = Arrays.asList(sum);
            rows.add(sums);
        }
        // not enough to reach the max row limit
        if (n > 1) {
            eu.createSheet();
            eu.writeToSheet(rows, header, footer);
        }
        eu.saveBook();
    }

    /**
     * find out headers
     *
     * @param joField
     * @param headers
     */
    private void findOutHeaders(JSONArray joField, Map<String, String> headers) {
        for (Object temp : joField) {
            String col = ((String) temp).split(":")[0];
            headers.put(col, ((String) temp).split(":")[1]);
        }
    }

}
