package com.af.plugins;

import com.aote.ThreadResource;
import com.aote.rs.mapper.WebException;
import com.aote.transaction.ClientSession;
import com.aote.transaction.SessionPool;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.*;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;

/**
 * REST风格请求插件
 */
@Component
public class RestTools {

	@Autowired
	private ClientSession clientSession;

	private static final int CONNECT_TIMEOUT = Config.getConnectTimeout();// 设置连接建立的超时时间
	private static final int SOCKET_TIMEOUT = Config.getConnectTimeout();

	/**
	 * 发送通用Http请求
	 * @param path  请求路径
	 * @param value 请求参数
	 * @param headersStr    请求头
	 * @param base  请求类型
	 * @return  请求结果
	 */
	public static String request(String path, String value, String headersStr, HttpEntityEnclosingRequestBase base){
		if (!path.startsWith("http")) {
			// 从系统配置中获得path对应的地址
			path = UrlTools.getUrl(path);
		}
		//设置请求地址
		base.setURI(URI.create(path));
		//设置请求体
		if(value != null && !"".equals(value)){
			StringEntity se = new StringEntity(value,StandardCharsets.UTF_8);
			base.setEntity(se);
		}
		//设置请求头
		setHeaders(headersStr, base);
		//设置超时时间
		RequestConfig requestConfig = RequestConfig.custom()
				.setConnectionRequestTimeout(CONNECT_TIMEOUT)
				.setConnectTimeout(CONNECT_TIMEOUT)
				.setSocketTimeout(SOCKET_TIMEOUT).build();
		base.setConfig(requestConfig);
		//发送请求
		HttpClient httpClient = HttpClientBuilder.create().build();
		try {
			HttpResponse response = httpClient.execute(base);
			int code = response.getStatusLine().getStatusCode();
			// 获取数据成功，返回数据
			if (code == 200) {
				return EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);
			} else {
				String data = response.getStatusLine().getReasonPhrase();
				// 返回错误码
				return "{status: " + code + ", data: '" + data + "'}";
			}
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 发送post请求
	 * @param path 请求路径 ps: http://127.0.0.1:8080/rs/logic/test
	 * @param header 请求头 ps: {"Accept":"application/text","Content-Type":"application/text;charset=utf-8"}
	 * @param content 请求内容 ps: A57F5E34BA47178E7A7699C776BBDF13D0449C8B9A2419EC4ED7E723ACE619B11BCA4670
	 * @return 请求结果 ps:{"code":200,"header":"{\"MacValue\":\"8A8EE314\",\"Cache-Control\":\"private\",\"Server\":\"Microsoft-IIS/10.0\",\"X-AspNet-Version\":\"4.0.30319\",\"Vary\":\"Accept-Encoding\",\"Date\":\"Mon, 23 Mar 2020 10:04:13 GMT\",\"Content-Type\":\"text/html; charset=utf-8\",\"X-AspNetMvc-Version\":\"4.0\",\"DisperseKey\":\"9949476137558769\",\"X-Powered-By\":\"ASP.NET\"}","content":"6AF3260CE77B7BA3D22CD10562DEB5991F1434E95563AD57B216A4454098328C820AB99CF3055D15957A5435F23F1EF7B2586710B18319192AD69EB8978E0CA7B281A6AA92E198C8"}
	 * @deprecated 请使用规范的 RestTools.post(String path, String value, String header)
	 */
	@Deprecated
	public static String requestPost(String path,String header,String content){
		return post(path, content, header);
	}

	/**
	 * 发送通用Http请求
	 * @param path  请求路径
	 * @param value 请求参数
	 * @param headersStr    请求头
	 * @param base  请求类型
	 * @return  请求结果
	 */
	public static HttpEntity requestPostFile(String path, String value, String headersStr, HttpEntityEnclosingRequestBase base){
		if (!path.startsWith("http")) {
			// 从系统配置中获得path对应的地址
			path = UrlTools.getUrl(path);
		}
		//设置请求地址
		base.setURI(URI.create(path));
		//设置请求体
		if(value != null && !"".equals(value)){
			StringEntity se = new StringEntity(value,StandardCharsets.UTF_8);
			base.setEntity(se);
		}
		//设置请求头
		setHeaders(headersStr, base);
		// 发送请求
		HttpClient httpClient = HttpClientBuilder.create().build();
		try {
			HttpResponse response = httpClient.execute(base);
			int code = response.getStatusLine().getStatusCode();

			// 获取数据成功，返回数据
			if (code == 200) {
				return response.getEntity();
			}else {
				throw new WebException(code,"获取文件失败");
			}
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 设置请求头，如果线程局部变量里有Token，把Token也发送过去
	 * @param headersStr
	 * @param base
	 */
	private static void setHeaders(String headersStr, HttpEntityEnclosingRequestBase base) {
		// 设置token
		String token = ThreadResource.Token.get();
		if (token != null) {
			base.setHeader("Token", token);
		}

		// 设置其他请求头
		if (headersStr != null && !"".equals(headersStr)) {
			JSONObject headers = new JSONObject(headersStr);
			Iterator keys = headers.keys();
			while (keys.hasNext()) {
				String key = (String) keys.next();
				String val = headers.getString(key);
				base.setHeader(key, val);
			}
		}
	}

	/**
	 * 发送post请求
	 *
	 * @param path  请求路径
	 * @return String结果
	 */
	public static String post(String path){
		return post(path, "",null);
	}

	/**
	 * 发送带请求体JSON的post请求
	 *
	 * @param path    请求路径
	 * @param value    请求参数
	 * @return String结果
	 */
	public static String post(String path, JSONObject value) {
		return post(path, value, null);
	}

	/**
	 * 发送带请求体JSON的post请求
	 *
	 * @param path    请求路径
	 * @param value    请求参数
	 * @return String结果
	 */
	public static HttpEntity postHttpEntity(String path, JSONObject value) {
		JSONObject headers = new JSONObject();
		headers.put("Content-Type", "application/json");
		return requestPostFile(path,value.toString(),headers.toString(),new HttpPost());
	}

	/**
	 * 发送带请求体字符串的post请求
	 *
	 * @param path  请求路径
	 * @param value 请求参数
	 * @return String结果
	 */
	public static String post(String path, String value) {
		JSONObject headers = new JSONObject();
		return request(path,value,headers.toString(),new HttpPost());
	}

	/**
	 * 发送带请求体字符串的post请求
	 *
	 * @param path  请求路径
	 * @param value 请求参数
	 * @return String结果
	 */
	public static String post(String path, String value, String header) {
		return request(path,value,header,new HttpPost());
	}

	/**
	 * 发送带请求体字符串的带请求头的post请求
	 *
	 * @param path  请求路径
	 * @param value 请求参数
	 * @return String结果
	 */
	public static String postheader(String path, String value) {
		JSONObject headers = new JSONObject();
		headers.put("Content-Type", "application/json");
		return request(path,value,headers.toString(),new HttpPost());
	}

	/**
	 * 发送带请求头和请求体的post请求
	 *
	 * @param path    请求路径
	 * @param value    请求参数
	 * @param headers    请求头
	 * @return 请求结果
	 */
	public static String post(String path, JSONObject value, JSONObject headers) {
		if(value != null){
			if(headers == null){
				headers = new JSONObject();
			}
			if(!headers.has("Content-Type")){
				//指定请求参数的数据格式是JSON。
				headers.put("Content-Type", "application/json");
			}
			return request(path,value.toString(),headers.toString(),new HttpPost());
		} else {
			if(headers == null){
				return request(path,"","",new HttpPost());
			}
			return request(path,"",headers.toString(),new HttpPost());
		}
	}

	/**
	 * 发送get请求
	 * @param path	请求路径
	 * @return String结果
	 */
	public static String get(String path) {
		return get(path, null);
	}

	/**
	 * 发送带请求头的get请求
	 *
	 * @param path    请求路径
	 * @param headers 请求头
	 * @return String结果
	 */
	public static String get(String path, JSONObject headers) {
		try {
			if (!path.startsWith("http")) {
				// 从系统配置中获得path对应的地址
				path = UrlTools.getUrl(path);
			}
			// GETMethod
			HttpGet getMethod = new HttpGet(path);
			//设置请求头
			if (headers != null) {
				Iterator keys = headers.keys();
				while (keys.hasNext()) {
					String key = (String) keys.next();
					String val = headers.getString(key);
					getMethod.setHeader(key, val);
				}
			}
			// 发送Get请求
			HttpClient httpClient = HttpClientBuilder.create().build();
			HttpResponse response = httpClient.execute(getMethod);
			int code = response.getStatusLine().getStatusCode();

			// 获取数据成功，返回数据
			if (code == 200) {
				return EntityUtils.toString(response.getEntity(),StandardCharsets.UTF_8);
			} else {
				String data = response.getStatusLine().getReasonPhrase();
				// 返回错误码
				return "{status: " + code + ", data: '" + data + "'}";
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 发送get请求
	 *
	 * @param path    请求路径
	 * @return 返回的byte[]数据
	 */
	public static byte[] getBytesForPost(String path) {
		try {
			// 从系统配置中获得path对应的地址
			path = UrlTools.getUrl(path);
			// GETMethod
			HttpGet getMethod = new HttpGet(path);
			// 发送Get请求
			HttpClient httpClient = HttpClientBuilder.create().build();
			HttpResponse response = httpClient.execute(getMethod);
			int code = response.getStatusLine().getStatusCode();

			// 获取数据成功，返回数据
			if (code == 200) {
				return EntityUtils.toByteArray(response.getEntity());
			} else {
				String data = response.getStatusLine().getReasonPhrase();
				// 返回错误码
				throw new WebException(code, data);
			}
		} catch (RuntimeException e) {
			throw e;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 发送getJson请求
	 *
	 * @param path	请求路径
	 * @return json对象
	 */
	public static JSONObject getJson(String path) {
		return getJson(path, null);
	}

	/**
	 * 发送带请求头的getJson请求
	 *
	 * @param path    请求路径
	 * @param headers 请求头
	 * @return json对象
	 */
	public static JSONObject getJson(String path, JSONObject headers) {
		try {
			if (!path.startsWith("http")) {
				// 从系统配置中获得path对应的地址
				path = UrlTools.getUrl(path);
			}
			// GETMethod
			HttpGet getMethod = new HttpGet(path);
			//设置请求头
			getMethod.setHeader("Content-Type", "application/json");
			if (headers != null) {
				Iterator keys = headers.keys();
				while (keys.hasNext()) {
					String key = (String) keys.next();
					String val = headers.getString(key);
					getMethod.setHeader(key, val);
				}
			}
			// 发送Get请求
			HttpClient httpClient = HttpClientBuilder.create().build();
			HttpResponse response = httpClient.execute(getMethod);
			int code = response.getStatusLine().getStatusCode();

			// 获取数据成功，返回数据
			if (code == 200) {
				String actual = EntityUtils.toString(response.getEntity(),StandardCharsets.UTF_8);
				Object array;
				if (actual.startsWith("{")) {
					array = new JSONObject(actual);
				} else if (actual.startsWith("[")) {
					array = new JSONArray(actual);
				} else {
					throw new RuntimeException("调用rest服务失败。");
				}
				JSONObject result = new JSONObject();
				result.put("code", code);
				result.put("data", array);
				return result;
			} else {
				JSONObject result = new JSONObject();
				String data = response.getStatusLine().getReasonPhrase();
				// 返回错误码
				result.put("code", code);
				result.put("data", data);
				return result;
			}
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 发送put请求
	 *
	 * @param path	请求路径
	 * @return String结果
	 */
	public static String put(String path){
		return put(path,null,null);
	}

	/**
	 * 发送带JSON串请求参数的put请求
	 *
	 * @param path	请求路径
	 * @param value	请求参数
	 * @return String结果
	 */
	public static String put(String path, JSONObject value){
		return put(path,value,null);
	}


	/**
	 * 发送带JSON串请求参数的put请求
	 *
	 * @param path	请求路径
	 * @param value	请求参数
	 * @return String结果
	 */
	public static String put(String path, String value){

		if(value != null){
			JSONObject	headers = new JSONObject();
			//指定请求参数的数据格式是JSON。
			headers.put("Content-Type", "application/json");
			return request(path, value,headers.toString(),new HttpPut());
		} else {
			return request(path,"","",new HttpPut());
		}
	}

	/**
	 * 发送带请求头的put请求
	 *
	 * @param path	请求路径
	 * @param value	请求参数
	 * @param headers	请求头
	 * @return String结果
	 */
	public static String put(String path,JSONObject value,JSONObject headers){
		if(value != null){
			if(headers == null){
				headers = new JSONObject();
			}
			//指定请求参数的数据格式是JSON。
			headers.put("Content-Type", "application/json");
			return request(path,value.toString(),headers.toString(),new HttpPut());
		} else {
			if(headers == null){
				return request(path,"","",new HttpPut());
			}
			return request(path,"",headers.toString(),new HttpPut());
		}
	}

	/**
	 * 发送delete请求
	 *
	 * @param path    请求路径
	 * @return String结果
	 */
	public static String delete(String path) {
		return delete(path, null,null);
	}

	/**
	 * 发送带JSON串请求参数的delete请求
	 *
	 * @param path	请求路径
	 * @param value	请求参数
	 * @return String结果
	 */
	public static String delete(String path,JSONObject value){
		return delete(path,value,null);
	}

	/**
	 * 发送带String请求参数的delete请求
	 * @param path
	 * @param value
	 * @return String结果
	 */
	public static String delete(String path,String value){
		return request(path,value,"",new HttpDeleteWithBody());
	}

	/**
	 * 发送带请求头的delete请求
	 *
	 * @param path	请求路径
	 * @param value	请求参数
	 * @param headers	请求头
	 * @return String结果
	 */
	public static String delete(String path,JSONObject value,JSONObject headers){
		if(value != null){
			if(headers == null){
				headers = new JSONObject();
			}
			//指定请求参数的数据格式是JSON。
			headers.put("Content-Type", "application/json");
			return request(path,value.toString(),headers.toString(),new HttpDeleteWithBody());
		} else {
			if(headers == null){
				return request(path,"","",new HttpDeleteWithBody());
			}
			return request(path,"",headers.toString(),new HttpDeleteWithBody());
		}
	}

	/**
	 * 执行给定action
	 *
	 * @param param	参数列表
	 * @return json对象
	 */
	public Object action(JSONObject param) {
		String url = param.getString("url");
		String data = param.getString("data");

		// 如果线程中有session内容，调用session传播
		String actual;
		Integer id = ThreadResource.SessionId.get();
		if (id != null) {
			// session传播
			int toId = SessionPool.getInstance().createCallSession(id, url);
			// 把活动的sessionid发送过去
			JSONObject headers = new JSONObject();
			headers.put("session", toId + "");
			actual = post(url, new JSONObject(data), headers);
		} else if (clientSession.sessionId != null && clientSession.sessionId != "") {
			// 启动了客户端事务，把客户端事务sessionId添加到请求头中
			JSONObject headers = new JSONObject();
			headers.put("session", clientSession.sessionId);
			actual = post(url, new JSONObject(data), headers);
		} else {
			actual = post(url, data);
		}

		// 返回结果
		JSONObject result = new JSONObject();
		result.put("code", 200);
		// 有可能是错误码
		if(actual.startsWith("{")) {
			JSONObject actualJson = new JSONObject(actual);
			// 如果不是数据，返回错误内容
			if (actualJson.has("status")) {
				result.put("code", actualJson.get("status"));
				result.put("data", actualJson.getString("data"));
			} else {
				result.put("data", actualJson);
			}
		}
		else if(actual.startsWith("[")) {
			result.put("data", new JSONArray(actual));
		} else {
			result.put("data", actual);
		}
		return result;
	}

	/**
	 * REST请求路径拼接参数
	 *
	 * @param url   请求路径
	 * @param params 请求参数
	 * @return String结果
	 */
	public static String formatURL(String url, JSONObject params) {
		//设置表单长度30字节*N个请求参数
		int capacity = params.length() * 30;
		//拼接请求路径
		StringBuilder buffer = new StringBuilder(capacity);
		buffer.append(url).append("?");
		//取出JSON里面的请求参数，添加到路径中。
		Iterator keys = params.keys();
		while (keys.hasNext()){
			String key = (String) keys.next();
			String val = String.valueOf(params.get(key));
			if(!"".equals(val)){
				buffer.append(key).append("=").append(val);
			} else {
				buffer.append(key);
			}
			if(keys.hasNext()){
				buffer.append("&");
			}
		}
		return buffer.toString();
	}

	/**
	 * 获取标准格式的请求路径
	 * @param protocolType 协议类型（http,https,ftp...）
	 * @param url 请求ip,域名
	 * @param port 端口，默认80
	 * @param paramStr 需要拼接的路径
	 * @return 标准格式的请求路径
	 */
	public static String getStandardURL(String protocolType,String url,String port,String paramStr){
		if (port == null){
			port = "80";
		}
		url = protocolType+"://"+url+":"+port;
		if(paramStr != null){
			url += paramStr;
		}
		return url;
	}
}
