package com.af.expression;

import org.json.JSONArray;

import java.math.BigDecimal;
import java.util.*;

/**
 * 表达式入口
 *
 * @author 何宁社
 */
public class Program {
    // 源程序
    public String Source;

    // Token队列，用于回退
    private final Queue<Token> tokens = new LinkedList<>();

    // 当前获取到的字符位置
    private int pos;

    // 当前是否在字符串处理环节
    private boolean inString;

    // 字符串处理环节堆栈，$进入字符串处理环节，“{}”中间部分脱离字符串处理环节
    private final Stack<Boolean> inStrings = new Stack<>();

    public Program(String source) {
        this.Source = source.replaceAll("//[^\n]*", "");
    }

    // 调用解析过程
    public Delegate parse() {
        //执行完以下方法后就已经诞生了一个完整的执行流程，也就是被翻译成了一套完整的指令集。
        return createDelegate(expression());
    }
    public Delegate parse(Expression expression) {
        //执行完以下方法后就已经诞生了一个完整的执行流程，也就是被翻译成了一套完整的指令集。
        return createDelegate(expression);
    }

    // 编译，产生可执行单元Delegate
    public Delegate createDelegate(Expression expression){
        return new Delegate(expression);
    }

    // 获取完整表达式
    public Expression expression() {
        return CommaExp();
    }

    // 逗号表达式=赋值表达式(,赋值表达式)*
    private Expression CommaExp() {
        List<Expression> exps = new ArrayList<>();

        Expression first = AssignExp();
        exps.add(first);
        Token t = GetToken();
        // 没有结束
        while (t.type == TokenType.Oper && (",".equals(t.value) || ";".equals(t.value) || "\n".equals(t.value))) {
            Expression r = AssignExp();
            exps.add(r);
            t = GetToken();
        }
        this.tokens.offer(t);

        return Expression.Comma(exps, Source, pos);
    }

    // 赋值表达式=对象属性=一般表达式|一般表达式
    private Expression AssignExp() {
        Token t = GetToken();

        // 把第一个Token的位置记录下来，方便后面回退
        int firstPos = t.startPos;

        if (t.type != TokenType.Identy) {
            this.pos = firstPos;
            this.tokens.clear();
            this.inString = false;
            return Exp();
        }
        Expression objExp = Expression.Identy(t.value, Source, pos);
        objExp = ObjectPath(objExp);

        // 只能给对象属性或者变量赋值
        if (objExp.type != ExpressionType.Property
                && objExp.type != ExpressionType.Identy) {
            this.pos = firstPos;
            this.tokens.clear();
            this.inString = false;
            return Exp();
        }
        t = GetToken();
        if (t.type != TokenType.Oper || !"=".equals(t.value)) {
            this.pos = firstPos;
            this.tokens.clear();
            this.inString = false;
            return Exp();
        }
        Expression exp = Exp();

        // 如果是属性赋值
        if (objExp.type == ExpressionType.Property) {
            // 从属性Expression中获取对象及属性名
            String name = (String) objExp.value;
            objExp = objExp.children.get(0);
            return Expression.Assign(objExp, exp, name, Source,
                    pos);
        } else {
            // Identy 变量赋值
            return Expression.Assign(null, exp,
                    objExp.value.toString(), Source, pos);
        }
    }


    // 表达式=条件:结果(,条件:结果)(,结果)?|结果
    // 条件=单个结果
    private Expression Exp() {
        Expression v = Logic();
        // 是':'，表示条件，否则直接返回单结果
        Token t = GetToken();
        if (t.type == TokenType.Oper && ":".equals(t.value)) {
            // 第一项转换
            Expression result = Logic();
            // 下一个是","，继续读取下一个条件结果串，由于是右结合，只能采用递归
            t = GetToken();
            if (t.type == TokenType.Oper && ",".equals(t.value)) {
                // 第二项转换
                Expression sExp = Exp();
                // 返回
                return Expression.Condition(v, result, sExp, Source, pos);
            } else {
                throw new RuntimeException(GetExceptionMessage("必须有默认值!"));
            }
        } else {
            tokens.offer(t);
            return v;
        }
    }

    // 单个结果项=逻辑运算 (and|or 逻辑运算)* | !表达式
    private Expression Logic() {
        Token t = GetToken();
        // !表达式
        if (t.type == TokenType.Oper && "!".equals(t.value)) {
            Expression exp = Logic();
            exp = Expression.Not(exp, Source, pos);
            return exp;
        }
        tokens.offer(t);
        Expression v = Compare();
        t = GetToken();
        while (t.type == TokenType.Oper && ("&&".equals(t.value) || "||".equals(t.value))) {
            // 第二项转换
            Expression exp = Logic();
            // 执行
            if ("&&".equals(t.value)) {
                v = Expression.And(v, exp, Source, pos);
            } else {
                v = Expression.Or(v, exp, Source, pos);
            }
            t = GetToken();
        }
        tokens.offer(t);
        return v;
    }

    // 逻辑运算=数字表达式 (比较运算符 数字表达式)?
    private Expression Compare() {
        Expression left = Math();
        Token t = GetToken();
        if (t.type == TokenType.Oper
                && (">".equals(t.value) || ">=".equals(t.value)
                || "<".equals(t.value) || "<=".equals(t.value))) {
            Expression rExp = Math();

            if (">".equals(t.value)) {
                return Expression.GreaterThan(left, rExp, Source, pos);
            }
            if (">=".equals(t.value)) {
                return Expression.GreaterThanOrEqual(left, rExp, Source, pos);
            }
            if ("<".equals(t.value)) {
                return Expression.LessThan(left, rExp, Source, pos);
            }
            if ("<=".equals(t.value)) {
                return Expression.LessThanOrEqual(left, rExp, Source, pos);
            }
        } else if (t.type == TokenType.Oper
                && ("==".equals(t.value) || "!=".equals(t.value))) {
            Expression rExp = Math();

            // 相等比较
            if ("==".equals(t.value)) {
                return Expression.Equal(left, rExp, Source, pos);
            }
            if ("!=".equals(t.value)) {
                return Expression.NotEqual(left, rExp, Source, pos);
            }
        }
        // 返回当个表达式结果
        tokens.offer(t);
        return left;
    }

    // 单个结果项=乘除项 (+|-) 乘除项
    private Expression Math() {
        Expression v = Mul();
        Token t = GetToken();
        while (t.type == TokenType.Oper
                && ("+".equals(t.value) || "-".equals(t.value))) {
            // 转换操作数2为数字
            Expression r = Mul();

            // 开始运算
            if ("+".equals(t.value)) {
                v = Expression.Add(v, r, Source, pos);
            } else {
                v = Expression.Subtract(v, r, Source, pos);
            }
            t = GetToken();
        }
        tokens.offer(t);
        return v;
    }

    // 乘除项=项 (*|/ 项)
    private Expression Mul() {
        Expression v = UnarySub();
        Token t = GetToken();
        while (t.type == TokenType.Oper
                && ("*".equals(t.value) || "/".equals(t.value) || "%".equals(t.value))) {
            // 转换第二个为数字型
            Expression r = UnarySub();

            // 开始运算
            if ("*".equals(t.value)) {
                v = Expression.Multiply(v, r, Source, pos);
            } else if ("/".equals(t.value)) {
                v = Expression.Divide(v, r, Source, pos);
            } else {
                v = Expression.Modulo(v, r, Source, pos);
            }
            t = GetToken();
        }
        tokens.offer(t);
        return v;
    }

    // 单目运算符
    private Expression UnarySub() {
        Token t = GetToken();
        if (t.type == TokenType.Oper && "-".equals(t.value)) {
            Expression r = Item();
            return Expression.Subtract(Expression.Constant(0, Source, pos), r,
                    Source, pos);
        }
        tokens.offer(t);
        return Item();
    }

    // 项=对象(.对象路径)*
    private Expression Item() {
        StringBuilder objName = new StringBuilder();
        // 获取对象表达式
        Expression objExp = ItemHead(objName);
        // 获取对象路径表达式
        objExp = ObjectPath(objExp);
        return objExp;
    }

    // 对象=(表达式)|常数|标识符|绑定序列|字符串拼接序列|数组定义，
    // 对象解析过程中，如果是标识符，则要返回找到的对象
    private Expression ItemHead(StringBuilder name) {
        Token t = GetToken();
        if (t.type == TokenType.Oper && "(".equals(t.value)) {
            Expression result = CommaExp();
            t = GetToken();
            if (t.type != TokenType.Oper || !")".equals(t.value)) {
                throw new RuntimeException(GetExceptionMessage("括号不匹配"));
            }
            return result;
        } else if (t.type == TokenType.Oper && "{".equals(t.value)) {
            // json对象
            return Json();
        } else if (t.type == TokenType.Oper && "[".equals(t.value)) {
            // json数组
            return JsonArray();
        } else if (t.type == TokenType.Oper && "$".equals(t.value)) {
            // 字符串拼接序列
            return StringUnion();
        } else if (t.type == TokenType.Int || t.type == TokenType.Double
                || t.type == TokenType.Bool) {
            return Expression.Constant(t.value, Source, pos);
        } else if (t.type == TokenType.Identy) {
            String objName = (String) t.value;
            if ("return".equals(objName)) {
                Expression result = CommaExp();
                return Expression.Return(objName, result, Source, pos);
            } else if ("try".equals(objName)) {
                tokens.offer(t);
                return Try();
            } else if ("throw".equals(objName)) {
                Expression result = CommaExp();
                return Expression.Throw(objName, result, Source, pos);
            } else if ("break".equals(objName)){
                return Expression.Break(objName, null, pos);
            } else if ("continue".equals(objName)){
                return Expression.Continue(objName, null, pos);
            }
            // 把对象名返回
            name.append(objName);
            // 对象名处理
            return ObjectName(objName);
        } else if (t.type == TokenType.Null) {
            return Expression.Constant(null, Source, pos);
        }
        throw new RuntimeException(GetExceptionMessage("单词类型错误，" + t));
    }

    private Expression Constant(String objName) {
        // TODO Auto-generated method stub
        return null;
    }

    // TryCatch:=try {expression} (catch(Exception类 名){表达式})* finally{表达式}?
    private Expression Try() {
        Token t = GetToken();
        if (t.value != null && !"try".equals(t.value)) {
            throw new RuntimeException(GetExceptionMessage("try-catch必须以try开始"));
        }

        t = GetToken();
        if (t.value != null && !"{".equals(t.value)) {
            throw new RuntimeException(GetExceptionMessage("try块必须以{开始"));
        }
        //取try块中的表达式
        Expression tryExp = CommaExp();
        t = GetToken();
        if (t.value != null && !"}".equals(t.value)) {
            throw new RuntimeException(GetExceptionMessage("try块必须以}结束"));
        }

        // 处理catch部分
        List<Expression> catches = Catch();

        // 处理finally
        Expression finallyExp = null;
        t = GetToken();
        if (t.value != null) {
            if ("finally".equals(t.value)) {
                t = GetToken();
                if (!"{".equals(t.value)) {
                    throw new RuntimeException(GetExceptionMessage("finally块必须以{开始"));
                }
                //取finally块中的表达式
                finallyExp = CommaExp();
                t = GetToken();
                if (!"}".equals(t.value)) {
                    throw new RuntimeException(GetExceptionMessage("finally块必须以}结束"));
                }
                t = GetToken();
            }
        }
        tokens.offer(t);

        //产生异常处理表达式
        return Expression.Try(tryExp, finallyExp, catches, Source, pos);
    }

    // catch部分处理
    private List<Expression> Catch() {
        List<Expression> result = new ArrayList<>();
        Expression catchExp;
        Token t = GetToken();
        while ("catch".equals(t.value)) {
            t = GetToken();
            if (!"(".equals(t.value)) {
                throw new RuntimeException(GetExceptionMessage("catch块参数必须以(开始"));
            }

            // 取类名
            t = GetToken();
            String className = (String) t.value;

            // 取变量名
            t = GetToken();
            String varName = (String) t.value;

            t = GetToken();
            if (!")".equals(t.value)) {
                throw new RuntimeException(GetExceptionMessage("catch块参数必须以)结束"));
            }
            // 取大括号
            t = GetToken();
            if (!"{".equals(t.value)) {
                throw new RuntimeException(GetExceptionMessage("catch块必须以{开始"));
            }

            //取catch块中的表达式
            catchExp = CommaExp();
            result.add(Expression.Catch(className, varName, catchExp, Source, pos));
            // 取大括号
            t = GetToken();
            if (!"}".equals(t.value)) {
                throw new RuntimeException(GetExceptionMessage("catch块必须以}结束"));
            }
            //取下个节点
            t = GetToken();
        }

        tokens.offer(t);
        return result;
    }

    // JSON对象::={}|{属性名:属性值,属性名:属性值}
    private Expression Json() {
        List<Expression> children = new LinkedList<>();

        Token t = GetToken();
        // 空对象，直接返回
        if (t.type == TokenType.Oper && "}".equals(t.value)) {
            return Expression.Json(children, Source, pos);
        }

        tokens.offer(t);
        children.add(Attr());
        t = GetToken();
        // 如果是"," 继续看下一个属性
        while (t.type == TokenType.Oper && ",".equals(t.value)) {
            children.add(Attr());
            t = GetToken();
        }
        if (t.type != TokenType.Oper || !"}".equals(t.value)) {
            throw new RuntimeException(GetExceptionMessage("必须是'}'"));
        }

        return Expression.Json(children, Source, pos);
    }

    // JSON数组::=[]
    private Expression JsonArray() {
        Token t = GetToken();
        // 空对象，直接返回
        if (t.type == TokenType.Oper && "]".equals(t.value)) {
            // 返回JSON数组常数
            return Expression.Constant(new JSONArray(), Source, pos);
        }

        // 循环获取数组内容
        tokens.offer(t);
        List<Expression> ps = Params();
        t = GetToken();
        if (t.type != TokenType.Oper || !"]".equals(t.value)) {
            throw new RuntimeException(GetExceptionMessage("必须是]" + t));
        }

        return Expression.Array(ps, Source, pos);
    }

    // 属性值对::=属性名: 属性值
    private Expression Attr() {
        Token name = GetToken();
        if (name.type != TokenType.Identy) {
            throw new RuntimeException(GetExceptionMessage("JSON对象必须是属性名"));
        }
        Token t = GetToken();
        if (t.type != TokenType.Oper || !":".equals(t.value)) {
            throw new RuntimeException(GetExceptionMessage("必须是':'"));
        }
        Expression exp = Exp();
        return Expression.Attr(name.value.toString(), exp, Source, pos);
    }

    // 对字符串拼接序列进行解析
    private Expression StringUnion() {
        // 字符串连接方法
        Expression exp = Expression.Constant("", Source, pos);
        Token t = GetToken();
        // 是对象序列
        while ((t.type == TokenType.Oper && "{".equals(t.value))
                || t.type == TokenType.String) {
            // 字符串，返回字符串连接结果
            if (t.type == TokenType.String) {
                exp = Expression.Concat(exp,
                        Expression.Constant(t.value, Source, pos), Source, pos);
            } else {
                // 按表达式调用{}里面的内容
                Expression objExp = Exp();
                t = GetToken();
                if (t.type != TokenType.Oper || !"}".equals(t.value)) {
                    throw new RuntimeException(GetExceptionMessage("缺少'}'"));
                }
                // 字符串连接
                exp = Expression.Concat(exp, objExp, Source, pos);
            }
            t = GetToken();
        }
        tokens.offer(t);
        return exp;
    }

    // 根据名称获取对象以及对象表达式
    private Expression ObjectName(String objName) {
        return Expression.Identy(objName, Source, pos);
    }

    private Expression ObjectPath(Expression inExp) {
        // 调用条件过滤解析，转换出条件过滤结果
        Expression objExp = ArrayIndex(inExp);
        Token t = GetToken();
        while (t.type == TokenType.Oper && ".".equals(t.value)) {
            // 获取对象成员
            Token nameToken = GetToken();
            // 继续看是否方法调用，如果是方法调用，执行方法调用过程
            Token n = GetToken();
            if (n.type == TokenType.Oper && "(".equals(n.value)) {
                String name = (String) nameToken.value;
                // each方法当循环使用
                if ("each".equals(name)) {
                    objExp = For(name, objExp);
                } else {
                    objExp = MethodCall(name, objExp);
                }
            } else {
                tokens.offer(n);
                // 取属性
                String pi = (String) nameToken.value;
                objExp = Expression.Property(objExp, pi, Source, pos);
                // 调用条件过滤解析，产生条件过滤结果
                objExp = ArrayIndex(objExp);
            }
            t = GetToken();
        }
        tokens.offer(t);

        return objExp;
    }

    // 数组下标=属性名([条件])?
    private Expression ArrayIndex(Expression objExp) {
        Token n = GetToken();
        // 是数组下标
        if (n.type == TokenType.Oper && "[".equals(n.value)) {
            Expression exp = Exp();
            // 读掉']'
            n = GetToken();
            if (n.type != TokenType.Oper || !"]".equals(n.value)) {
                throw new RuntimeException(GetExceptionMessage("缺少']'"));
            }
            // 产生数组下标节点
            return Expression.ArrayIndex(objExp, exp, Source, pos);
        }
        tokens.offer(n);
        return objExp;
    }

    // For循环
    private Expression For(String name, Expression obj) {
        // 产生完整的逗号表达式
        Expression exp = this.CommaExp();
        Token t = GetToken();
        if (t.type != TokenType.Oper || !")".equals(t.value)) {
            throw new RuntimeException(GetExceptionMessage("函数调用括号不匹配"));
        }
        return Expression.For(obj, exp, Source, pos);
    }

    // 函数调用
    private Expression MethodCall(String name, Expression obj) {
        // 如果是")"，调用无参函数处理
        Token t = GetToken();
        if (t.type == TokenType.Oper && ")".equals(t.value)) {
            List<Expression> ps = new ArrayList<>();
            return Expression.Call(obj, name, ps, Source, pos);
        }

        tokens.offer(t);
        List<Expression> ps = Params();
        t = GetToken();
        if (t.type != TokenType.Oper || !")".equals(t.value)) {
            throw new RuntimeException(GetExceptionMessage("函数调用括号不匹配"));
        }

        return Expression.Call(obj, name, ps, Source, pos);
    }

    // 函数参数列表
    private List<Expression> Params() {
        List<Expression> ps = new ArrayList<>();
        String name = paramName();
        Expression exp = Exp();
        procParam(ps, exp, name);
        // 如果exp中含有data对象，说明是集合处理函数中每一项值，要当做Delegate处理。
        Token t = GetToken();
        while (t.type == TokenType.Oper && ",".equals(t.value)) {
            name = paramName();
            exp = Exp();
            procParam(ps, exp, name);
            t = GetToken();
        }
        tokens.offer(t);
        return ps;
    }

    private String paramName() {
        Token t = GetToken();
        if (t.type == TokenType.Identy) {
            Token t1 = GetToken();
            if (":".equals(t1.value)) {
                return t.value.toString();
            } else {
                tokens.offer(t);
                tokens.offer(t1);
                return "";
            }
        } else {
            tokens.offer(t);
            return "";
        }
    }

    // 处理参数，如果exp中含有data对象，说明是集合处理函数中每一项值，要当做Delegate处理。
    private void procParam(List<Expression> ps, Expression exp, String name) {
        Delegate delegate = createDelegate(exp);
        if (delegate.containsKey("data")) {
            ps.add(Expression.Constant(delegate, Source, pos));
        } else {
            ps.add(Expression.Param(exp, name, Source, pos));
        }
    }

    /**
     * 获取异常信息输出
     * */
    private String GetExceptionMessage(String msg) {
        // 对出错的语句位置进行处理
        String result = Source.substring(0, pos) + " <- "
                + Source.substring(pos);
        return msg + ", 错误片段：\n " + result;
    }

    /**
     * 获取单词
     * */
    public Token GetToken() {
        // 如果队列里有，直接取上次保存的
        if (tokens.size() != 0) {
            return tokens.poll();
        }
        // 记录单词的起始位置，包括空格等内容
        int sPos = pos;

        // 如果直接是'{'，保存上一个状态，当前状态改为非字符状态
        if (pos < Source.length() && Source.charAt(pos) == '{') {
            inStrings.push(inString);
            inString = false;
            // 返回'{'单词
            String str = Source.substring(pos, pos + 1);
            pos += 1;
            return new Token(TokenType.Oper, str, sPos);
        }

        // 如果是字符串处理状态，把除过特殊字符的所有字符全部给字符串
        if (inString) {
            int startPos = pos;
            // 在特殊情况下，字符串要使用$结束
            while (pos < Source.length() && Source.charAt(pos) != '{'
                    && Source.charAt(pos) != '$' && Source.charAt(pos) != '}') {
                pos++;
            }
            // 不是'{'，退出字符串处理环节，回到上一环节。'{'会在下次进入时，保存当前状态
            if (!(pos < Source.length() && Source.charAt(pos) == '{')) {
                inString = inStrings.pop();
            }
            Token t = new Token(TokenType.String, Source.substring(startPos,
                    pos), sPos);
            // 如果是采用$让字符串结束了，要把$读去
            if (pos < Source.length() && Source.charAt(pos) == '$') {
                pos++;
            }
            return t;
        }
        // 读去所有空白以及注释
        while (
            // 普通空白
                (pos < Source.length() && (Source.charAt(pos) == ' '
                        || Source.charAt(pos) == '\n'
                        || Source.charAt(pos) == '\t'))
                        ||
                        // 是注释
                        (pos < Source.length() - 2 && Source.charAt(pos) == '/' && Source
                                .charAt(pos + 1) == '/')) {
            // 普通空白
            if (Source.charAt(pos) != ' ' && Source.charAt(pos) != '\n' && Source.charAt(pos) != '\t') {
                // 注释
                pos += 2;
                while (pos < Source.length() && Source.charAt(pos) != '\n') {
                    pos++;
                }
                // 读掉行尾
            }
            pos++;
        }

        // 如果完了，返回结束
        if (pos == Source.length()) {
            return new Token(TokenType.End, null, sPos);
        }
        // 如果是数字，循环获取直到非数字为止
        if (Source.charAt(pos) >= '0' && Source.charAt(pos) <= '9') {
            int oldPos = pos;
            while (pos < Source.length() && Source.charAt(pos) >= '0'
                    && Source.charAt(pos) <= '9') {
                pos++;
            }
            // 如果后面是"."，按double数字对待，否则按整形数返回
            if (pos < Source.length() && Source.charAt(pos) == '.') {
                pos++;
                while (pos < Source.length() && Source.charAt(pos) >= '0'
                        && Source.charAt(pos) <= '9') {
                    pos++;
                }
                String str = Source.substring(oldPos, pos);
                return new Token(TokenType.Double, new BigDecimal(str), sPos);
            } else {
                String str = Source.substring(oldPos, pos);
                return new Token(TokenType.Int, Integer.parseInt(str), sPos);
            }
        }
        // 如果是字符，按标识符对待
        else if ((Source.charAt(pos) >= 'a' && Source.charAt(pos) <= 'z')
                || (Source.charAt(pos) >= 'A' && Source.charAt(pos) <= 'Z')
                || Source.charAt(pos) == '_') {
            int oldPos = pos;
            while (pos < Source.length()
                    && ((Source.charAt(pos) >= 'a' && Source.charAt(pos) <= 'z')
                    || (Source.charAt(pos) >= 'A' && Source.charAt(pos) <= 'Z')
                    || (Source.charAt(pos) >= '0' && Source.charAt(pos) <= '9') || Source
                    .charAt(pos) == '_')) {
                pos++;
            }
            String str = Source.substring(oldPos, pos);
            // 是bool常量
            if ("false".equals(str) || "true".equals(str)) {
                return new Token(TokenType.Bool, Boolean.parseBoolean(str),
                        sPos);
            }

            if ("null".equals(str)) {
                return new Token(TokenType.Null, null, sPos);
            }
            return new Token(TokenType.Identy, str, sPos);
        }
        // && 及 ||
        else if ((Source.charAt(pos) == '&' && Source.charAt(pos + 1) == '&')
                || (Source.charAt(pos) == '|' && Source.charAt(pos + 1) == '|')) {
            String str = Source.substring(pos, pos + 2);
            pos += 2;
            return new Token(TokenType.Oper, str, sPos);
        }
        // +、-、*、/、>、<、!等后面可以带=的处理
        else if (Source.charAt(pos) == '+' || Source.charAt(pos) == '-'
                || Source.charAt(pos) == '*' || Source.charAt(pos) == '/'
                || Source.charAt(pos) == '%' || Source.charAt(pos) == '>'
                || Source.charAt(pos) == '<' || Source.charAt(pos) == '!') {
            // 后面继续是'='，返回双操作符，否则，返回单操作符
            String str;
            if (Source.charAt(pos + 1) == '=') {
                str = Source.substring(pos, pos + 2);
                pos += 2;
            } else {
                str = Source.substring(pos, pos + 1);
                pos += 1;
            }
            return new Token(TokenType.Oper, str, sPos);
        }
        // =号开始有三种，=本身，==，=>
        else if (Source.charAt(pos) == '=') {
            String str;
            if (Source.charAt(pos + 1) == '=' || Source.charAt(pos + 1) == '>') {
                str = Source.substring(pos, pos + 2);
                pos += 2;
            } else {
                str = Source.substring(pos, pos + 1);
                pos += 1;
            }
            return new Token(TokenType.Oper, str, sPos);
        }
        // 单个操作符
        else if (Source.charAt(pos) == '(' || Source.charAt(pos) == ')'
                || Source.charAt(pos) == ',' || Source.charAt(pos) == ';'
                || Source.charAt(pos) == '.' || Source.charAt(pos) == ':'
                || Source.charAt(pos) == '@' || Source.charAt(pos) == '$'
                || Source.charAt(pos) == '{' || Source.charAt(pos) == '}'
                || Source.charAt(pos) == '[' || Source.charAt(pos) == ']') {
            // 进入字符串处理环节，保留原来状态
            if (Source.charAt(pos) == '$') {
                inStrings.push(inString);
                inString = true;
            }
            // 碰到'{'，保存当前模式
            if (Source.charAt(pos) == '{' && inStrings.size() != 0) {
                inStrings.push(inString);
                inString = false;
            }
            // 碰到'}'，返回前面模式。
            if (Source.charAt(pos) == '}' && inStrings.size() != 0) {
                inString = inStrings.pop();
            }
            String str = Source.substring(pos, pos + 1);
            if (Source.charAt(pos) == '\n') {
                pos += 1;
                return new Token(TokenType.Oper, str, sPos);
            }
            pos += 1;
            return new Token(TokenType.Oper, str, sPos);
        } else {
            throw new RuntimeException(GetExceptionMessage("无效单词："
                    + Source.charAt(pos)));
        }
    }
}
