package com.af.expression;

import com.af.expression.exception.ReturnWayException;
import com.af.expression.util.MethodSignatureMatcher;
import com.af.expression.exception.ServiceException;
import org.json.JSONArray;
import org.json.JSONObject;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;

public class Expression {
	// 节点类型
	public final ExpressionType type;

	// 节点值
	public final Object value;

	// 子节点
	public List<Expression> children;

	// 运行时对应的Delegate，Delegate中保存有实参
	private Delegate delegate;

	// 节点对应的源程序及位置
	private final String source;
	private final int pos;

	private Expression(ExpressionType type, String source, int pos) {
		this.type = type;
		this.value = null;
		this.source = source;
		this.pos = pos;
		this.children = new ArrayList<>();
	}

	private Expression(ExpressionType type, Object value, String source, int pos) {
		this.type = type;
		this.value = value;
		this.source = source;
		this.pos = pos;
		this.children = new ArrayList<>();
	}

	private Expression(ExpressionType type, String source, int pos, List<Expression> expressions) {
		this.type = type;
		this.value = null;
		this.source = source;
		this.pos = pos;
		this.children = expressions;
	}

	private Expression(ExpressionType type, Object value, String source, int pos, List<Expression> expressions) {
		this.type = type;
		this.value = value;
		this.source = source;
		this.pos = pos;
		this.children = expressions;
	}

	private Expression(ExpressionType type, Object value, String source, int pos, Expression expression) {
		this.type = type;
		this.value = value;
		this.source = source;
		this.pos = pos;
		List<Expression> expressions = new ArrayList<>();
		expressions.add(expression);
		this.children = expressions;
	}

	// 产生常数
	public static Expression Constant(Object value, String source, int pos) {
		return new Expression(ExpressionType.Constant, value,
				source, pos);
	}

	// 产生Return
	public static Expression Return(Object value, Expression objExp, String source, int pos) {
		return new Expression(ExpressionType.Return, value,
				source, pos, objExp);
	}

	// 产生Throw
	public static Expression Throw(Object value, Expression objExp, String source, int pos) {
		return new Expression(ExpressionType.Throw, value,
				source, pos, objExp);
	}

	// 产生标识符
	public static Expression Identy(Object value, String source, int pos) {
		return new Expression(ExpressionType.Identy, value,
				source, pos);
	}

	// 产生Json对象，value=null, children=属性值对
	public static Expression Json(List<Expression> attrs, String source, int pos) {
		Expression result = new Expression(ExpressionType.Json, source, pos);
		result.children.addAll(attrs);
		return result;
	}

	// 逗号表达式, value=null, children=各表达式
	public static Expression Comma(List<Expression> children, String source,
								   int pos) {
		Expression result = new Expression(ExpressionType.Comma, source, pos);
		result.children.addAll(children);
		return result;
	}

	// 函数调用
	public static Expression Call(Expression objExp, String name,
								  List<Expression> params, String source, int pos) {
		Expression result = new Expression(ExpressionType.Call, name, source,
				pos);
		result.children.add(objExp);
		// 把所有参数加入函数调用子中
		result.children.addAll(params);
		return result;
	}

	// 覆盖toString方法，显示树状的节点信息，方便调试
	@Override
	public String toString() {
		return toString(0);
	}

	// 执行条件处理
	private Object condition() {
		// 条件
		Expression condExp = this.children.get(0);
		// 为真时表达式
		Expression isTrue = this.children.get(1);
		// 为假时表达式
		Expression isFalse = this.children.get(2);
		// 如果条件返回的不是bool值，则非空值为真，空值为假
		Object obj = condExp.invoke();
		boolean cond = getBoolean(obj);
		// 为真，返回为真的表达式，否则，返回为假的表达式
		if (cond) {
			return isTrue.invoke();
		} else {
			return isFalse.invoke();
		}
	}

	// JSON数组
	public static Expression Array(List<Expression> params, String source, int pos) {
		return new Expression(ExpressionType.Array, source, pos, params);
	}

	// 产生参数
	public static Expression Param(Expression exp, String value, String source, int pos) {
		Expression result = new Expression(ExpressionType.Param, value, source, pos);
		result.children.add(exp);
		return result;
	}

	// 获取对象属性
	public static Expression Property(Expression objExp, String name,
									  String source, int pos) {
		Expression result = new Expression(ExpressionType.Property, name,
				source, pos);
		result.children.add(objExp);
		return result;
	}

	// 产生属性表达式，value=属性名，children[0]=属性值
	public static Expression Attr(String name, Expression value, String source,
								  int pos) {
		Expression result = new Expression(ExpressionType.Attr, name, source, pos);
		result.children.add(value);
		return result;
	}

	// 赋值语句，value=属性名/变量名，child[0]=赋值对象，child[1]=赋值内容
	public static Expression Assign(Expression objExp, Expression exp,
									String name, String source, int pos) {
		Expression result = new Expression(ExpressionType.Assign, name, source,
				pos);
		result.children.add(objExp);
		result.children.add(exp);
		return result;
	}

	// 执行for循环
	private Object loop() {
		Expression objExp = this.children.get(0);
		// 获取对象，for循环
		Object obj = objExp.invoke();
		if (obj instanceof JSONArray) {
			JSONArray array = (JSONArray) obj;
			// 获取循环体，循环体中row代表每一项对象, 把对象传递给循环体执行
			Expression body = this.children.get(1);
			for (int i = 0; i < array.length(); i++) {
				delegate.put("rowIndex", i);
				delegate.put("row", array.get(i));
				body.invoke();
			}
		} else {
			int start = 0;
			int length;
			if(objExp.children.size() == 2){
				start = Integer.parseInt(objExp.children.get(0).invoke().toString());
				length = Integer.parseInt(objExp.children.get(1).invoke().toString());
			} else if(obj instanceof Integer){
				length = (int) obj;
			} else {
				throw new RuntimeException("for循环只接收集合对象，整数或两个整数的组合");
			}
			// 获取循环体，循环体中row代表索引
			for (int i = start; i < length; i++) {
				Expression body = this.children.get(1);
				delegate.put("row", i);
				body.invoke();
			}
		}
		return obj;
	}

	// 执行集合的where过滤
	private Object where() {
		JSONArray result = new JSONArray();

		Expression objExp = this.children.get(0);
		// 获取对象，where循环只针对JSONArray及list
		JSONArray array;

		Object ret = objExp.invoke();
		if (ret instanceof JSONArray) {
			array = (JSONArray) ret;
		} else if (ret instanceof Collection) {
			Collection list = (Collection) ret;
			// 把List中的数据转到JSONArray里
			array = new JSONArray();
			for (Object value : list) {
				array.put(value);
			}
		} else {
			throw new RuntimeException("where函数只支持JSONArray, List");
		}

		// 获取过滤内容，其中row代表每一项对象, 把对象传递进去执行
		Expression body = this.children.get(1);
		for (int i = 0; i < array.length(); i++) {
			Object value = array.get(i);
			delegate.put("row", value);
			Object value2;
			Object params = body.invoke();
			if (params instanceof HashMap.SimpleEntry) {
				value2 = ((HashMap.SimpleEntry) params).getValue();
			} else {
				value2 = params;
			}
			if (value2.equals(true)) {
				result.put(value);
			}
		}
		return result;
	}

	public String toString(int level) {
		// 显示自己的类型及名称
		StringBuilder result = new StringBuilder(space(level) + "type: " + this.type + ", value: "
				+ (this.value != null ? this.value.toString() : "null") + "[\n");
		// 递归显示子
		for (Expression child : this.children) {
			result.append(child != null ? child.toString(level + 1) : space(level + 1) + "null\n");
		}
		result.append(space(level)).append("]\n");
		return result.toString();
	}

	// 执行集合的sum求和，求和结果为BigDecimal
	private Object sum() {
		BigDecimal result = new BigDecimal("0");

		Expression objExp = this.children.get(0);
		// 获取对象，sum循环只针对JSONArray
		JSONArray array = (JSONArray) objExp.invoke();
		// 获取过滤内容，其中row代表每一项对象, 把对象传递进去执行
		Expression body = this.children.get(1);
		for (int i = 0; i < array.length(); i++) {
			Object value = array.get(i);
			delegate.put("row", value);
			Object params = body.invoke();
			String decimal;
			if(params instanceof HashMap.SimpleEntry){
				decimal = ((HashMap.SimpleEntry)params).getValue().toString();
			} else {
				decimal = String.valueOf(params);
			}
			// 空值当0对待
			if (decimal == null) {
				result = result.add(new BigDecimal(0));
			} else {
				result = result.add(new BigDecimal(decimal));
			}
		}

		return result;
	}

	// 对函数或者取属性结果进行转换，double转换成Decimal，整数转换成 int
	private Object convert(Object source) {
		if (source instanceof Double) {
			return new BigDecimal(source.toString());
		} else if (source instanceof Byte || source instanceof Short
				|| source instanceof Integer) {
			return Integer.parseInt(source.toString());
		} else {
			return source;
		}
	}

	// 看异常类型是否与catch一致
	private boolean isEqual(Throwable e, String name) {
		// 循环向上找父类
		Class<?> c = e.getClass();
		while (c != null) {
			String s = c.getName();
			int index = s.lastIndexOf('.');
			String catchName = s.substring(index + 1);
			if (catchName.equals(name)) {
				return true;
			} else {
				c = c.getSuperclass();
			}
		}
		return false;
	}

	// 返回Json对象的结果，返回一个Json对象
	private Object Json() {
		JSONObject result = new JSONObject();
		for (Expression child : this.children) {
			String name = child.value.toString();
			Object value = child.children.get(0).invoke();
			result.put(name, value);
		}
		return result;
	}

	// 返回JSONArray结果
	private Object array() {
		JSONArray result = new JSONArray();
		for (Expression child : this.children) {
			// 调用处理结果
			Object value = child.invoke();
			result.put(value);
		}
		return result;
	}

	// 根据值返回boolean结果
	private boolean getBoolean(Object obj) {
		// 如果条件返回的不是bool值，则非空值为真，空值为假
		boolean cond;
		if (obj instanceof Boolean) {
			cond = (Boolean) obj;
		} else {
			cond = (obj != null);
		}
		return cond;
	}

	// 执行赋值过程
	private Object assign() {
		// 属性值
		Expression right = this.children.get(1);
		Object value = right.invoke();
		// 获取属性
		String name = (String) this.value;
		// 要赋值的对象，空代表给变量赋值
		Expression left = this.children.get(0);
		if (left == null) {
			delegate.put(name, value);
		} else {
			Object obj = left.invoke();
			// 如果时JSONObject，调用JSONObject的赋值过程
			if (obj instanceof JSONObject) {
				JSONObject json = (JSONObject) obj;
				json.put(name, value);
			} else {
				try {
					Field field = obj.getClass().getField(name);
					// 给属性设置值
					field.set(obj, value);
					return value;
				} catch (Exception e) {
					throw new RuntimeException("属性赋值错误：" + name);
				}
			}
		}
		return value;
	}

	// 获取对象属性
	private Object property() throws Exception {
		Expression objExp = this.children.get(0);
		// 获取对象
		Object obj = objExp.invoke();
		// 属性名
		String name = (String) this.value;

		Object rObj;
		// 是Map
		if (obj.getClass() == HashMap.class) {
			Map<String, Object> map = (Map<String, Object>) obj;
			if (!map.containsKey(name)) {
				return null;
			}
			rObj = map.get(name);
		}
		// 是JSONObject
		else if (obj instanceof JSONObject) {
			JSONObject json = (JSONObject) obj;
			if (!json.has(name)) {
				return null;
			}
			rObj = json.get(name);
		} else {
			// 利用反射机制获得属性值
			Field field = obj.getClass().getField(name);
			rObj = field.get(obj);
		}

		return convert(rObj);
	}

	// 获取数组元素
	private Object arrayIndex() {
		Expression objExp = this.children.get(0);
		Expression indexExp = this.children.get(1);
		// 获取对象
		Object obj = objExp.invoke();
		// 获取下标值
		int index = Integer.parseInt(indexExp.invoke().toString());
		// 如果对象为JSONArray，调用JSONArray的方法
		if (obj instanceof JSONArray) {
			JSONArray array = (JSONArray) obj;
			Object oRet = array.get(index);
			return convert(oRet);
		} else if (obj.getClass().isArray()) {
			//如果对象为数组类型，调用数组的方法
			if (obj.getClass() == int[].class) {
				int[] array = (int[]) obj;
				return convert(array[index]);
			} else if (obj.getClass() == double[].class) {
				double[] array = (double[]) obj;
				return convert(array[index]);
			} else if (obj.getClass() == byte[].class) {
				byte[] array = (byte[]) obj;
				return convert(array[index]);
			} else if (obj.getClass() == boolean[].class) {
				boolean[] array = (boolean[]) obj;
				return convert(array[index]);
			} else if (obj.getClass() == long[].class) {
				long[] array = (long[]) obj;
				return convert(array[index]);
			} else if(obj.getClass() == short[].class){
				short[] array = (short[]) obj;
				return convert(array[index]);
			} else if (obj.getClass() == float[].class){
				float[] array = (float[]) obj;
				return convert(array[index]);
			} else if(obj.getClass() == char[].class){
				char[] array = (char[]) obj;
				return convert(array[index]);
			} else {
				Object[] array = (Object[]) obj;
				return convert(array[index]);
			}
		}
		throw new ExpressionException(this.source, this.pos);
	}

	// 产生多个空格
	private String space(int n) {
		StringBuilder result = new StringBuilder();
		for (int i = 0; i < n; i++) {
			result.append("    ");
		}
		return result.toString();
	}

	// 跳出业务
	private Object returnBusiness() {
		Object value = this.children.get(0).invoke();
		throw new ReturnWayException(value);
	}

	// 抛出业务异常
	private Object throwException() {
		Object result = this.children.get(0).invoke();
		if(result instanceof JSONObject){
			JSONObject obj = (JSONObject) result;
			throw new ServiceException(obj.getString("msg"), obj.getInt("status"));
		} else {
			throw new ServiceException(result.toString());
		}
	}

	private Object Param() {
		return this.children.get(0).invoke();
	}

	//执行try-catch
	private Object Try() {
		Expression[] exps = (Expression[]) this.value;
		Expression tryExp = exps[0];
		try {
			//执行try块
			return tryExp.invoke();
		} catch (ExpressionException ex) {
			Throwable e = ex.getCause();

			//执行catch块
			for (Expression exp : this.children) {
				String[] catchValues = (String[]) exp.value;
				String typeName = catchValues[0];
				String varName = catchValues[1];
				// 找到了catch部分
				//获取异常类型
				if (isEqual(e, typeName)) {
					List<Expression> catchExp = exp.children;
					// 把异常值变成变量
					delegate.put(varName, e);
					return catchExp.get(0).invoke();
				}
			}
			// 未匹配到异常
			throw ex;
		} finally {
			//执行finally块
			if(exps.length > 1) {
				Expression finallyExp = exps[1];
				if (finallyExp != null) {
					finallyExp.invoke();
				}
			}
		}
	}


	// 产生异常
	public static Expression Try(Expression tryExp, Expression finallyExp, List<Expression> catches, String source, int pos) {
		Expression[] tryFinally = new Expression[2];
		tryFinally[0] = tryExp;
		tryFinally[1] = finallyExp;

		Expression result = new Expression(ExpressionType.Try, tryFinally, source, pos);
		result.children = catches;
		return result;
	}
	//产生Catch
	public static Expression Catch(String className,String varName, Expression catches, String source, int pos) {
		String[] catchParameter = new String[2];
		catchParameter[0] = className;
		catchParameter[1] = varName;

		Expression result = new Expression(ExpressionType.Catch, catchParameter, source, pos);
		result.children.add(catches);
		return result;
	}

	// 执行
	public Object invoke() {
		try {
			switch (type) {
				//逻辑运算
				case Or: {
					Expression left = this.children.get(0);
					boolean l = getBoolean(left.invoke());
					if (l) {
						return true;
					} else {
						Expression right = this.children.get(1);
						return getBoolean(right.invoke());
					}
				}
				case And: {
					Expression left = this.children.get(0);
					Expression right = this.children.get(1);
					boolean l = getBoolean(left.invoke());
					if (l) {
						return getBoolean(right.invoke());
					} else {
						return false;
					}
				}
				case Not: {
					Expression left = this.children.get(0);
					boolean l = getBoolean(left.invoke());
					return !l;
				}
				//算数运算
				case Add:
				case Subtract:
				case Multiply:
				case Divide:
				case Modulo: {
					return math();
				}
				//字符串连接
				case Concat: {
					Expression left = this.children.get(0);
					Expression right = this.children.get(1);
					Object l = left.invoke();
					Object r = right.invoke();
					return l.toString() + r.toString();
				}
				//返回Json对象
				case Json: {
					return Json();
				}
				//返回JSONArray
				case Array: {
					return array();
				}
				//比较运算
				case GreaterThan:
				case GreaterThanOrEqual:
				case LessThan:
				case LessThanOrEqual: {
					Expression left = this.children.get(0);
					Expression right = this.children.get(1);
					Object l = left.invoke();
					Object r = right.invoke();
					// 两个字符串比较
					if (l instanceof String && r instanceof String) {
						int result = ((String) l).compareTo((String) r);
						switch (type) {
							case GreaterThan:
								return result > 0;
							case GreaterThanOrEqual:
								return result >= 0;
							case LessThan:
								return result < 0;
							case LessThanOrEqual:
								return result <= 0;
							default:
								throw new RuntimeException("未知类型");
						}
					}
					// 字符串与数字比较，把字符串转换成数字
					else {
						BigDecimal dl = new BigDecimal(l.toString());
						BigDecimal dr = new BigDecimal(r.toString());
						int cr = dl.compareTo(dr);
						switch (type) {
							case GreaterThan:
								return cr > 0;
							case GreaterThanOrEqual:
								return cr >= 0;
							case LessThan:
								return cr < 0;
							case LessThanOrEqual:
								return cr <= 0;
							default:
								throw new RuntimeException("未知类型");
						}
					}
				}
				//相等比较
				case Equal:
				case NotEqual: {
					Expression left = this.children.get(0);
					Expression right = this.children.get(1);
					Object l = left.invoke();
					if (l == null) {
						l = "null";
					}
					Object r = right.invoke();
					if (r == null) {
						r = "null";
					}
					boolean equals;
					if (l instanceof Number && r instanceof Number) {
						equals = new BigDecimal(String.valueOf(l)).compareTo(new BigDecimal(String.valueOf(r))) == 0;
					} else {
						equals = l.toString().equals(r.toString());
					}
					switch (type) {
						case Equal:
							return equals;
						case NotEqual:
							return !equals;
						default:
							throw new RuntimeException("未知类型");
					}
				}
				//常数
				case Constant: {
					// 数字
					if (this.value instanceof Double) {
						return new BigDecimal(this.value.toString());
					}
					// 其它常数，直接返回值
					return this.value;
				}
				//标识符，获取实参对象
				case Identy: {
					// 如果参数不存在，返回空
					String name = (String) this.value;
					if (!delegate.containsKey(name)) {
						return null;
					}
					return delegate.get(name);
				}
				//条件语句
				case Condition: {
					return condition();
				}
				//获取属性值
				case Property: {
					return property();
				}
				//获取数组元素
				case ArrayIndex: {
					return arrayIndex();
				}
				//函数调用
				case Call: {
					return call();
				}
				//for循环
				case For: {
					return loop();
				}
				//属性赋值
				case Assign: {
					return assign();
				}
				//逗号表达式
				case Comma: {
					Object value = 0;
					for (Expression child : this.children) {
						value = child.invoke();
					}
					return value;
				}
				//异常处理
				case Try: {
					return Try();
				}
				//参数
				case Param: {
					return Param();
				}
				//Return
				case Return: {
					return returnBusiness();
				}
				//Throw
				case Throw: {
					return throwException();
				}
				default: {
					throw new RuntimeException("无效操作");
				}
			}
		} catch (ReturnWayException ex) {
			throw ex;
		} catch (ServiceException ex) {
			throw ex;
		} catch (ExpressionException ex) {
			throw ex;
		} catch (Exception ex) {
			throw new ExpressionException(source, pos, ex);
		}
	}

	// 执行数学运算
	private Object math() {
		Expression left = this.children.get(0);
		Expression right = this.children.get(1);
		Object oLeft = left.invoke();
		Object oRight = right.invoke();
		// 只要有BigDecimal，按BigDecimal计算，否则，按int计算
		if (oLeft instanceof BigDecimal || oRight instanceof BigDecimal ||
				oLeft.toString().contains(".") || oRight.toString().contains(".")) {
			BigDecimal l = new BigDecimal(oLeft.toString());
			BigDecimal r = new BigDecimal(oRight.toString());
			switch (type) {
				case Add:
					return l.add(r);
				case Subtract:
					return l.subtract(r);
				case Multiply:
					return l.multiply(r);
				case Divide:
					return l.divide(r,4, RoundingMode.DOWN);
				case Modulo:
					return new BigDecimal(String.valueOf(l.intValue() % r.intValue()));
				default:
					throw new RuntimeException("未知类型");
			}
		} else {
			int l = Integer.parseInt(left.invoke().toString());
			int r = Integer.parseInt(right.invoke().toString());
			switch (type) {
				case Add:
					return l + r;
				case Subtract:
					return l - r;
				case Multiply:
					return l * r;
				case Divide:
					return l / r;
				case Modulo:
					return l % r;
				default:
					throw new RuntimeException("不识别的算数操作符");
			}
		}
	}

	// 执行函数调用
	private Object call() throws Exception {
		// 函数名
		String name = (String) this.value;
		// 如果是where函数，调用where过滤
		if ("where".equals(name)) {
			return where();
		}
		// 如果是sum求和函数，调用sum求和
		if ("sum".equals(name)) {
			return sum();
		}

		Expression objExp = this.children.get(0);
		// 获取对象
		Object obj = objExp.invoke();

		if (obj == null){
			throw new RuntimeException("对象不存在或为null，请检查对象的定义或赋值情况");
		}

		// 获得参数计算结果
		List<Object> params = new ArrayList<>();
		for (int i = 1; i < this.children.size(); i++) {
			Expression paramExp = this.children.get(i);
			params.add(paramExp.invoke());
		}
		Class<?> c = obj.getClass();

		// 转换参数类型
		Object[] types = new Object[params.size()];
		for (int i = 0; i < params.size(); i++) {
			Object param = params.get(i);
			if (param == null) {
				types[i] = null;
			} else {
				types[i] = param.getClass();
			}
		}

		// 利用反射机制获得函数
		Method method = MethodSignatureMatcher.getMatchingMethod(c, name, types);
		if (method == null) {
			StringBuilder error = new StringBuilder("找不到方法:" + c.getName() + "." + name + "(");
			// 添加方法参数
			for (Object t : types) {
				if (t != null) {
					error.append(t).append(",");
				} else {
					error.append("null,");
				}
			}
			error.append(")");
			throw new RuntimeException(error.toString());
		}
		// 函数返回的double转BigDecimal，整形等转int
		Object oCall = method.invoke(obj, params.toArray());
		return convert(oCall);
	}

	// 产生一个条件语句, test:条件，ifTrue:为真时结果，ifFalse:为假时结果
	public static Expression Condition(Expression test, Expression ifTrue,
			Expression ifFalse, String source, int pos) {
		Expression result = new Expression(ExpressionType.Condition, source,
				pos);
		result.children.add(test);
		result.children.add(ifTrue);
		result.children.add(ifFalse);
		return result;
	}

	// 产生逻辑非语句
	public static Expression Not(Expression exp, String source, int pos) {
		Expression result = new Expression(ExpressionType.Not, source, pos);
		result.children.add(exp);
		return result;
	}

	// 产生逻辑与语句
	public static Expression And(Expression left, Expression right,
			String source, int pos) {
		Expression result = new Expression(ExpressionType.And, source, pos);
		result.children.add(left);
		result.children.add(right);
		return result;
	}

	// 产生逻辑或语句
	public static Expression Or(Expression left, Expression right,
			String source, int pos) {
		Expression result = new Expression(ExpressionType.Or, source, pos);
		result.children.add(left);
		result.children.add(right);
		return result;
	}

	// 产生>比较运算
	public static Expression GreaterThan(Expression left, Expression right,
			String source, int pos) {
		Expression result = new Expression(ExpressionType.GreaterThan, source,
				pos);
		result.children.add(left);
		result.children.add(right);
		return result;
	}

	// 产生>=比较运算
	public static Expression GreaterThanOrEqual(Expression left,
			Expression right, String source, int pos) {
		Expression result = new Expression(ExpressionType.GreaterThanOrEqual,
				source, pos);
		result.children.add(left);
		result.children.add(right);
		return result;
	}

	// 产生<比较运算
	public static Expression LessThan(Expression left, Expression right,
			String source, int pos) {
		Expression result = new Expression(ExpressionType.LessThan, source, pos);
		result.children.add(left);
		result.children.add(right);
		return result;
	}

	// 产生<=比较运算
	public static Expression LessThanOrEqual(Expression left, Expression right,
			String source, int pos) {
		Expression result = new Expression(ExpressionType.LessThanOrEqual,
				source, pos);
		result.children.add(left);
		result.children.add(right);
		return result;
	}

	// 产生==比较运算
	public static Expression Equal(Expression left, Expression right,
			String source, int pos) {
		Expression result = new Expression(ExpressionType.Equal, source, pos);
		result.children.add(left);
		result.children.add(right);
		return result;
	}

	// 产生!=比较运算
	public static Expression NotEqual(Expression left, Expression right,
			String source, int pos) {
		Expression result = new Expression(ExpressionType.NotEqual, source, pos);
		result.children.add(left);
		result.children.add(right);
		return result;
	}

	// 产生+运算
	public static Expression Add(Expression left, Expression right,
			String source, int pos) {
		Expression result = new Expression(ExpressionType.Add, source, pos);
		result.children.add(left);
		result.children.add(right);
		return result;
	}

	// 产生-运算
	public static Expression Subtract(Expression left, Expression right,
			String source, int pos) {
		Expression result = new Expression(ExpressionType.Subtract, source, pos);
		result.children.add(left);
		result.children.add(right);
		return result;
	}

	// 产生*运算
	public static Expression Multiply(Expression left, Expression right,
			String source, int pos) {
		Expression result = new Expression(ExpressionType.Multiply, source, pos);
		result.children.add(left);
		result.children.add(right);
		return result;
	}

	// 产生除法运算
	public static Expression Divide(Expression left, Expression right,
			String source, int pos) {
		Expression result = new Expression(ExpressionType.Divide, source, pos);
		result.children.add(left);
		result.children.add(right);
		return result;
	}

	// 产生求余运算
	public static Expression Modulo(Expression left, Expression right,
			String source, int pos) {
		Expression result = new Expression(ExpressionType.Modulo, source, pos);
		result.children.add(left);
		result.children.add(right);
		return result;
	}

	// 产生字符串连接运算
	public static Expression Concat(Expression left, Expression right,
			String source, int pos) {
		Expression result = new Expression(ExpressionType.Concat, source, pos);
		result.children.add(left);
		result.children.add(right);
		return result;
	}

	// 产生数组下标
	public static Expression ArrayIndex(Expression objExp, Expression indexExp,
			String source, int pos) {
		Expression result = new Expression(ExpressionType.ArrayIndex, source,
				pos);
		result.children.add(objExp);
		result.children.add(indexExp);
		return result;
	}

	// 产生for循环
	public static Expression For(Expression objExp, Expression forExp,
			String source, int pos) {
		Expression result = new Expression(ExpressionType.For, source, pos);
		result.children.add(objExp);
		result.children.add(forExp);
		return result;
	}

	public void setDelegate(Delegate delegate){
		this.delegate = delegate;
	}
}
