package com.aote.webmeter.tools;

import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;

import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.Query;
import java.lang.management.ManagementFactory;
import java.util.Set;

/**
 * 物联网表基础工具类
 * @author 物联网表小组
 */
public class BasicTools {
    private static final Logger LOGGER = Logger.getLogger(BasicTools.class);
    /**
     * 服务端口
     */
    private static final String PORT;
    /**
     * 本地服务地址
     */
    private static final String SIMPLE_URL;
    private static final String LOCAL_URL;

    static {
        //获取当前服务的端口
        MBeanServer beanServer = ManagementFactory.getPlatformMBeanServer();
        Set<ObjectName> objectNames = null;
        boolean isOK;
        try {
            objectNames = beanServer.queryNames(new ObjectName("*:type=Connector,*"), Query.match(Query.attr("protocol"), Query.value("org.apache.coyote.http11.Http11AprProtocol")));
            isOK = true;
        } catch (Exception e) {
            try {
                objectNames = beanServer.queryNames(new ObjectName("*:type=Connector,*"), Query.match(Query.attr("protocol"), Query.value("HTTP/1.1")));
                isOK = true;
            } catch (MalformedObjectNameException e2) {
                LOGGER.error("获取服务端口时出现异常：");
                isOK = false;
            }
        }
        if (isOK && objectNames.iterator().hasNext()) {
            PORT = objectNames.iterator().next().getKeyProperty("port");
        } else {
            PORT = "8445";
        }
        SIMPLE_URL = "http://localhost:" + PORT + "/";
        LOCAL_URL = SIMPLE_URL + "webmeter/";
        LOGGER.debug("本地Tomcat服务地址：" + LOCAL_URL);
    }

    public static String getPort() {
        return PORT != null ? PORT : "8445";
    }

    public static String getLocalUrl() {
        return LOCAL_URL != null ? LOCAL_URL : "http://localhost:8445/webmeter/";
    }

    public static String getSimpleUrl() {
        return SIMPLE_URL != null ? LOCAL_URL : "http://localhost:8445/";
    }

    /**
     * 业务逻辑线程睡眠
     *
     * @param millis 睡眠时间，单位：毫秒
     */
    public static void sleep(Integer millis) throws InterruptedException {
        Thread.sleep(millis);
    }

    /**
     * 获取银光系统的阶梯集合
     * @param object 阶梯JSON
     * @param gasMaxValue 阶梯气量最大基数值
     * @return 阶梯集合
     */
    public static JSONArray formatStairArray(JSONObject object, Integer gasMaxValue) {
        JSONArray result = new JSONArray();
        for (int i = 1; i <= object.length(); i++) {
            if (object.has("f_stair" + i + "price")) {
                JSONObject stair = new JSONObject();
                stair.put("f_price", Util.convertBigDecimal(object.get("f_stair" + i + "price")));
                int gas = Util.convertBigDecimal(object.get("f_stair" + i + "amount")).intValue();
                if (gas > gasMaxValue || gas < 0) {
                    stair.put("f_gas", gasMaxValue + i);
                } else {
                    stair.put("f_gas", gas);
                }
                result.put(stair);
            }
        }
        return formatStairArray(result);
    }

    public static JSONArray formatStairArray(JSONObject object, Integer gasMaxValue, Boolean isHeat){
        return formatStairArray(object, gasMaxValue);
    }

    public static JSONArray formatStairArray(JSONObject object) {
        return formatStairArray(object, 6550);
    }

    /**
     * 格式化阶梯集合
     *
     * @param array       阶梯集合
     * @param gasMaxValue 阶梯气量最大基数值
     * @param isHeat      是否采暖价格
     * @return 格式化后的阶梯集合
     */
    public static JSONArray formatStairArray(JSONArray array, Integer gasMaxValue, Boolean isHeat){
        JSONArray result = new JSONArray();
        JSONObject item = null;
        int length = array.length();
        if (length < 3) {
            //最小三个阶梯
            length = 3;
        }
        for (int i = 0; i < length; i++) {
            if (array.length() >= i + 1) {
                item = array.getJSONObject(i);
                int gas = item.getInt("f_gas");
                //如果是采暖气价，需要追加采暖增量
                if(isHeat){
                    gas = gas + item.getInt("f_add_gas_heat");
                    item.put("f_gas", gas);
                }
                if (gas > gasMaxValue || gas < 0) {
                    item.put("f_gas", gasMaxValue + i);
                }
                result.put(item);
            } else {
                if (item != null) {
                    item.put("f_gas", gasMaxValue + i);
                    result.put(item);
                }
            }
        }
        return result;
    }
    public static JSONArray formatStairArray(JSONArray array, Integer gasMaxValue) {
        return formatStairArray(array,gasMaxValue,false);
    }

    public static JSONArray formatStairArray(JSONArray array, Integer gasMaxValue, String addGas){
        JSONArray result = new JSONArray();
        JSONObject item = null;
        int length = array.length();
        if (length < 3) {
            //最小三个阶梯
            length = 3;
        }
        for (int i = 0; i < length; i++) {
            if (array.length() >= i + 1) {
                item = array.getJSONObject(i);
                int gas = item.getInt("f_gas") + Integer.parseInt(addGas);
                if (gas > gasMaxValue || gas < 0) {
                    item.put("f_gas", gasMaxValue + i);
                } else {
                    item.put("f_gas", gas);
                }
                result.put(item);
            } else {
                if (item != null) {
                    item.put("f_gas", gasMaxValue + i);
                    result.put(item);
                }
            }
        }
        return result;
    }
    public static JSONArray formatStairArray(JSONArray array) {
        return formatStairArray(array, 6550);
    }
}
