package com.aote.webmeter.timer;

import com.aote.sql.SqlServer;
import com.aote.webmeter.enums.DataSourceTypeEnum;
import com.aote.webmeter.tools.CompatTools;
import com.aote.webmeter.tools.WebMeterInfo;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Profile;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

// 指令同步定时任务
@Component
@Profile("prod")
public class InstructSyncTimer {

    private static final Logger LOG_MANAGER = LogManager.getLogger(InstructSyncTimer.class);
    private static final int PAGE_SIZE = 1000; // 每次处理的记录数

    @Autowired
    private SqlServer sqlServer;

    @Autowired
    private InstructSyncTimerMain timerMain;

    // 定时执行数据同步
    @Scheduled(cron = "0 46 9,12,18,21 * * ?")
    public void runDataSync() throws Exception {
        LOG_MANAGER.info("=====开始迁移历史指令=====");
        String instructTableName = CompatTools.getInstructTableName();
        DataSourceTypeEnum dataSourceTypeEnum = WebMeterInfo.getDataSourceTypeEnum();
        String whereClause = "WHERE f_instruct_state IN ('执行成功', '忽略冗余', '无响应', '已取消') " +
                "OR (f_instruct_state = '响应超时' AND f_inputtor IN ('系统生成','系统档案流程'))";

        // 获取需要处理的总记录数
        String countSql = "SELECT COUNT(*) total FROM " + instructTableName + " " + whereClause;
        int totalCount = sqlServer.querySQL(countSql).
                getJSONObject(0).getInt("total");
        LOG_MANAGER.info("需要迁移的历史指令总数: {}", totalCount);

        if (totalCount == 0) {
            LOG_MANAGER.info("没有需要迁移的历史指令");
            return;
        }

        // 计算总页数
        int totalPages = (totalCount + PAGE_SIZE - 1) / PAGE_SIZE;
        int processedCount = 0;
        int batchCount = 0;

        while (processedCount < totalCount) {
            batchCount++;
            LOG_MANAGER.info("开始处理第 {} 批，共需处理约 {} 批", batchCount, totalPages);

            // 每次查询前1000条符合条件的记录
            String topSQL = generatePageSQL(instructTableName, whereClause, PAGE_SIZE, dataSourceTypeEnum);

            timerMain.run(dataSourceTypeEnum, instructTableName, topSQL);

            // 更新已处理的记录数
            processedCount += PAGE_SIZE;
            LOG_MANAGER.info("已处理 {} 条记录，总计 {} 条", processedCount, totalCount);
        }
        LOG_MANAGER.info("指令迁移完成，共处理 {} 条记录", processedCount);
    }

    private String generatePageSQL(String tableName, String whereClause, int limit, DataSourceTypeEnum dbType) {
        switch (dbType) {
            case ORACLE:
                return String.format("SELECT * FROM (SELECT * FROM " + tableName + " %s ORDER BY ID) WHERE ROWNUM <= %d",
                    whereClause, limit);
            case SQLSERVER:
                return String.format("SELECT TOP %d * FROM " + tableName + " %s ORDER BY ID",
                    limit, whereClause);
            default:
                return String.format("SELECT * FROM " + tableName + " %s ORDER BY ID LIMIT %d",
                            whereClause, limit);
        }
    }
}
