package com.aote.webmeter.tools.iot.onenet;

import com.af.plugins.RestAsyncTools;
import com.af.plugins.RestTools;
import com.aote.webmeter.enums.WebmeterPropertiesIOTEnum;
import com.aote.webmeter.tools.WebMeterInfo;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 移动Onenet-IOT平台 API调用工具
 *
 * @see <a href="https://open.iot.10086.cn/doc/v5/fuse">中国移动物联网开放平台</a>
 * @see <a href="https://open.iot.10086.cn/doc/v5/develop/detail/139">旧版接口</>
 * @author Mr.river
 */
@Component
public class OnenetApiService {
    private static final Logger LOGGER = LoggerFactory.getLogger(OnenetApiService.class);

    @Autowired
    private OnenetApiService2 onenetApiService2;

    @Autowired
    private OnenetTools onenetTools;

    /**
     * 创建设备
     * @param imei imei码
     * @param deviceNo 设备号
     * @return 响应结果
     */
    public JSONObject createDevice(String imei, String deviceNo) throws InterruptedException {
        if (onenetTools.isNewVersion()) {
            return onenetApiService2.createDevice(imei, deviceNo);
        }
        String path = OnenetApi.DEVICE_URL;
        JSONObject headers = new JSONObject();
        String apiKey = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_APIKEY);
        headers.put("api-key", apiKey);
        // 设置请求参数
        JSONObject params = new JSONObject();
        params.put("title", deviceNo);
        params.put("protocol", "LwM2M");
        params.put("auth_info", new JSONObject().put(imei, "000000000000000"));
        // 发送请求
        JSONObject result = new JSONObject(RestTools.post(path, params.toString(), headers.toString()));
        LOGGER.info("api: {}，header: {}，body: {}，response: {}，type: sync", path, headers, params, result);
        int errno = result.getInt("errno");
        String msg;
        if (errno == 0) {
            msg = result.getJSONObject("data").getString("device_id");
        } else {
            msg = "注册IOT设备失败(" + errno + ")："+result.getString("error");
            errno = -1;
        }
        return new JSONObject().put("code", errno).put("msg", msg);
    }

    /**
     * 获取设备信息
     *
     * @param deviceId 设备ID
     * @return 响应结果
     */
    public JSONObject getDevice(String deviceId) {
        String path = OnenetApi.DEVICE_URL + "/" + deviceId;
        //设置请求头
        JSONObject headers = new JSONObject();
        String apiKey = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_APIKEY);
        headers.put("api-key", apiKey);
        //发送请求
        return new JSONObject(RestTools.get(path, headers.toString()));
    }

    /**
     * 删除设备
     *
     * @param deviceId 设备ID
     * @return 响应结果
     */
    public JSONObject deleteDevice(String imei, String deviceId) throws InterruptedException {
        if (onenetTools.isNewVersion()) {
            return onenetApiService2.deleteDevice(imei);
        }
        String path = OnenetApi.DEVICE_URL + "/" + deviceId;
        //设置请求头
        JSONObject headers = new JSONObject();
        String apiKey = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_APIKEY);
        headers.put("api-key", apiKey);
        //发送请求
        JSONObject result = new JSONObject(RestTools.delete(path, null, headers.toString()));
        LOGGER.info("api: {}，header: {}，body: {}，response: {}，type: sync", path, headers, null, result);
        int errno = result.getInt("errno");
        String msg;
        if(errno == 0){
            msg = "成功";
        } else {
            if(errno == 3){
                errno = 0;
                msg = "尝试删除时已不存在";
            } else {
                errno = -1;
                msg = "删除IOT设备失败："+result.getString("error");
            }
        }
        return new JSONObject().put("code",errno).put("msg",msg);
    }

    /**
     * 读取设备资源
     * @param imei 设备imei
     * @param timeout 超时时间，默认25秒
     * @return 响应结果
     */
    public JSONObject readDeviceRes(String imei, String timeout) {
        JSONObject params = new JSONObject();
        params.put("imei", imei);
        params.put("obj_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_ID));
        params.put("obj_inst_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_INST_ID));
        params.put("res_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_RES_ID));
        params.put("timeout", timeout == null ? "25" : timeout);
        //指定请求路径
        String path = RestTools.formatURL(OnenetApi.RESOURCE_URL, params);
        //设置请求头
        JSONObject headers = new JSONObject();
        String apiKey = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_APIKEY);
        headers.put("api-key", apiKey);
        //发送请求
        return new JSONObject(RestTools.get(path, headers.toString()));
    }

    /**
     * 写入设备资源
     * @param imei 设备imei
     * @param mode 写入模式（1或2）
     * @param timeout 超时时间，默认25秒
     * @param content 请求内容(实际请求参数)
     * @return 响应结果
     */
    public JSONObject writeDeviceRes(String imei, String mode, String timeout, String content) throws InterruptedException {
        if (onenetTools.isNewVersion()) {
            return onenetApiService2.writeDeviceRes(imei, mode, timeout, content);
        }
        JSONObject params = new JSONObject();
        params.put("imei", imei);
        params.put("obj_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_ID));
        params.put("obj_inst_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_INST_ID));
        params.put("mode", mode == null ? 1 : mode);
        params.put("timeout", timeout == null ? "25" : timeout);
        //指定请求路径
        String path = RestTools.formatURL(OnenetApi.RESOURCE_URL, params);
        //设置请求头
        JSONObject headers = new JSONObject();
        String apiKey = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_APIKEY);
        headers.put("api-key", apiKey);
        //发送请求
        JSONObject body = new JSONObject();
        JSONArray data = new JSONArray();
        JSONObject dataItem = new JSONObject();
        dataItem.put("res_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_RES_ID));
        dataItem.put("type", 1);
        dataItem.put("val", content);
        data.put(dataItem);
        body.put("data",data);
        RestAsyncTools.post(path, body.toString(), headers.toString());
        LOGGER.info("api: {}，header: {}，body: {}，type: async", path, headers, body);
        return new JSONObject().put("code",0);
    }

    /**
     * 命令下发
     * @param imei 设备imei
     * @param timeout 超时时间，默认25秒
     * @param content 请求内容(实际请求参数)
     * @return 响应结果
     */
    public JSONObject sendCommand(String imei, String timeout, String content) throws InterruptedException {
        if (onenetTools.isNewVersion()) {
            return onenetApiService2.sendCommand(imei, timeout, content);
        }
        JSONObject params = new JSONObject();
        params.put("imei", imei);
        params.put("obj_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_ID));
        params.put("obj_inst_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_INST_ID));
        params.put("res_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_RES_ID));
        params.put("timeout", timeout == null ? "25" : timeout);
        //指定请求路径
        String path = RestTools.formatURL(OnenetApi.COMMAND_URL, params);
        //设置请求头
        JSONObject headers = new JSONObject();
        String apiKey = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_APIKEY);
        headers.put("api-key", apiKey);
        //发送请求
        JSONObject body = new JSONObject();
        body.put("args", content);
        RestAsyncTools.post(path, body.toString(), headers.toString());
        LOGGER.info("api: {}，header: {}，body: {}", path, headers, content);
        return new JSONObject().put("code", 0);
    }

    /**
     * 订阅
     *
     * @param params 请求参数
     * @return 响应结果
     */
    public JSONObject subscribe(JSONObject params) {
        //指定请求路径
        String path = RestTools.formatURL(OnenetApi.SUBSCRIPTIONS_URL, params);
        //设置请求头
        JSONObject headers = new JSONObject();
        String apiKey = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_APIKEY);
        headers.put("api-key", apiKey);
        //发送请求
        String result = RestTools.post(path, params.toString(), headers.toString());
        return new JSONObject(result);
    }
}
