package com.aote.webmeter.tools;

import com.af.plugins.JsonTools;
import lombok.extern.slf4j.Slf4j;
import org.json.JSONArray;
import org.json.JSONObject;

import java.math.BigDecimal;

/**
 * @author 张琪
 * @date 2022/8/24 18:26
 */
@Slf4j
public class ByteDataBuffer {

    /**
     * 输入的缓存大小为 256字节
     */
    private static int INCREASE_DATA_SIZE = 256;

    /**
     * 标注数据的存储方式, 这里的含义为  是否为大端存储(具体含义请百度)
     */
    private boolean inBigEndian;

    /**
     * 用来存储数据的数组
     */
    private byte[] dataBuffer;

    /**
     * 当前位置
     */
    private int pos;

    /**
     * 数据总容量
     */
    private int dataSize;

    /**
     * 编码格式, 一般设置为 UTF-8, 咱不使用
     */
    private String encoding;

    /**
     * 默认填充字符
     */
    private char defaultFillChar;

    public ByteDataBuffer() {
        this(INCREASE_DATA_SIZE);
    }

    public ByteDataBuffer(int size) {
        inBigEndian = false;
        pos = 0;
        dataBuffer = new byte[size];
        dataSize = 0;
    }

    /**
     * 默认小端存储
     * @param data 报文
     */
    public ByteDataBuffer(byte[] data) {
        this(data, false);
    }

    public ByteDataBuffer(byte[] data, boolean inBigEndian) {
        this.inBigEndian = inBigEndian;
        pos = 0;
        if (data != null) {
            dataBuffer = data;
            dataSize = data.length;
        } else {
            dataBuffer = new byte[INCREASE_DATA_SIZE];
            dataSize = 0;
        }
    }

    /**
     * 确保容量, 用于写数据
     * @param minCapacity 当前容量
     */
    public void ensureCapacity(int minCapacity) {
        if (dataBuffer.length < minCapacity) {
            int nextBufSize = INCREASE_DATA_SIZE
                    * (minCapacity / INCREASE_DATA_SIZE + 1);
            byte[] data = new byte[nextBufSize];
            System.arraycopy(dataBuffer, 0, data, 0, dataBuffer.length);
            dataBuffer = data;
        }
    }


    /**
     * 将数据转化为byte数组, 用于写数据
     * @return 待发送数据
     */
    public byte[] getBytes() {
        byte[] data = new byte[getDataSize()];
        System.arraycopy(dataBuffer, 0, data, 0, data.length);
        return data;
    }

    /**
     * 获取当前数据长度, 一般在写数据时调用
     * @return 数据长度
     */
    public int getDataSize() {
        if (pos > dataSize) {
            dataSize = pos;
        }
        return dataSize;
    }

    /**
     * 读取 8bit = 1byte 数据
     * @return 返回读取的值
     */
    public byte readInt8() {
        byte result = dataBuffer[pos];
        pos++;
        return result;
    }

    /**
     读取 16bit = 2byte 数据
     * @return 返回读取的值
     */
    public short readInt16() {
        short word0;
        if (inBigEndian) {
            word0 = (short) (((dataBuffer[pos] & 255) << 8) + (dataBuffer[pos + 1] & 255));
        } else {
            word0 = (short) ((dataBuffer[pos] & 255) + ((dataBuffer[pos + 1] & 255) << 8));
        }
        pos += 2;
        return word0;
    }


    /**
     读取 16bit = 2byte 数据
     * @return 返回读取的值
     */
    public int readInt24() {
        int i;
        if (inBigEndian) {

            i = ((dataBuffer[pos] & 255) << 24)
                    + ((dataBuffer[pos + 1] & 255) << 16)
                    + ((dataBuffer[pos + 2] & 255) << 8);
        } else {
            i = (dataBuffer[pos] & 255) + ((dataBuffer[pos + 1] & 255) << 8)
                    + ((dataBuffer[pos + 2] & 255) << 16);
        }
        pos += 3;
        return i;
    }

    /**
     * 读取32bit = 4byte 数据
     * @return 返回读取到的数据
     */
    public int readInt32() {
        int i;
        if (inBigEndian) {

            i = ((dataBuffer[pos] & 255) << 24)
                    + ((dataBuffer[pos + 1] & 255) << 16)
                    + ((dataBuffer[pos + 2] & 255) << 8)
                    + (dataBuffer[pos + 3] & 255);
        } else {
            i = (dataBuffer[pos] & 255) + ((dataBuffer[pos + 1] & 255) << 8)
                    + ((dataBuffer[pos + 2] & 255) << 16)
                    + ((dataBuffer[pos + 3] & 255) << 24);
        }
        pos += 4;
        return i;
    }

    /**
     * 将数据读取到传入的数组中
     * @param data 待写入数组
     * @return 读取的数据总长度
     */
    public int readBytes(byte[] data) {
        return readBytes(data, 0, data.length);
    }


    /**
     * 将指定起始位和长度的数据读取到传入的数组中
     * @param data 待写入数组
     * @param destPos 其实位置
     * @param length 读取长度
     * @return 返回读取长度
     */
    public int readBytes(byte[] data, int destPos, int length) {
        if (length > dataBuffer.length - pos) {
            length = dataBuffer.length - pos;
        }
        if (length > 0) {
            System.arraycopy(dataBuffer, pos, data, destPos, length);
        }
        pos += length;
        return length;
    }

    /**
     * 通用解析报文
     * @param config 配置文件
     * @return 解析后报文
     * @throws Exception 解析异常报错信息
     */
    public JSONObject commonParse(JSONObject config) throws Exception {
        if (!config.has("name")) {
            throw new Exception("缺少字段名, 请检查配置");
        }
        if (!config.has("length")) {
            throw new Exception("缺少字段长度, 请检查配置");
        }
        if (!config.has("type")) {
            throw new Exception("缺少字段类型, 请检查配置");
        }

        JSONObject jsonObject = new JSONObject();

        String type = config.getString("type");
        int length = config.getInt("length");
        Object value= null;
        if ("Number".equals(type)) {
            value = commonReadNumber(length);
        } else if ("String".equals(type)) {
            value = commonReadStr(length);
            // 解析报文出现嵌套翻译
            if (config.has("transConfigPath")) {
                ByteDataBuffer str = new ByteDataBuffer(ByteTool.hexString2Bytes((String) value));
                value = str.parse(JsonTools.readJsonArrayFile(config.getString("transConfigPath")));
            }
        } else if ("Array".equals(type)) {
            value = commonReadArray(config);
        }
        if (config.has("children")) {
            value = commonReadBit((int)value, config.getJSONArray("children"));
        }
        if (config.has("magnification")) {
            BigDecimal bigDecimal = new BigDecimal((int) value);
            value = bigDecimal.divide(new BigDecimal(config.getInt("magnification")));
        }
        jsonObject.put("key", config.getString("name"));
        jsonObject.put("value", value);

        return jsonObject;
    }

    /**
     * 通用解析数组
     * @param config 报文格式
     * @return 解析后数组信息
     */
    public JSONArray commonReadArray(JSONObject config) {
        int length = config.getInt("length");
        int nodeLength = config.getInt("nodeLength");
        String nodeType = config.getString("nodeType");
        String nodeTransParsePath = config.getString("nodeTransConfigPath");
        int nodeMagnification = config.optInt("nodeMagnification");
        JSONArray array = new JSONArray();
        // 子节点类型为 数字, 直接解析
        if ("Number".equals(nodeType)) {
            for (int i = 0; i < length / nodeLength; i++) {
                int i1 = commonReadNumber(nodeLength);
                if (nodeMagnification != 0) {
                    array.put(new BigDecimal(i1).divide(new BigDecimal(nodeMagnification)));
                } else {
                    array.put(i1);
                }
            }
        } else {
            // 子节点为其他类型, 按照通用格式进行解析
            for (int i = 0; i < length / nodeLength; i++) {
                JSONObject jsonObject = parse(JsonTools.readJsonArrayFile(nodeTransParsePath));
                array.put(jsonObject);
            }
        }
        return array;
    }

    /**
     * 默认读字符串方法
     * @param length 长度
     * @return 返回16进制字符串
     */
    public String commonReadStr(int length) {
        byte[] str = new byte[length];
        int i = readBytes(str);
        log.debug("读取{}字节数据", i);
        return ByteTool.bytesToHexString(str);
    }

    /**
     * 通用读数字
     * @param length 需要读取的位数
     * @return 读取的数据
     */
    public int commonReadNumber(int length) {
        if (length == 1) {
            return readInt8();
        } else if (length == 2) {
            return readInt16();
        } else if (length == 3) {
            return readInt24();
        } else if (length == 4) {
            return readInt32();
        } else {
            return Integer.MAX_VALUE;
        }
    }

    /**
     * 默认读取 bit
     * @param data 数据
     * @param children bit信息数组
     * @return JSONObject对象
     */
    public Object commonReadBit(int data, JSONArray children) {
        JSONObject childResult = new JSONObject();
        for (int i = 0; i < children.length(); i++) {
            JSONObject childConfig = (JSONObject) children.get(i);
            int bitData = data & Integer.parseInt(childConfig.getString("value"), 2);
            if (!childConfig.optBoolean("shift")) {
                bitData = bitData >>> (Integer.toBinaryString(Integer.parseInt(childConfig.getString("value"), 2)).length() - 1);
            }
            childResult.put(childConfig.getString("name"), bitData);
        }
        return childResult;
    }


    /**
     * 对外转化方法
     * @param configs 报文配置
     * @return 解析后内容
     */
    public JSONObject parse(JSONArray configs) {
        JSONObject result = new JSONObject();
        for (int i = 0; i < configs.length(); i++) {
            JSONObject config = configs.getJSONObject(i);
            try {
                JSONObject jsonObject = commonParse(config);
                result.put(jsonObject.getString("key"), jsonObject.get("value"));
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return result;
    }

    /**
     * 报文解析
     * @param filePath 报文格式配置文件相对路径
     * @return 解析后报文
     */
    public JSONObject dataTrans(String filePath) {
        return parse(JsonTools.readJsonArrayFile(filePath));
    }

    /**
     * 写 1byte = 8bit 数据
     * @param data 待写入数据
     */
    public void writeInt8(byte data) {
        ensureCapacity(pos + 1);
        dataBuffer[pos] = data;
        pos++;
    }

    /**
     * 写 2byte = 16bit 数据
     * @param i 待写入数据
     */
    public void writeInt16(int i) {
        ensureCapacity(pos + 2);
        if (inBigEndian) {
            dataBuffer[pos] = (byte) (i >>> 8 & 255);
            dataBuffer[pos + 1] = (byte) (i & 255);
        } else {
            dataBuffer[pos] = (byte) (i & 255);
            dataBuffer[pos + 1] = (byte) (i >>> 8 & 255);
        }
        pos += 2;
    }

    /**
     * 写 3byte = 24bit 数据
     * @param i 待写入数据
     */
    public void writeInt24(int i) {
        ensureCapacity(pos + 2);
        if (inBigEndian) {
            dataBuffer[pos] = (byte) (i >>> 16 & 255);
            dataBuffer[pos + 1] = (byte) (i >>> 8& 255);
            dataBuffer[pos + 2] = (byte) (i & 255);
        } else {
            dataBuffer[pos] = (byte) (i & 255);
            dataBuffer[pos + 1] = (byte) (i >>> 8 & 255);
            dataBuffer[pos + 2] = (byte) (i >>> 16 & 255);
        }
        pos += 3;
    }

    /**
     * 写 4byte = 32bit 数据
     * @param i 待写入数据
     */
    public void writeInt32(int i) {
        ensureCapacity(pos + 4);
        if (inBigEndian) {
            dataBuffer[pos] = (byte) (i >>> 24 & 255);
            dataBuffer[pos + 1] = (byte) (i >>> 16 & 255);
            dataBuffer[pos + 2] = (byte) (i >>> 8 & 255);
            dataBuffer[pos + 3] = (byte) (i & 255);
        } else {
            dataBuffer[pos] = (byte) (i & 255);
            dataBuffer[pos + 1] = (byte) (i >>> 8 & 255);
            dataBuffer[pos + 2] = (byte) (i >>> 16 & 255);
            dataBuffer[pos + 3] = (byte) (i >>> 24 & 255);
        }
        pos += 4;
    }

    /**
     * 将数组中数据写入报文
     * @param data 待写入数据
     */
    public void writeBytes(byte[] data) {
        writeBytes(data, 0, data.length);
    }

    /**
     * 将数组中数据写入报文
     * @param data 待写入数据
     * @param srcPos 写入起点下标
     * @param length 写入总长度
     */
    public void writeBytes(byte[] data, int srcPos, int length) {
        if (data == null || length <= 0){
            return;
        }
        if (srcPos + length > data.length)
            length = data.length - srcPos;
        ensureCapacity(pos + length);
        System.arraycopy(data, srcPos, dataBuffer, pos, length);
        pos += length;
    }

    /**
     * 填充字段方法
     * @param data 待填充字段
     * @param length 填充长度
     */
    public void writeFillerBytes(byte data, int length) {
        ensureCapacity(pos + length);
        for (int j = 0; j < length; j++) {
            dataBuffer[pos] = data;
            pos++;
        }
    }

    public static void main(String[] args) {
        JSONArray array = JsonTools.readJsonArrayFile("dataBuffer/QinChuanNB21/1003.json");
        System.out.println(array.toString());
    }
}
